% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/HyperSelect.R
\name{HyperSelect}
\alias{HyperSelect}
\title{Hyperparameter selection for iMOM prior density}
\usage{
HyperSelect(
  X,
  resp,
  eff_size = 0.7,
  nlptype = "piMOM",
  iter = 10000,
  mod_prior = c("unif", "beta"),
  family = c("logistic", "survival")
)
}
\arguments{
\item{X}{The design matrix. \code{NA}'s should be removed and columns be
scaled. It is recommended that the \code{PreProcess} function is run first
and its output used for this argument. The columns are genes and rows
represent the observations. The column names are used as gene names.}

\item{resp}{For logistic regression models, it is the binary response
vector. For Cox proportional hazard model, this is a two column matrix
where the first column contains survival time vector and the second column
is the censoring status for each observation.}

\item{eff_size}{This is the expected effect size in the model for a
standardized design matrix, which is basically the coefficient value that is
expected to occur the most based on some prior knowledge.}

\item{nlptype}{Determines the type of nonlocal prior that is used in the
analyses. It can be "piMOM" for product inverse moment prior, or "pMOM" for
product moment prior. The default is set to piMOM prior.}

\item{iter}{The number of iteration needed to simulate from null model in
order to approximate the null MLE density.}

\item{mod_prior}{Type of prior used for model space. \code{uniform} is for
uniform binomial and \code{beta} is for beta binomial prior. In the former
case, both hyper parameters in the beta prior are equal to \code{1} but in
the latter case those two hyper parameters are chosen as explained in the
reference papers.}

\item{family}{Determines the type of data analysis. \code{logistic} is for
binary outcome data where logistic regression modeling is used whereas
\code{survival} is for survival outcome data using Cox proportional
hazard model.}
}
\value{
It returns a list having following object:
\item{tau}{The hyperparameter for iMOM prior density function, calculated
using the proposed algorithm for the given dataset.}
}
\description{
This function finds data specific hyperparameters for inverse
moment prior density so that the overlap between the iMOM prior and null
MLE density is \eqn{1/\sqrt p}. In this algorithm \code{r} is always chosen
to be equal to 1 and \code{tau} is found based on the mentioned overlap.
}
\examples{
### Simulating Logistic Regression Data
n <- 20
p <- 100
set.seed(321)
Sigma <- diag(p)
full <- matrix(c(rep(0.5, p * p)), ncol = p)
Sigma <- full + 0.5 * Sigma
cholS <- chol(Sigma)
Beta <- c(1, 1.8, 2.5)
X = matrix(rnorm(n*p, 1, 2), ncol = p)
X = X\%*\%cholS
beta <- numeric(p)
beta[c(1:length(Beta))] <- Beta
XB <- X\%*\%beta
probs <- as.vector(exp(XB)/(1+exp(XB)))
y <- rbinom(n,1,probs)
colnames(X) <- paste("gene_",c(1:p),sep="")
Xout <- PreProcess(X)
XX <- Xout$X
hparam <- HyperSelect(XX, y, iter = 1000, mod_prior = "beta",
                      family = "logistic")

hparam$tau
}
\references{
Nikooienejad, A., Wang, W., and Johnson, V. E. (2016). Bayesian
variable selection for binary outcomes in high dimensional genomic studies
using nonlocal priors. Bioinformatics, 32(9), 1338-1345.\cr\cr
Johnson, V. E., and Rossell, D. (2010). On the use of nonlocal prior
densities in Bayesian hypothesis tests. Journal of the Royal Statistical
Society: Series B (Statistical Methodology), 72(2), 143-170.
}
\author{
Amir Nikooienejad
}
