% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/bktr.R
\name{BKTRRegressor}
\alias{BKTRRegressor}
\title{R6 class encapsulating the BKTR regression elements}
\description{
A BKTRRegressor holds all the key elements to accomplish the MCMC sampling
algorithm (\strong{Algorithm 1} of the paper).
}
\examples{
\dontshow{if (torch::torch_is_installed()) (if (getRversion() >= "3.4") withAutoprint else force)(\{ # examplesIf}
# Create a BIXI data collection instance containing multiple dataframes
bixi_data <- BixiData$new(is_light = TRUE) # Use light version for example

# Create a BKTRRegressor instance
bktr_regressor <- BKTRRegressor$new(
  formula = nb_departure ~ 1 + mean_temp_c + area_park,
  data_df <- bixi_data$data_df,
  spatial_positions_df = bixi_data$spatial_positions_df,
  temporal_positions_df = bixi_data$temporal_positions_df,
  burn_in_iter = 5, sampling_iter = 10) # For example only (too few iterations)

# Launch the MCMC sampling
bktr_regressor$mcmc_sampling()

# Get the summary of the bktr regressor
summary(bktr_regressor)

# Get estimated response variables for missing values
bktr_regressor$imputed_y_estimates

# Get the list of sampled betas for given spatial, temporal and feature labels
bktr_regressor$get_iterations_betas(
  spatial_label = bixi_data$spatial_positions_df$location[1],
  temporal_label = bixi_data$temporal_positions_df$time[1],
  feature_label = 'mean_temp_c')

# Get the summary of all betas for the 'mean_temp_c' feature
bktr_regressor$get_beta_summary_df(feature_labels = 'mean_temp_c')
\dontshow{\}) # examplesIf}
\dontshow{if (torch::torch_is_installed()) (if (getRversion() >= "3.4") withAutoprint else force)(\{ # examplesIf}
## PREDICTION EXAMPLE ##
# Create a light version of the BIXI data collection instance
bixi_data <- BixiData$new(is_light = TRUE)
# Simplify variable names
data_df <- bixi_data$data_df
spa_pos_df <- bixi_data$spatial_positions_df
temp_pos_df <- bixi_data$temporal_positions_df

# Keep some data aside for prediction
new_spa_pos_df <- spa_pos_df[1:2, ]
new_temp_pos_df <- temp_pos_df[1:5, ]
reg_spa_pos_df <- spa_pos_df[-(1:2), ]
reg_temp_pos_df <- temp_pos_df[-(1:5), ]
reg_data_df_mask <- data_df$location \%in\% reg_spa_pos_df$location &
  data_df$time \%in\% reg_temp_pos_df$time
reg_data_df <- data_df[reg_data_df_mask, ]
new_data_df <- data_df[!reg_data_df_mask, ]

# Launch mcmc sampling on regression data
bktr_regressor <- BKTRRegressor$new(
  formula = nb_departure ~ 1 + mean_temp_c + area_park,
  data_df = reg_data_df,
  spatial_positions_df = reg_spa_pos_df,
  temporal_positions_df = reg_temp_pos_df,
  burn_in_iter = 5, sampling_iter = 10) # For example only (too few iterations)
bktr_regressor$mcmc_sampling()

# Predict response values for new data
bktr_regressor$predict(
  new_data_df = new_data_df,
  new_spatial_positions_df = new_spa_pos_df,
  new_temporal_positions_df = new_temp_pos_df)
\dontshow{\}) # examplesIf}
}
\section{Public fields}{
\if{html}{\out{<div class="r6-fields">}}
\describe{
\item{\code{data_df}}{The dataframe containing all the covariates through time and space (including
the response variable)}

\item{\code{y}}{The response variable tensor}

\item{\code{omega}}{The tensor indicating which response values are not missing}

\item{\code{covariates}}{The tensor containing all the covariates}

\item{\code{covariates_dim}}{The dimensions of the covariates tensor}

\item{\code{logged_params_tensor}}{The tensor containing all the sampled hyperparameters}

\item{\code{tau}}{The precision hyperparameter}

\item{\code{spatial_decomp}}{The spatial covariate decomposition}

\item{\code{temporal_decomp}}{The temporal covariate decomposition}

\item{\code{covs_decomp}}{The feature covariate decomposition}

\item{\code{result_logger}}{The result logger instance used to store the results of the MCMC sampling}

\item{\code{has_completed_sampling}}{Boolean showing wheter the MCMC sampling has been completed}

\item{\code{spatial_kernel}}{The spatial kernel used}

\item{\code{temporal_kernel}}{The temporal kernel used}

\item{\code{spatial_positions_df}}{The dataframe containing the spatial positions}

\item{\code{temporal_positions_df}}{The dataframe containing the temporal positions}

\item{\code{spatial_params_sampler}}{The spatial kernel hyperparameter sampler}

\item{\code{temporal_params_sampler}}{The temporal kernel hyperparameter sampler}

\item{\code{tau_sampler}}{The tau hyperparameter sampler}

\item{\code{precision_matrix_sampler}}{The precision matrix sampler}

\item{\code{spatial_ll_evaluator}}{The spatial likelihood evaluator}

\item{\code{temporal_ll_evaluator}}{The temporal likelihood evaluator}

\item{\code{rank_decomp}}{The rank of the CP decomposition}

\item{\code{burn_in_iter}}{The number of burn in iterations}

\item{\code{sampling_iter}}{The number of sampling iterations}

\item{\code{max_iter}}{The total number of iterations}

\item{\code{a_0}}{The initial value for the shape in the gamma function generating tau}

\item{\code{b_0}}{The initial value for the rate in the gamma function generating tau}

\item{\code{formula}}{The formula used to specify the relation between the response variable and the covariates}

\item{\code{spatial_labels}}{The spatial labels}

\item{\code{temporal_labels}}{The temporal labels}

\item{\code{feature_labels}}{The feature labels}

\item{\code{geo_coords_projector}}{The geographic coordinates projector}
}
\if{html}{\out{</div>}}
}
\section{Active bindings}{
\if{html}{\out{<div class="r6-active-bindings">}}
\describe{
\item{\code{summary}}{A summary of the BKTRRegressor instance}

\item{\code{beta_covariates_summary}}{A dataframe containing the summary of the beta covariates}

\item{\code{y_estimates}}{A dataframe containing the y estimates}

\item{\code{imputed_y_estimates}}{A dataframe containing the imputed y estimates}

\item{\code{beta_estimates}}{A dataframe containing the beta estimates}

\item{\code{hyperparameters_per_iter_df}}{A dataframe containing the beta estimates per iteration}

\item{\code{decomposition_tensors}}{List of all used decomposition tensors}
}
\if{html}{\out{</div>}}
}
\section{Methods}{
\subsection{Public methods}{
\itemize{
\item \href{#method-BKTRRegressor-new}{\code{BKTRRegressor$new()}}
\item \href{#method-BKTRRegressor-mcmc_sampling}{\code{BKTRRegressor$mcmc_sampling()}}
\item \href{#method-BKTRRegressor-predict}{\code{BKTRRegressor$predict()}}
\item \href{#method-BKTRRegressor-get_iterations_betas}{\code{BKTRRegressor$get_iterations_betas()}}
\item \href{#method-BKTRRegressor-get_beta_summary_df}{\code{BKTRRegressor$get_beta_summary_df()}}
\item \href{#method-BKTRRegressor-clone}{\code{BKTRRegressor$clone()}}
}
}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-BKTRRegressor-new"></a>}}
\if{latex}{\out{\hypertarget{method-BKTRRegressor-new}{}}}
\subsection{Method \code{new()}}{
Create a new \code{BKTRRegressor} object.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{BKTRRegressor$new(
  data_df,
  spatial_positions_df,
  temporal_positions_df,
  rank_decomp = 10,
  burn_in_iter = 500,
  sampling_iter = 500,
  formula = NULL,
  spatial_kernel = KernelMatern$new(smoothness_factor = 3),
  temporal_kernel = KernelSE$new(),
  sigma_r = 0.01,
  a_0 = 1e-06,
  b_0 = 1e-06,
  has_geo_coords = TRUE,
  geo_coords_scale = 10
)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{data_df}}{data.table: A dataframe containing all the covariates
through time and space. It is important that the dataframe has a two
indexes named `location` and `time` respectively. The dataframe should
also contain every possible combinations of `location` and `time`
(i.e. even missing rows should be filled present but filled with NaN).
So if the dataframe has 10 locations and 5 time points, it should have
50 rows (10 x 5). If formula is None, the dataframe should contain
the response variable `Y` as the first column. Note that the covariate
columns cannot contain NaN values, but the response variable can.}

\item{\code{spatial_positions_df}}{data.table: Spatial kernel input tensor used
to calculate covariates' distance. Vector of length equal to the number of location points.}

\item{\code{temporal_positions_df}}{data.table: Temporal kernel input tensor used to
calculate covariate distance. Vector of length equal to the number of time points.}

\item{\code{rank_decomp}}{Integer: Rank of the CP decomposition (Paper -- \eqn{R}). Defaults to 10.}

\item{\code{burn_in_iter}}{Integer: Number of iteration before sampling (Paper -- \eqn{K_1}). Defaults to 500.}

\item{\code{sampling_iter}}{Integer: Number of sampling iterations (Paper -- \eqn{K_2}). Defaults to 500.}

\item{\code{formula}}{A Wilkinson R formula to specify the relation
between the response variable `Y` and the covariates. If Null, the first
column of the data frame will be used as the response variable and all the
other columns will be used as the covariates.  Defaults to Null.}

\item{\code{spatial_kernel}}{Kernel: Spatial kernel Used. Defaults to
a KernelMatern(smoothness_factor=3).}

\item{\code{temporal_kernel}}{Kernel: Temporal kernel used. Defaults to KernelSE().}

\item{\code{sigma_r}}{Numeric:  Variance of the white noise process (\eqn{\tau^{-1}})
defaults to 1E-2.}

\item{\code{a_0}}{Numeric: Initial value for the shape (\eqn{\alpha}) in the gamma function
generating tau defaults to 1E-6.}

\item{\code{b_0}}{Numeric: Initial value for the rate (\eqn{\beta}) in the gamma function
generating tau defaults to 1E-6.}

\item{\code{has_geo_coords}}{Boolean: Whether the spatial positions df use geographic coordinates
(latitude, longitude). Defaults to TRUE.}

\item{\code{geo_coords_scale}}{Numeric: Scale factor to convert geographic coordinates to euclidean
2D space via Mercator projection using x & y domains of [-scale/2, +scale/2]. Only used if
has_geo_coords is TRUE. Defaults to 10.}
}
\if{html}{\out{</div>}}
}
\subsection{Returns}{
A new \code{BKTRRegressor} object.
}
}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-BKTRRegressor-mcmc_sampling"></a>}}
\if{latex}{\out{\hypertarget{method-BKTRRegressor-mcmc_sampling}{}}}
\subsection{Method \code{mcmc_sampling()}}{
Launch the MCMC sampling process. \cr
For a predefined number of iterations:
\enumerate{
\item{Sample spatial kernel hyperparameters}
\item{Sample temporal kernel hyperparameters}
\item{Sample the precision matrix from a wishart distribution}
\item{Sample a new spatial covariate decomposition}
\item{Sample a new feature covariate decomposition}
\item{Sample a new temporal covariate decomposition}
\item{Calculate respective errors for the iterations}
\item{Sample a new tau value}
\item{Collect all the important data for the iteration}
}
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{BKTRRegressor$mcmc_sampling()}\if{html}{\out{</div>}}
}

\subsection{Returns}{
NULL Results are stored and can be accessed via summary()
}
}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-BKTRRegressor-predict"></a>}}
\if{latex}{\out{\hypertarget{method-BKTRRegressor-predict}{}}}
\subsection{Method \code{predict()}}{
Use interpolation to predict betas and response values for new data.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{BKTRRegressor$predict(
  new_data_df,
  new_spatial_positions_df = NULL,
  new_temporal_positions_df = NULL,
  jitter = 1e-05
)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{new_data_df}}{data.table: New covariates. Must have the same columns as
the covariates used to fit the model. The index should contain the combination
of all old spatial coordinates with all new temporal coordinates, the combination
of all new spatial coordinates with all old temporal coordinates, and the
combination of all new spatial coordinates with all new temporal coordinates.}

\item{\code{new_spatial_positions_df}}{data.table or NULL: A data frame containing the new
spatial positions. Defaults to NULL.}

\item{\code{new_temporal_positions_df}}{data.table or NULL: A data frame containing the new
temporal positions. Defaults to NULL.}

\item{\code{jitter}}{Numeric or NULL: A small value to add to the diagonal of the precision matrix.
Defaults to NULL.}
}
\if{html}{\out{</div>}}
}
\subsection{Returns}{
List: A list of two dataframes. The first represents the beta
  forecasted for all new spatial locations or temporal points.
  The second represents the forecasted response for all new spatial
  locations or temporal points.
}
}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-BKTRRegressor-get_iterations_betas"></a>}}
\if{latex}{\out{\hypertarget{method-BKTRRegressor-get_iterations_betas}{}}}
\subsection{Method \code{get_iterations_betas()}}{
Return all sampled betas through sampling iterations for a given
set of spatial, temporal and feature labels. Useful for plotting the
distribution of sampled beta values.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{BKTRRegressor$get_iterations_betas(
  spatial_label,
  temporal_label,
  feature_label
)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{spatial_label}}{String: The spatial label for which we want to get the betas}

\item{\code{temporal_label}}{String: The temporal label for which we want to get the betas}

\item{\code{feature_label}}{String: The feature label for which we want to get the betas}
}
\if{html}{\out{</div>}}
}
\subsection{Returns}{
A list containing the sampled betas through iteration for the given labels
}
}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-BKTRRegressor-get_beta_summary_df"></a>}}
\if{latex}{\out{\hypertarget{method-BKTRRegressor-get_beta_summary_df}{}}}
\subsection{Method \code{get_beta_summary_df()}}{
Get a summary of estimated beta values. If no labels are given,
then the summary is for all the betas. If labels are given, then the summary
is for the given labels.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{BKTRRegressor$get_beta_summary_df(
  spatial_labels = NULL,
  temporal_labels = NULL,
  feature_labels = NULL
)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{spatial_labels}}{vector: The spatial labels used in summary. If NULL,
then all spatial labels are used. Defaults to NULL.}

\item{\code{temporal_labels}}{vector: The temporal labels used in summary. If NULL,
then all temporal labels are used. Defaults to NULL.}

\item{\code{feature_labels}}{vector: The feature labels used in summary. If NULL,
then all feature labels are used. Defaults to NULL.}
}
\if{html}{\out{</div>}}
}
\subsection{Returns}{
A new data.table with the beta summary for the given labels.
}
}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-BKTRRegressor-clone"></a>}}
\if{latex}{\out{\hypertarget{method-BKTRRegressor-clone}{}}}
\subsection{Method \code{clone()}}{
The objects of this class are cloneable with this method.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{BKTRRegressor$clone(deep = FALSE)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{deep}}{Whether to make a deep clone.}
}
\if{html}{\out{</div>}}
}
}
}
