#' Plots for the Hazard and Survival Funcion Estimates
#' 
#' Plots the hazard function and the survival function
#' estimates defined by the bayesian semiparametric cure 
#' rate model with an unknown threshold
#' (Nieto-Barajas & Yin, 2008).
#' 
#' This function return estimators plots for the resulting hazard rate as it is computed
#' by \link{CuMRes} and the cure time (quantile of Tao specified by the user), 
#' together with credible intervals. Additionally, it plots the survival function and the cure proportion estimates
#' with their corresponding credible intervals.
#' 
#' @param M tibble. Contains the output generated by \code{CuMRres}.
#' @param type.h character. "segment"= use segments to plot hazard rates,
#' "line" = link hazard rates by a line
#' @param intervals logical. If TRUE, plots credible intervals.
#' @param confidence Numeric. Confidence level.
#' @param qn Numeric. Quantile for Tao that should be visualized on the plot.
#' @param summary Logical. If \code{TRUE}, a summary for hazard and survival
#' functions is returned as a tibble.
#' @param position_label character. Labels on the right or left side of the
#' plot.
#' @return \item{SUM.h}{Numeric tibble. Summary for the mean, median, and a
#' \code{confint / 100} confidence interval for each segment of the hazard
#' function. If \code{summary = TRUE}} \item{SUM.S}{Numeric tibble. Summary for
#' the mean, median, and a \code{confint / 100} confidence interval for a grid of 
#' the survival function. If \code{summary = TRUE}}
#' @seealso \link{CuMRes},
#' @references - Nieto-Barajas, L. E. (2003). Discrete time Markov gamma
#' processes and time dependent covariates in survival analysis. \emph{Bulletin
#' of the International Statistical Institute 54th Session}. Berlin. (CD-ROM).
#' 
#' -Nieto-Barajas, L. E., & Yin, G. (2008). Bayesian semiparametric cure rate
#' model with an unknown threshold. \emph{Scandinavian Journal of Statistics},
#' \strong{35(3)}, 540-556. https://doi.org/10.1111/j.1467-9469.2007.00589.x
#' @examples
#' 
#' 
#' 
#' ## Simulations may be time intensive. Be patient.
#' 
#' ## Example 1
#' # data(crm3)
#'     # times<-crm3$times
#'     # delta<-crm3$delta
#'     # res <- CuMRes(times, delta, type.t = 2, length = .1,
#'     #                   K = 100, alpha = rep(1, 100  ),
#'     #                   beta = rep(1, 100),c.r = rep(50, 99),
#'     #                   iterations = 100, burn.in = 10, thinning = 1, type.c = 2)
#'     # CuPloth(res, type.h = "segment",qn=.5, summary = T)
#'     # CuPloth(res, type.h = "line",qn=.5)
#' 
#' 
#' 
#' 
#' @export CuPloth
CuPloth <-
  function(M, type.h = "segment", intervals = T,
           confidence = 0.95, qn = 0.5, summary = FALSE, position_label = "right") {
    SUM <- CuLambdaSumm(M, confidence)
    v <- tibble::deframe(SUM)
    tao <- extract(M, "tao")
    K <- extract(M, "K")
    z <- extract(M, c("simulations","Z"))
    ribbon <- tibble::tibble(x = seq(to = tao[dplyr::pull(v$SUM.z, 4) + 1],
                             from = tao[dplyr::pull(v$SUM.z, 2) + 1], by = 0.1),
                     y = max(v$SUM.h$upper))
    if(position_label == "left") {
      position <- round(tao[dplyr::pull(v$SUM.z, 2) + 1],2)
      pos_just <- 1.1
    }
    if(position_label == "right") {
      position <- round(tao[dplyr::pull(v$SUM.z, 4) + 1],2)
      pos_just <- -.1
    }
    if(type.h == "segment") {
      h <- ggplot2::ggplot(v$SUM.h) + 
      ggplot2::geom_segment(ggplot2::aes(x = tao[-(K+1)], xend = tao[-1], 
                       y = mean, yend = mean)) + 
      ggplot2::xlab("Time") + ggplot2::ylab("Hazard rate") + ggplot2::scale_alpha_continuous(guide = F) + 
      ggplot2::ggtitle(paste0("Estimate of hazard rates with intervals at ",confidence * 100,"% of credibility")) +
      ggplot2::geom_vline(xintercept = round(tao[quantile(z,qn)+1],2), linetype = "dotted") + 
      ggplot2::annotate("text",x = position, y = max(v$SUM.h$upper),
               label = paste0(expression(tau[z])," ==", round(tao[quantile(z,qn)+1],2)),
               hjust = pos_just, vjust = 1, parse = T) +
      ggplot2::annotate("text",x = position, y = max(v$SUM.h$upper),
               label = paste0(expression(pi)," == ", round(dplyr::pull(v$SUM.pi, mean),2)),
               hjust = pos_just, vjust = 2.5,parse = T) +
      ggthemes::theme_tufte() +
      ggplot2::theme(axis.line = ggplot2::element_line(colour = "black"))
      if(intervals){
        h <- h + ggplot2::geom_errorbar(ggplot2::aes(ymin = lower, ymax = upper, x = (tao[-(K+1)] + tao[-1])/2, width = tao[-1]-tao[-(K+1)]), 
                                        alpha = 0.5, color = "gray50") +
          ggplot2::geom_ribbon(data = ribbon, ggplot2::aes(x= x, ymin = 0, ymax = y), 
                               alpha = .1, fill = "red")
      }
    }
    
    if(type.h == "line") {
      h <- ggplot2::ggplot(v$SUM.h) + 
      ggplot2::geom_line(ggplot2::aes(x = (tao[-(K+1)] + tao[-1])/2, y = mean)) +
      ggplot2::xlab("Time") + ggplot2::ylab("Hazard rate") + ggplot2::scale_alpha_continuous(guide = F) + 
      ggplot2::ggtitle(paste0("Estimate of hazard rates with intervals at ",confidence * 100,"%  of credibility")) +
      ggplot2::geom_vline(xintercept = round(tao[quantile(z,qn)+1], 2), linetype = "dotted") + 
      ggplot2::annotate("text",x = position, y = max(v$SUM.h$upper),
               label = paste0(expression(tau[z])," ==", round(tao[quantile(z,qn)+1],2)),
               hjust = pos_just, vjust = 1,parse = T) +
      ggplot2::annotate("text",x = position, y = max(v$SUM.h$upper),
               label = paste0(expression(pi)," == ", round(dplyr::pull(v$SUM.pi, mean),2)),
               hjust = pos_just, vjust = 2.5,parse = T) +
      ggthemes::theme_tufte() +
      ggplot2::theme(axis.line = ggplot2::element_line(colour = "black"))
      if(intervals){
        h <- h + ggplot2::geom_ribbon(data = ribbon, ggplot2::aes(x= x, ymin = 0, ymax = y), 
                                      alpha = .1, fill = "red") + 
          ggplot2::geom_ribbon(ggplot2::aes(x = (tao[-(K+1)] + tao[-1])/2, ymin = lower, ymax = upper), alpha = .5, fill = "gray70")
      }
    }
    S <- ggplot2::ggplot(v$SUM.S) + ggplot2::geom_line(ggplot2::aes(x = t, y = `S^(t)`)) + 
      
      ggplot2::scale_y_continuous(limits = c(0,1)) + 
      ggplot2::ggtitle(paste0("Estimate of Survival Function with intervals at ", confidence * 100,"%  of credibility")) +
      ggplot2::labs(x = "t",
           y = expression(S^{(t)})) +
      ggplot2::geom_vline(xintercept = round(tao[quantile(z,qn)+1],2), linetype = "dotted") +
      ggplot2::geom_hline(yintercept = round(dplyr::pull(v$SUM.pi, mean),4), linetype = "dotted") +
      ggplot2::annotate("text",x = position, y = 1,
               label = paste0(expression(tau[z])," ==", round(tao[quantile(z,qn)+1],2)),
               hjust = pos_just, vjust = 1,parse = T) +
      ggplot2::annotate("text",x = 0, y = round(dplyr::pull(v$SUM.pi, mean),4),
               label = paste0(expression(pi)," == ", round(dplyr::pull(v$SUM.pi, mean),4)),
               hjust = 0, vjust = -2.5,parse = T) +
      ggthemes::theme_tufte() +
      ggplot2::theme(axis.line = ggplot2::element_line(colour = "black"))
    if(intervals){
      S <- S + ggplot2::geom_ribbon(data = ribbon, ggplot2::aes(x= x, ymin = 0, ymax = 1), 
                                    alpha = .1, fill = "red") +
        ggplot2::geom_ribbon(ggplot2::aes(x = t, ymin = lower, ymax = upper), fill = "gray50", alpha = 0.3)
    }
    if (summary) {
      return(list(h,S,SUM))
    } else{
      return(list(h,S))
    }
  }
