use std::{
    env,
    ffi::{OsStr, OsString},
    fs, io,
    io::{Error, ErrorKind},
    path::{Path, PathBuf},
    process::{exit, Command},
};

#[cfg(target_family = "unix")]
use std::os::unix::ffi::OsStrExt;

#[cfg(target_family = "windows")]
use std::os::windows::ffi::OsStringExt;









const ENVVAR_R_INCLUDE_DIR: &str = "R_INCLUDE_DIR";
const ENVVAR_R_HOME: &str = "R_HOME";




const ENVVAR_R_VERSION: &str = "LIBRSYS_R_VERSION";


#[cfg(not(feature = "use-bindgen"))]
const ENVVAR_BINDINGS_PATH: &str = "LIBRSYS_BINDINGS_PATH";



#[cfg(feature = "use-bindgen")]
const ENVVAR_LIBCLANG_INCLUDE_PATH: &str = "LIBRSYS_LIBCLANG_INCLUDE_PATH";



#[cfg(feature = "use-bindgen")]
const ENVVAR_BINDINGS_OUTPUT_PATH: &str = "LIBRSYS_BINDINGS_OUTPUT_PATH";

#[derive(Debug)]
struct InstallationPaths {
    r_home: PathBuf,
    include: PathBuf,
    library: PathBuf,
}

impl InstallationPaths {
    fn get_r_binary(&self) -> PathBuf {
        if cfg!(windows) {
            Path::new(&self.library).join("R.exe")
        } else {
            Path::new(&self.r_home).join("bin").join("R")
        }
    }
}

#[allow(dead_code)]
#[derive(Debug)]
struct RVersionInfo {
    major: String,
    minor: String,
    patch: String,
    devel: bool,
    full: String,
}

impl RVersionInfo {


    fn get_r_bindings_filename(&self, target_os: &str, target_arch: &str) -> PathBuf {
        let devel_suffix = if self.devel { "-devel" } else { "" };
        let major = &self.major;
        let minor = &self.minor;
        PathBuf::from(format!(
            "bindings-{target_os}-{target_arch}-R{major}.{minor}{devel_suffix}.rs"
        ))
    }
}

#[derive(Debug)]
enum EnvVarError {
    EnvVarNotPresent,
    InvalidEnvVar(&'static str),
    RInvocationError(io::Error),
    InvalidROutput(&'static str),
}



#[cfg(target_family = "unix")]
fn byte_array_to_os_string(bytes: &[u8]) -> OsString {
    let os_str = OsStr::from_bytes(bytes);
    os_str.to_os_string()
}

#[link(name = "kernel32")]
#[cfg(target_family = "windows")]
extern "system" {
    #[link_name = "GetConsoleCP"]
    fn get_console_code_page() -> u32;
    #[link_name = "MultiByteToWideChar"]
    fn multi_byte_to_wide_char(
        CodePage: u32,
        dwFlags: u32,
        lpMultiByteStr: *const u8,
        cbMultiByte: i32,
        lpWideCharStr: *mut u16,
        cchWideChar: i32,
    ) -> i32;
}



#[cfg(target_family = "windows")]
fn wide_from_console_string(bytes: &[u8]) -> Vec<u16> {
    assert!(bytes.len() < std::i32::MAX as usize);
    let mut wide;
    let mut len;
    unsafe {
        let cp = get_console_code_page();
        len = multi_byte_to_wide_char(
            cp,
            0,
            bytes.as_ptr() as *const u8,
            bytes.len() as i32,
            std::ptr::null_mut(),
            0,
        );
        wide = Vec::with_capacity(len as usize);
        len = multi_byte_to_wide_char(
            cp,
            0,
            bytes.as_ptr() as *const u8,
            bytes.len() as i32,
            wide.as_mut_ptr(),
            len,
        );
        wide.set_len(len as usize);
    }
    wide
}

#[cfg(target_family = "windows")]
fn byte_array_to_os_string(bytes: &[u8]) -> OsString {

    let wide = wide_from_console_string(bytes);

    OsString::from_wide(&wide)
}


fn r_command<S: AsRef<OsStr>>(r_binary: S, script: &str) -> io::Result<OsString> {
    let out = Command::new(r_binary)
        .args(&["-s", "-e", script])
        .output()?;


    if !out.stderr.is_empty() {
        println!(
            "> {}",
            byte_array_to_os_string(&out.stderr)
                .as_os_str()
                .to_string_lossy()
        );
    }

    Ok(byte_array_to_os_string(&out.stdout))
}


fn get_r_home() -> io::Result<PathBuf> {

    if let Some(r_home) = env::var_os(ENVVAR_R_HOME) {
        return Ok(PathBuf::from(r_home));
    }




    let rout = r_command("R", r#"cat(normalizePath(R.home()))"#)?;
    if !rout.is_empty() {
        Ok(PathBuf::from(rout))
    } else {
        Err(Error::new(ErrorKind::Other, "Cannot find R home."))
    }
}


fn get_r_library(r_home: &Path) -> PathBuf {
    let pkg_target_arch = env::var("CARGO_CFG_TARGET_ARCH").unwrap();
    match (cfg!(windows), pkg_target_arch.as_str()) {

        (true, "x86_64") => Path::new(r_home).join("bin").join("x64"),
        (true, "x86") => Path::new(r_home).join("bin").join("i386"),
        (true, _) => panic!("Unknown architecture"),

        (false, _) => Path::new(r_home).join("lib"),
    }
}


fn get_r_include(r_home: &Path, library: &Path) -> io::Result<PathBuf> {

    if let Some(include) = env::var_os(ENVVAR_R_INCLUDE_DIR) {
        return Ok(PathBuf::from(include));
    }



    let r_binary = InstallationPaths {
        r_home: r_home.to_path_buf(),
        include: PathBuf::new(), // get_r_binary() doesn't use `include` so fill with an empty PathBuf.
        library: library.to_path_buf(),
    }
    .get_r_binary();

    let rout = r_command(&r_binary, r#"cat(normalizePath(R.home('include')))"#)?;
    if !rout.is_empty() {
        Ok(PathBuf::from(rout))
    } else {
        Err(Error::new(ErrorKind::Other, "Cannot find R include."))
    }
}

fn probe_r_paths() -> io::Result<InstallationPaths> {

    let r_home = get_r_home()?;


    let library = get_r_library(&r_home);


    let include = get_r_include(&r_home, &library)?;

    Ok(InstallationPaths {
        r_home,
        include,
        library,
    })
}


fn parse_r_version(r_version: String) -> Result<RVersionInfo, EnvVarError> {
    let full = r_version.clone();

    let (r_version, devel) = match *r_version.split('-').collect::<Vec<&str>>().as_slice() {
        [r_version, devel] => (r_version, Some(devel)),
        [r_version] => (r_version, None),

        _ => return Err(EnvVarError::InvalidEnvVar("Invalid format")),
    };


    let r_version_split = r_version
        .split('.')
        .map(|s| {






            if !s.is_empty() && s.chars().all(|c| c.is_digit(10)) {
                Some(s)
            } else {
                None
            }
        })
        .collect::<Vec<Option<&str>>>();

    let (major, minor, patch) = match *r_version_split.as_slice() {

        [] | [None, ..] => return Err(EnvVarError::InvalidEnvVar("Cannot find R major version")),
        [_, None, ..] => return Err(EnvVarError::InvalidEnvVar("Cannot find R minor version")),
        [_, _, None, ..] => return Err(EnvVarError::InvalidEnvVar("Cannot find R patch level")),

        [Some(major), Some(minor), Some(patch)] => {
            (major.to_string(), minor.to_string(), patch.to_string())
        }

        _ => return Err(EnvVarError::InvalidEnvVar("Invalid format")),
    };

    let devel = match devel {
        Some("devel") => true,
        Some(_) => {
            return Err(EnvVarError::InvalidEnvVar(
                "Cannot find R development status",
            ))
        }
        None => false,
    };

    Ok(RVersionInfo {
        major,
        minor,
        patch,
        devel,
        full,
    })
}

fn get_r_version_from_env(r_version_env_var: &str) -> Result<RVersionInfo, EnvVarError> {
    std::env::var(r_version_env_var)

        .map_err(|_| EnvVarError::EnvVarNotPresent)
        .and_then(parse_r_version)
}

fn get_r_version_from_r(r_paths: &InstallationPaths) -> Result<RVersionInfo, EnvVarError> {
    let r_binary = r_paths.get_r_binary();














    let out = r_command(
        &r_binary,
        r#"cat(sprintf('%s.%s%s\n', R.version$major, R.version$minor, if(isTRUE(grepl('devel', R.version$status, fixed = TRUE))) '-devel' else ''))"#,
    )
        .map_err(EnvVarError::RInvocationError)?;

    let out = out.as_os_str().to_string_lossy().into_owned();
    let mut lines = out.lines();


    match lines.next() {
        Some(v) => parse_r_version(v.to_string()),
        None => Err(EnvVarError::InvalidROutput("Cannot find R version")),
    }
}

fn get_r_version(
    r_version_env_var: &str,
    r_paths: &InstallationPaths,
) -> Result<RVersionInfo, EnvVarError> {

    match get_r_version_from_env(r_version_env_var) {

        Ok(v) => Ok(v),

        Err(EnvVarError::EnvVarNotPresent) => get_r_version_from_r(r_paths),


        e @ Err(_) => e,
    }
}

fn set_r_version_vars(ver: &RVersionInfo) {
    println!("cargo:r_version_major={}", ver.major); // Becomes DEP_R_R_VERSION_MAJOR for clients
    println!("cargo:r_version_minor={}", ver.minor); // Becomes DEP_R_R_VERSION_MINOR for clients
    println!("cargo:r_version_patch={}", ver.patch); // Becomes DEP_R_R_VERSION_PATCH for clients
    println!("cargo:r_version_devel={}", ver.devel); // Becomes DEP_R_R_VERSION_DEVEL for clients
}

#[cfg(feature = "use-bindgen")]
fn get_non_api() -> std::collections::HashSet<String> {



    const REQUIRED_NON_API: [&str; 6] = [
        "R_CStackLimit",
        "R_CleanTempDir",
        "R_RunExitFinalizers",
        "Rf_endEmbeddedR",
        "Rf_initialize_R",
        "setup_Rmainloop",
    ];




    let non_api = include_str!("./nonAPI.txt")
        .lines()
        .filter(|e| !REQUIRED_NON_API.contains(e))
        .map(|s| s.to_string());

    std::collections::HashSet::from_iter(non_api)
}

#[cfg(feature = "use-bindgen")]

fn generate_bindings(r_paths: &InstallationPaths, version_info: &RVersionInfo) {
    use clang::EntityKind::*;
    use clang::*;
    use std::collections::HashSet;

    let clang = Clang::new().unwrap();
    let index = Index::new(&clang, false, false);


    let tu = index
        .parser("wrapper.h")
        .arguments(&[format!("-I{}", r_paths.include.display())])
        .skip_function_bodies(true)
        .detailed_preprocessing_record(true)
        .parse()
        .unwrap();



    let r_ast_entities: HashSet<_> = tu
        .get_entity()
        .get_children()
        .into_iter()
        .filter(|x| !x.is_in_system_header())
        .collect();


    let mut allowlist: HashSet<_> = r_ast_entities
        .into_iter()
        .filter(|e| {




            !e.is_anonymous()
        })
        .flat_map(|e| match e.get_kind() {
            EnumDecl | FunctionDecl | StructDecl | TypedefDecl | VarDecl | UnionDecl
            | MacroDefinition | MacroExpansion => e.get_name(),
            _ => None,
        })
        .collect();



    allowlist.insert("R_GE_version".to_string());


    let allowlist_pattern = allowlist

        .difference(&get_non_api())
        .cloned()
        .collect::<Vec<_>>();
    let allowlist_pattern = allowlist_pattern.join("|");




    let mut bindgen_builder = bindgen::Builder::default()
        .allowlist_function(&allowlist_pattern)
        .allowlist_var(&allowlist_pattern)
        .allowlist_type(&allowlist_pattern)
        .translate_enum_integer_types(true)


        .header("wrapper.h")


        .parse_callbacks(Box::new(bindgen::CargoCallbacks));

    let target = env::var("TARGET").expect("Could not get the target triple");
    let target_os = std::env::var("CARGO_CFG_TARGET_OS").unwrap();
    let target_arch = env::var("CARGO_CFG_TARGET_ARCH").unwrap();

    println!(
        "Generating bindings for target: {target}, os: {target_os}, architecture: {target_arch}"
    );


    bindgen_builder = bindgen_builder.clang_args(&[
        format!("-I{}", r_paths.include.display()),
        format!("--target={target}"),
    ]);


    if let Some(alt_include) = env::var_os(ENVVAR_LIBCLANG_INCLUDE_PATH) {
        bindgen_builder =
            bindgen_builder.clang_arg(format!("-I{}", PathBuf::from(alt_include).display()));
    }



    let bindgen_builder = bindgen_builder
        .blocklist_item("M_E")
        .blocklist_item("M_LOG2E")
        .blocklist_item("M_LOG10E")
        .blocklist_item("M_LN2")
        .blocklist_item("M_LN10")
        .blocklist_item("M_PI")
        .blocklist_item("M_PI_2")
        .blocklist_item("M_PI_4")
        .blocklist_item("M_1_PI")
        .blocklist_item("M_2_PI")
        .blocklist_item("M_2_SQRTPI")
        .blocklist_item("M_SQRT2")
        .blocklist_item("M_SQRT1_2");


    let bindgen_builder = bindgen_builder.blocklist_item("VECTOR_PTR");


    let bindings = bindgen_builder
        .raw_line(format!(
            "/* libR-sys version: {} */",
            env!("CARGO_PKG_VERSION")
        ))
        .raw_line(format!(
            "/* bindgen clang version: {} */",
            bindgen::clang_version().full
        ))
        .raw_line(format!("/* clang-rs version: {} */", clang::get_version()))
        .raw_line(format!("/* r version: {} */", version_info.full))
        .generate_comments(true)
        .parse_callbacks(Box::new(TrimCommentsCallbacks))
        .clang_arg("-fparse-all-comments")
        .generate()

        .expect("Unable to generate bindings");

    bindings.emit_warnings();


    let out_path = PathBuf::from(env::var_os("OUT_DIR").unwrap());

    bindings
        .write_to_file(out_path.join("bindings.rs"))
        .expect("Couldn't write bindings to default output path!");


    if let Some(alt_target) = env::var_os(ENVVAR_BINDINGS_OUTPUT_PATH) {
        let out_path = PathBuf::from(alt_target);

        if !out_path.exists() {
            fs::create_dir(&out_path).expect(&format!(
                "Couldn't create output directory for bindings: {}",
                out_path.display()
            ));
        }

        let bindings_file_full = version_info.get_r_bindings_filename(&target_os, &target_arch);
        let out_file = out_path.join(bindings_file_full);

        bindings
            .write_to_file(&out_file)
            .expect(&format!("Couldn't write bindings: {}", out_file.display()));
    } else {
        println!(
            "Warning: Couldn't write the bindings since `LIBRSYS_BINDINGS_OUTPUT_PATH` is not set."
        );
    }
}

#[cfg(not(feature = "use-bindgen"))]

fn retrieve_prebuild_bindings(version_info: &RVersionInfo) {
    let target_os = std::env::var("CARGO_CFG_TARGET_OS").unwrap();
    let target_arch = env::var("CARGO_CFG_TARGET_ARCH").unwrap();
    let bindings_path = PathBuf::from(
        env::var_os(ENVVAR_BINDINGS_PATH).unwrap_or_else(|| OsString::from("bindings")),
    );


    let bindings_file_full = version_info.get_r_bindings_filename(&target_os, &target_arch);
    let bindings_file_novers = PathBuf::from(format!("bindings-{target_os}-{target_arch}.rs"));

    let mut from = bindings_path.join(bindings_file_full);
    if !from.exists() {
        from = bindings_path.join(bindings_file_novers);
        if !from.exists() {
            panic!(
                "Cannot find libR-sys bindings file for R {}.{}.{}{} on {} in {}. Consider compiling with --features use-bindgen.",
                version_info.major, version_info.minor, version_info.patch, version_info.devel, target_os, bindings_path.display()
            )
        } else {
            println!(
                "Warning: using generic {}-{} libR-sys bindings. These may not work for R {}.{}.{}{}.",
                target_os, target_arch, version_info.major, version_info.minor, version_info.patch, version_info.devel
            );
        }
    }

    fs::copy(
        &from,
        PathBuf::from(env::var_os("OUT_DIR").unwrap()).join("bindings.rs"),
    )
    .expect("No precomputed bindings available!");
    println!("cargo:rerun-if-changed={}", from.display());
}


#[cfg(feature = "use-bindgen")]
#[derive(Debug)]
struct TrimCommentsCallbacks;

#[cfg(feature = "use-bindgen")]
impl bindgen::callbacks::ParseCallbacks for TrimCommentsCallbacks {
    fn process_comment(&self, comment: &str) -> Option<String> {
        let trim_comment = comment.trim();
        Some(trim_comment.to_string())
    }
}

fn main() {
    let r_paths = probe_r_paths();

    let r_paths = match r_paths {
        Ok(result) => result,
        Err(error) => {
            println!("Problem locating local R install: {:?}", error);
            exit(1);
        }
    };

    println!("cargo:rustc-env=R_HOME={}", r_paths.r_home.display());
    println!("cargo:r_home={}", r_paths.r_home.display()); // Becomes DEP_R_R_HOME for clients





    if let Ok(r_library) = r_paths.library.canonicalize() {
        println!("cargo:rustc-link-search={}", r_library.display());
    }
    println!("cargo:rustc-link-lib=dylib=R");

    println!("cargo:rerun-if-changed=build.rs");
    println!("cargo:rerun-if-changed=wrapper.h");


    let version_info =
        get_r_version(ENVVAR_R_VERSION, &r_paths).expect("Could not obtain R version");
    set_r_version_vars(&version_info);

    #[cfg(feature = "use-bindgen")]
    generate_bindings(&r_paths, &version_info);
    #[cfg(not(feature = "use-bindgen"))]
    retrieve_prebuild_bindings(&version_info);
}
