\name{dwt}
\alias{dwt}
\alias{dwt.nondyadic}
\alias{idwt}
\title{Discrete Wavelet Transform (DWT)}
\description{
  This function performs a level \eqn{J} decomposition of the input
  vector or time series using the pyramid algorithm (Mallat 1989).
}
\usage{
dwt(x, wf="la8", n.levels=4, boundary="periodic")
dwt.nondyadic(x)
}
\arguments{
  \item{x}{a vector or time series containing the data be to
    decomposed.  This must be a dyadic length vector (power of 2).}
  \item{wf}{
    Name of the wavelet filter to use in the decomposition.  By default
    this is set to \code{"la8"}, the Daubechies orthonormal compactly
    supported wavelet of length \eqn{L=8} (Daubechies, 1992), least
    asymmetric family. 
  }
  \item{n.levels}{
    Specifies the depth of the decomposition.  This must be a number
    less than or equal to
    \eqn{\log(\mbox{length}(x),2)}{log(length(x),2)}.
  }
  \item{boundary}{
    Character string specifying the boundary condition.   If
    \code{boundary=="periodic"} the default, then the vector you
    decompose is assumed to be periodic on its defined interval,\cr
    if \code{boundary=="reflection"}, the vector beyond its boundaries
    is assumed to be a symmetric reflection of itself.
  }
}
\value{
  Basically, a list with the following components
  \item{d?}{Wavelet coefficient vectors.}
  \item{s?}{Scaling coefficient vector.}
  \item{wavelet}{Name of the wavelet filter used.}
  \item{boundary}{How the boundaries were handled.}
}
\details{
  The code implements the one-dimensional DWT using the pyramid
  algorithm (Mallat, 1989).  The actual transform is performed in C
  using pseudocode from Percival and Walden (2001).  That means
  convolutions, not inner products, are used to apply the wavelet
  filters.

  For a non-dyadic length vector or time series, \code{dwt.nondyadic}
  pads with zeros, performs the orthonormal DWT on this dyadic length
  series and then truncates the wavelet coefficient vectors
  appropriately.
}
\references{
  Daubechies, I. (1992)
  \emph{Ten Lectures on Wavelets},
  CBMS-NSF Regional Conference Series in Applied Mathematics,
  SIAM: Philadelphia.

  Gencay, R., F. Selcuk and B. Whitcher (2001)
  \emph{An Introduction to Wavelets and Other Filtering Methods in
    Finance and Economics},
  Academic Press.
  
  Mallat, S. G. (1989)
  A theory for multiresolution signal decomposition: the wavelet
  representation,
  \emph{IEEE Transactions on Pattern Analysis and Machine Intelligence},
  \bold{11}, No. 7, 674-693.

  Percival, D. B. and A. T. Walden (2000)
  \emph{Wavelet Methods for Time Series Analysis},
  Cambridge University Press.
}
\seealso{
  \code{\link{modwt}}, \code{\link{mra}}.
}
\examples{
## Figures 4.17 and 4.18 in Gencay, Selcuk and Whitcher (2001).
data(ibm)     
ibm.returns <- diff(log(ibm))
## Haar
ibmr.haar <- dwt(ibm.returns, "haar")
names(ibmr.haar) <- c("w1", "w2", "w3", "w4", "v4")
## plot partial Haar DWT for IBM data
par(mfcol=c(6,1), pty="m", mar=c(5-2,4,4-2,2))
plot.ts(ibm.returns, axes=FALSE, ylab="", main="(a)")
for(i in 1:4)
  plot.ts(up.sample(ibmr.haar[[i]], 2^i), type="h", axes=FALSE,
          ylab=names(ibmr.haar)[i])
plot.ts(up.sample(ibmr.haar$v4, 2^4), type="h", axes=FALSE,
        ylab=names(ibmr.haar)[5])
axis(side=1, at=seq(0,368,by=23), 
     labels=c(0,"",46,"",92,"",138,"",184,"",230,"",276,"",322,"",368))
## LA(8)
ibmr.la8 <- dwt(ibm.returns, "la8")
names(ibmr.la8) <- c("w1", "w2", "w3", "w4", "v4")
## must shift LA(8) coefficients
ibmr.la8$w1 <- c(ibmr.la8$w1[-c(1:2)], ibmr.la8$w1[1:2])
ibmr.la8$w2 <- c(ibmr.la8$w2[-c(1:2)], ibmr.la8$w2[1:2])
for(i in names(ibmr.la8)[3:4])
  ibmr.la8[[i]] <- c(ibmr.la8[[i]][-c(1:3)], ibmr.la8[[i]][1:3])
ibmr.la8$v4 <- c(ibmr.la8$v4[-c(1:2)], ibmr.la8$v4[1:2])
## plot partial LA(8) DWT for IBM data
par(mfcol=c(6,1), pty="m", mar=c(5-2,4,4-2,2))
plot.ts(ibm.returns, axes=FALSE, ylab="", main="(b)")
for(i in 1:4)
  plot.ts(up.sample(ibmr.la8[[i]], 2^i), type="h", axes=FALSE,
          ylab=names(ibmr.la8)[i])
plot.ts(up.sample(ibmr.la8$v4, 2^4), type="h", axes=FALSE,
        ylab=names(ibmr.la8)[5])
axis(side=1, at=seq(0,368,by=23), 
  labels=c(0,"",46,"",92,"",138,"",184,"",230,"",276,"",322,"",368))
}
\author{B. Whitcher}
\keyword{ts}
