% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/racusum_sim.R
\name{racusum_arloc_h_sim}
\alias{racusum_arloc_h_sim}
\title{Compute alarm threshold (Out of Control ARL) of RA-CUSUM control charts using simulation}
\usage{
racusum_arloc_h_sim(L0, df, coeff, coeff2, R0 = 1, RA = 2, RQ = 1,
  m = 100, nc = 1, verbose = FALSE)
}
\arguments{
\item{L0}{double. Prespecified in-control Average Run Length.}

\item{df}{DataFrame. First column are Parsonnet Score values within a range of zero to 100 representing
the preoperative patient risk. The second column are binary (0/1) outcome values of each operation.}

\item{coeff}{NumericVector. Estimated coefficients \eqn{\alpha}{alpha} and \eqn{\beta}{beta}
from the binary logistic regression model.}

\item{coeff2}{NumericVector. Estimated coefficients \eqn{\alpha}{alpha} and \eqn{\beta}{beta}
from the binary logistic regression model of a resampled dataset.}

\item{R0}{double. Odds ratio of death under the null hypotheses.}

\item{RA}{double. Odds ratio of death under the alternative hypotheses. Detecting deterioration
in performance with increased mortality risk by doubling the odds Ratio RA=2. Detecting
improvement in performance with decreased mortality risk by halving the odds ratio of death
RA=1/2.}

\item{RQ}{double. Defines the performance of a surgeon with the odds ratio ratio of death Q.}

\item{m}{integer. Number of simulation runs.}

\item{nc}{integer. Number of cores.}

\item{verbose}{boolean. If TRUE verbose output is included, if FALSE a quiet calculation of h is done.}
}
\value{
Returns a single value which is the control limit h for a given in-control ARL.
}
\description{
Compute alarm threshold (Out of Control ARL) of RA-CUSUM control charts using
simulation.
}
\details{
The function \code{racusum_arloc_h_sim} determines the control limit for given in-control ARL (L0) by applying a
multi-stage search procedure which includes secant rule and the parallel version of \code{\link{racusum_arloc_sim}}
using \code{\link{mclapply}}.
}
\examples{
\dontrun{
require("vlad")
# Set seed for reproducibility
RNGkind("L'Ecuyer-CMRG")
set.seed(1234)
parallel::mc.reset.stream()
# Datasets
data("cardiacsurgery")
s5000 <- dplyr::sample_n(cardiacsurgery, size=5000, replace=TRUE)
df1 <- subset(cardiacsurgery, select=c(Parsonnet, status))
df2 <- subset(s5000, select=c(Parsonnet, status))

# Estimate coefficients from logit model
coeff1 <- round(coef(glm(status~Parsonnet, data=df1, family="binomial")), 3)
coeff2 <- round(coef(glm(status~Parsonnet, data=df2, family="binomial")), 3)

# Number of simulation runs
m <- 10^3

# Deterioration:
# 1. Determine critical value for given ARL
racusum_arloc_h_sim(L0=370, df=df1, coeff=coeff1, coeff2=coeff2, m=m, RA=2, nc=6)
# h=2.030933

# 2. Determine ARL and Standard Error
RLS <- do.call(c, parallel::mclapply(1:m, racusum_arloc_sim, h=2.035, df=df1, RA=2, coeff=coeff1,
                                     coeff2=coeff2, mc.cores=6))
data.frame(cbind("ARL"=mean(RLS), "ARLSE"=sd(RLS)/sqrt(m)))
# ARL=371.125; ARLSE=11.36053

# Improvement:
# 1. Determine critical value for given ARL
racusum_arloc_h_sim(L0=370, df=df1, coeff=coeff1, coeff2=coeff2, m=m, RA=1/2, nc=6)
# h=1.710999
#
# 2. Determine ARL and Standard Error
RLS <- do.call(c, parallel::mclapply(1:m, racusum_arloc_sim, h=1.760, df=df1, RA=1/2, coeff=coeff1,
                                     coeff2=coeff2, mc.cores=6))
data.frame(cbind("ARL"=mean(RLS), "ARLSE"=sd(RLS)/sqrt(m)))
# ARL=399.613; ARLSE=10.7601
}
}
\author{
Philipp Wittenberg
}
