% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/h_visibility.R
\name{h_visibility}
\alias{h_visibility}
\title{Compute horizontal visibility}
\usage{
h_visibility(
  data,
  position = c(0, 0, 0),
  layer_tickness = 0.1,
  angular_res = 1,
  scene_radius
)
}
\arguments{
\item{data}{LAS class object containing the xyz coordinates of a 3D point
cloud.}

\item{position}{vector of length 3 containing the xyz coordinates of the
animal location. Default = c(0,0,0).}

\item{layer_tickness}{numeric. The thickness of the disk that defines the viewshed.}

\item{angular_res}{numeric. The angular resolution of a single sightline.
Default = 1.}

\item{scene_radius}{(optional) numeric. Defines the radius of the scene
relative to the animal position. Can be used to apply a cut-off distance to
visibility analyses.}
}
\value{
A list containing a data.table of the visibility
(\code{$visibility}) as a function of distance to the animal location (r), a data.table of
the viewshed (\code{$viewshed}) defined as the radius (r) for each azimuth (phi) with non occluded
sightlines distance set to scene_radius, a data.table (\code{$vegetation_distance}) of the vegetation distance (r)
in each azimuth (phi) for occluded sightlines only and different viewshed metrics (\code{$metrics}).
The metrics are: the viewshed fractal dimension (computed from Chen, 2020), the
viewshed area (i.e. the visible area), the proportion visible (the area visible
 / potential area visible), the viewshed coefficient (the area under the curve of
visibility as a function of distance) and the vegetation fractal dimension.
}
\description{
Computes the horizontal visibility in a viewshed defined as a disk.
}
\examples{
\donttest{
#- import the tree_line_plot dataset
file <- system.file("extdata", "tree_line_plot.laz", package="viewshed3d")
tls <- lidR::readLAS(file)

center <- c(0,0,2) # defines the scene center for the entire process
angle <- 1 # defines the angular resolution for the entire process

#- remove noise to avoid visibility estimates error
tls_clean <- viewshed3d::denoise_scene(tls,method="sd",
                                       filter=6)

#- compute the horizontal visibility.
view.data <- viewshed3d::h_visibility(data = tls_clean,
                                      position = center,
                                      angular_res = angle,
                                      scene_radius = 17)

# viewshed metrics
view.data$metrics

# plot the view sheds in a radial plot
plotrix::radial.plot(view.data$viewshed$r,rp.type = "p",poly.col = "blue",
                     radial.lim = c(0,max(view.data$viewshed$r)))
}
}
\references{
Chen, Y. (2020). Two Sets of Simple Formulae to Estimating Fractal Dimension
of Irregular Boundaries. Mathematical Problems in Engineering, 2020.
https://doi.org/10.1155/2020/7528703
}
