% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/umx_build_high_level_models.R
\name{umxTwoStage}
\alias{umxTwoStage}
\alias{umxMR}
\title{Build a SEM implementing the instrumental variable design}
\usage{
umxTwoStage(
  formula = Y ~ X,
  instruments = ~qtl,
  data,
  subset,
  contrasts = NULL,
  name = "IV_model",
  tryHard = c("no", "yes", "ordinal", "search"),
  ...
)
}
\arguments{
\item{formula}{The structural equation to be estimated (default = Y ~ X). A constant is implied if not explicitly deleted.}

\item{instruments}{A one-sided formula specifying instrumental variables (default = qtl).}

\item{data}{Frame containing the variables in the model.}

\item{subset}{(optional) vector specifying a subset of observations to be used in fitting the model.}

\item{contrasts}{an optional list (not supported)}

\item{name}{for the model (default = "IVmodel")}

\item{tryHard}{Default ('no') uses normal mxRun. "yes" uses mxTryHard. Other options: "ordinal", "search"}

\item{...}{arguments to be passed along. (not supported)}
}
\value{
\itemize{
\item \code{\link[=mxModel]{mxModel()}}
}
}
\description{
\code{umxMR} (\code{umxTwoStage}) implements a Mendelian randomization or instrumental variable Structural Equation Model.
For ease of learning, the parameters follow the \code{tsls()} function in the sem package.
}
\details{
The example is a \href{https://en.wikipedia.org/wiki/Mendelian_randomization}{Mendelian Randomization}
analysis showing the utility of SEM over two-stage regression.

The following figure shows how the MR model appears as a path diagram:

\if{html}{\figure{TSLS.png}{options: width=50\% alt="Figure: Mendelian Randomisation analysis.png"}}
\if{latex}{\figure{TSLS.pdf}{options: width=7cm}}
}
\examples{
\dontrun{
library(umx)


# ====================================
# = Mendelian Randomization analysis =
# ====================================

df = umx_make_MR_data(10e4)
m1 = umxMR(Y ~ X, instruments = ~ qtl, data = df)
parameters(m1)
plot(m1, means = FALSE, min="") # help DiagrammaR layout the plot.
m2 = umxModify(m1, "qtl_to_X", comparison=TRUE, tryHard="yes", name="QTL_affects_X") # yip
m3 = umxModify(m1, "X_to_Y"  , comparison=TRUE, tryHard="yes", name="X_affects_Y") # nope
plot(m3, means = FALSE)

# Errant analysis using ordinary least squares regression (WARNING this result is CONFOUNDED!!)
m1 = lm(Y ~ X    , data = df); coef(m1) # incorrect .35 effect of X on Y
m1 = lm(Y ~ X + U, data = df); coef(m1) # Controlling U reveals the true 0.1 beta weight


df = umx_make_MR_data(10e4) 
m1 = umxMR(Y ~ X, instruments = ~ qtl, data = df)
coef(m1)

# ======================
# = Now with sem::tsls =
# ======================
# library(sem) # may require you to install X11
m2 = sem::tsls(formula = Y ~ X, instruments = ~ qtl, data = df)
coef(m2)

# Try with missing value for one subject: A benefit of the FIML approach in OpenMx.
m3 = tsls(formula = Y ~ X, instruments = ~ qtl, data = (df[1, "qtl"] = NA))
}
}
\references{
\itemize{
\item Fox, J. (1979) Simultaneous equation models and two-stage least-squares. In Schuessler, K. F. (ed.) \emph{Sociological Methodology}, Jossey-Bass.
}
\itemize{
\item Greene, W. H. (1993) \emph{Econometric Analysis}, Second Edition, Macmillan.
\item Sekula, P., Del Greco, M. F., Pattaro, C., & Kottgen, A. (2016). Mendelian Randomization as an Approach to
Assess Causality Using Observational Data. \emph{Journal of the American Society of Nephrology}, \strong{27}), 3253-3265. \doi{10.1681/ASN.2016010098}
}
}
\seealso{
\itemize{
\item \code{\link[=umx_make_MR_data]{umx_make_MR_data()}}, \code{\link[=umxDiffMZ]{umxDiffMZ()}}, \code{\link[=umxDoC]{umxDoC()}}, \code{\link[=umxDiscTwin]{umxDiscTwin()}}
}

Other Super-easy helpers: 
\code{\link{umxEFA}()},
\code{\link{umx}}
}
\concept{Super-easy helpers}
