% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/simulateGSMAR.R
\name{simulateGSMAR}
\alias{simulateGSMAR}
\title{Simulate values from GMAR, StMAR, and G-StMAR processes}
\usage{
simulateGSMAR(gsmar, nsimu, initvalues, ntimes = 1, drop = TRUE)
}
\arguments{
\item{gsmar}{object of class \code{'gsmar'} created with the function \code{fitGSMAR} or \code{GSMAR}.}

\item{nsimu}{a positive integer specifying how many values (ahead from \code{initvalues}) will be simulated.}

\item{initvalues}{a numeric vector with length \code{>=p} specifying the initial values for the simulation. The \strong{last}
element will be used as the initial value for the first lag, the second last element will be initial value for the second lag, etc.
If not specified, initial values will be simulated from the process's stationary distribution.}

\item{ntimes}{a positive integer specifying how many sets of simulations should be performed.}

\item{drop}{if \code{TRUE} (default) then the components of the returned list are coerced to lower dimension if \code{ntimes==1},
i.e., \code{$sample} and \code{$component} will be vectors and \code{$mixing_weights} will be matrix.}
}
\value{
If \code{drop==TRUE} and \code{ntimes==1} (default): \code{$sample} and \code{$component} are vectors
 and \code{$mixing_weights} is a (\code{nsimu}\eqn{xM}) matrix. Otherwise, returns a list with...
  \describe{
    \item{\code{$sample}}{a size (\code{nsimu}\eqn{x}\code{ntimes}) matrix containing the simulated values.}
    \item{\code{$component}}{a size (\code{nsimu}\eqn{x}\code{ntimes}) matrix containing the information from which
     mixture component each value was generated from.}
    \item{\code{$mixing_weights}}{a size (\code{nsimu}\eqn{xMx}\code{ntimes}) array containing the mixing weights corresponding to the
     sample: the dimension \code{[i, , ]} is the time index, the dimension \code{[, i, ]} indicates the regime, and the dimension
     \code{[, , i]} indicates the i:th set of simulations.}
  }
}
\description{
\code{simulateGSMAR} simulates values from the specified GMAR, StMAR, or G-StMAR process.
 Can be utilized for forecasting future values of the process.
}
\details{
The argument \code{ntimes} is intended for forecasting: a GSMAR process can be forecasted by simulating its
 possible future values. One can perform a large number of sets of simulations and calculate the sample quantiles from
 the simulated values to obtain prediction intervals. See the forecasting example below for a hand-on demonstration.
}
\examples{
 \donttest{
 # GMAR model:
 params12 <- c(0.18, 0.93, 0.01, 0.86, 0.68, 0.02, 0.88)
 gmar12 <- GSMAR(p=1, M=2, params=params12, model="GMAR")
 sim12 <- simulateGSMAR(gmar12, nsimu=500)
 ts.plot(sim12$sample)
 ts.plot(sim12$component)
 ts.plot(sim12$mixing_weights, col=rainbow(2), lty=2)


 # G-StMAR model, with initial values:
 params12gs <- c(1.38, 0.88, 0.27, 3.8, 0.74, 3.15, 0.8, 3.6)
 gstmar12 <- GSMAR(p=1, M=c(1, 1), params=params12gs,
 model="G-StMAR")
 sim12gs <- simulateGSMAR(gstmar12, nsimu=500, initvalues=5:6)
 ts.plot(sim12gs$sample)
 ts.plot(sim12gs$component)
 ts.plot(sim12gs$mixing_weights, col=rainbow(2), lty=2)


 # FORECASTING EXAMPLE:
 # Restricted GMAR model, 10000 sets of simulations with initial values 6 and 6.2.
 params22r <- c(1.4, 1.8, 0.8, -0.1, 0.29, 3.18, 0.84)
 gmar22r <- GSMAR(p=2, M=2, params=params22r, model="GMAR",
  restricted=TRUE)
 sim22r <- simulateGSMAR(gmar22r, nsimu=5, initval=c(6, 6.2), ntimes=10000)
 apply(sim22r$sample, 1, median) # Point forecast
 apply(sim22r$sample, 1, quantile, probs=c(0.025, 0.975)) # 95\% interval
 apply(sim22r$mixing_weights, MARGIN=1:2, FUN=median) # mix.weight point forecast
 apply(sim22r$mixing_weights, MARGIN=1:2, FUN=quantile,
  probs=c(0.025, 0.975)) # mix.weight 95\% intervals
}
}
\references{
\itemize{
   \item Galbraith, R., Galbraith, J. 1974. On the inverses of some patterned matrices arising
           in the theory of stationary time series. \emph{Journal of Applied Probability} \strong{11}, 63-71.
   \item Kalliovirta L. (2012) Misspecification tests based on quantile residuals.
           \emph{The Econometrics Journal}, \strong{15}, 358-393.
   \item Kalliovirta L., Meitz M. and Saikkonen P. 2015. Gaussian Mixture Autoregressive model for univariate time series.
           \emph{Journal of Time Series Analysis}, \strong{36}, 247-266.
   \item Meitz M., Preve D., Saikkonen P. 2018. A mixture autoregressive model based on Student's t-distribution.
           arXiv:1805.04010 \strong{[econ.EM]}.
   \item Virolainen S. 2020. A mixture autoregressive model based on Gaussian and Student's t-distribution.	arXiv:2003.05221 [econ.EM].
 }
}
\seealso{
\code{\link{fitGSMAR}}, \code{\link{GSMAR}}, \code{\link{predict.gsmar}},
 \code{\link{add_data}}, \code{\link{condMoments}}, \code{\link{mixingWeights}}
}
