% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/tune_bayes.R
\name{tune_bayes}
\alias{tune_bayes}
\alias{tune_bayes.model_spec}
\alias{tune_bayes.workflow}
\title{Bayesian optimization of model parameters.}
\usage{
tune_bayes(object, ...)

\method{tune_bayes}{model_spec}(
  object,
  preprocessor,
  resamples,
  ...,
  iter = 10,
  param_info = NULL,
  metrics = NULL,
  objective = exp_improve(),
  initial = 5,
  control = control_bayes()
)

\method{tune_bayes}{workflow}(
  object,
  resamples,
  ...,
  iter = 10,
  param_info = NULL,
  metrics = NULL,
  objective = exp_improve(),
  initial = 5,
  control = control_bayes()
)
}
\arguments{
\item{object}{A \code{parsnip} model specification or a \code{\link[workflows:workflow]{workflows::workflow()}}.}

\item{...}{Options to pass to \code{\link[GPfit:GP_fit]{GPfit::GP_fit()}} (mostly for the \code{corr} argument).}

\item{preprocessor}{A traditional model formula or a recipe created using
\code{\link[recipes:recipe]{recipes::recipe()}}.}

\item{resamples}{An \code{rset()} object.}

\item{iter}{The maximum number of search iterations.}

\item{param_info}{A \code{\link[dials:parameters]{dials::parameters()}} object or \code{NULL}. If none is given,
a parameters set is derived from other arguments. Passing this argument can
be useful when parameter ranges need to be customized.}

\item{metrics}{A \code{\link[yardstick:metric_set]{yardstick::metric_set()}} object containing information on how
models will be evaluated for performance. The first metric in \code{metrics} is the
one that will be optimized.}

\item{objective}{A character string for what metric should be optimized or
an acquisition function object.}

\item{initial}{An initial set of results in a tidy format (as would result
from \code{\link[=tune_grid]{tune_grid()}}) or a positive integer. It is suggested that the number of
initial results be greater than the number of parameters being optimized.}

\item{control}{A control object created by \code{\link[=control_bayes]{control_bayes()}}}
}
\value{
A tibble of results that mirror those generated by \code{\link[=tune_grid]{tune_grid()}}.
However, these results contain an \code{.iter} column and replicate the \code{rset}
object multiple times over iterations (at limited additional memory costs).
}
\description{
\code{\link[=tune_bayes]{tune_bayes()}} uses models to generate new candidate tuning parameter
combinations based on previous results.
}
\details{
The optimization starts with a set of initial results, such as those
generated by \code{\link[=tune_grid]{tune_grid()}}. If none exist, the function will create several
combinations and obtain their performance estimates.

Using one of the performance estimates as the \emph{model outcome}, a Gaussian
process (GP) model is created where the previous tuning parameter combinations
are used as the predictors.

A large grid of potential hyperparameter combinations is predicted using
the model and scored using an \emph{acquisition function}. These functions
usually combine the predicted mean and variance of the GP to decide the best
parameter combination to try next. For more information, see the
documentation for \code{\link[=exp_improve]{exp_improve()}} and the corresponding package vignette.

The best combination is evaluated using resampling and the process continues.
}
\section{Parallel Processing}{


The \code{foreach} package is used here. To execute the resampling iterations in
parallel, register a parallel backend function. See the documentation for
\code{\link[foreach:foreach]{foreach::foreach()}} for examples.

For the most part, warnings generated during training are shown as they occur
and are associated with a specific resample when
\code{control_bayes(verbose = TRUE)}. They are (usually) not aggregated until the
end of processing.

For Bayesian optimization, parallel processing is used to estimate the
resampled performance values once a new candidate set of values are estimated.
}

\section{Initial Values}{


The results of \code{\link[=tune_grid]{tune_grid()}}, or a previous run of \code{\link[=tune_bayes]{tune_bayes()}} can be used
in the \code{initial} argument. \code{initial} can also be a positive integer. In this
case, a space-filling design will be used to populate a preliminary set of
results. For good results, the number of initial values should be more than
the number of parameters being optimized.
}

\section{Parameter Ranges and Values}{


In some cases, the tuning parameter values depend on the dimensions of the
data (they are said to contain \link[dials:unknown]{unknown} values). For
example, \code{mtry} in random forest models depends on the number of predictors.
In such cases, the unknowns in the tuning parameter object must be determined
beforehand and passed to the function via the \code{param_info} argument.
\code{\link[dials:finalize]{dials::finalize()}} can be used to derive the data-dependent parameters.
Otherwise, a parameter set can be created via \code{\link[dials:parameters]{dials::parameters()}}, and the
\code{dials} \code{update()} function can be used to specify the ranges or values.
}

\section{Performance Metrics}{


To use your own performance metrics, the \code{\link[yardstick:metric_set]{yardstick::metric_set()}} function
can be used to pick what should be measured for each model. If multiple
metrics are desired, they can be bundled. For example, to estimate the area
under the ROC curve as well as the sensitivity and specificity (under the
typical probability cutoff of 0.50), the \code{metrics} argument could be given:

\preformatted{
  metrics = metric_set(roc_auc, sens, spec)
}

Each metric is calculated for each candidate model.

If no metric set is provided, one is created:
\itemize{
\item For regression models, the root mean squared error and coefficient
of determination are computed.
\item For classification, the area under the ROC curve and overall accuracy
are computed.
}

Note that the metrics also determine what type of predictions are estimated
during tuning. For example, in a classification problem, if metrics are used
that are all associated with hard class predictions, the classification
probabilities are not created.

The out-of-sample estimates of these metrics are contained in a list column
called \code{.metrics}. This tibble contains a row for each metric and columns
for the value, the estimator type, and so on.

\code{\link[=collect_metrics]{collect_metrics()}} can be used for these objects to collapse the results
over the resampled (to obtain the final resampling estimates per tuning
parameter combination).
}

\section{Obtaining Predictions}{


When \code{control_bayes(save_pred = TRUE)}, the output tibble contains a list
column called \code{.predictions} that has the out-of-sample predictions for each
parameter combination in the grid and each fold (which can be very large).

The elements of the tibble are tibbles with columns for the tuning
parameters, the row number from the original data object (\code{.row}), the
outcome data (with the same name(s) of the original data), and any columns
created by the predictions. For example, for simple regression problems, this
function generates a column called \code{.pred} and so on. As noted above, the
prediction columns that are returned are determined by the type of metric(s)
requested.

This list column can be \code{unnested} using \code{\link[tidyr:nest]{tidyr::unnest()}} or using the
convenience function \code{\link[=collect_predictions]{collect_predictions()}}.
}

\section{Extracting Information}{


The \code{extract} control option will result in an additional function to be
returned called \code{.extracts}. This is a list column that has tibbles
containing the results of the user's function for each tuning parameter
combination. This can enable returning each model and/or recipe object that
is created during resampling. Note that this could result in a large return
object, depending on what is returned.

The control function contains an option (\code{extract}) that can be used to
retain any model or recipe that was created within the resamples. This
argument should be a function with a single argument. The value of the
argument that is given to the function in each resample is a workflow
object (see \code{\link[workflows:workflow]{workflows::workflow()}} for more information). Several
helper functions can be used to easily pull out the preprocessing
and/or model information from the workflow, such as
\code{\link[workflows:extract-workflow]{extract_preprocessor()}} and
\code{\link[workflows:extract-workflow]{extract_fit_parsnip()}}.

As an example, if there is interest in getting each parsnip model fit back,
one could use:
\preformatted{
  extract = function (x) extract_fit_parsnip(x)
}

Note that the function given to the \code{extract} argument is evaluated on
every model that is \emph{fit} (as opposed to every model that is \emph{evaluated}).
As noted above, in some cases, model predictions can be derived for
sub-models so that, in these cases, not every row in the tuning parameter
grid has a separate R object associated with it.
}

\examples{
\dontshow{if ((tune:::should_run_examples(suggests = "kernlab"))) (if (getRversion() >= "3.4") withAutoprint else force)(\{ # examplesIf}
library(recipes)
library(rsample)
library(parsnip)

# define resamples and minimal recipe on mtcars
set.seed(6735)
folds <- vfold_cv(mtcars, v = 5)

car_rec <-
  recipe(mpg ~ ., data = mtcars) \%>\%
  step_normalize(all_predictors())

# define an svm with parameters to tune
svm_mod <-
  svm_rbf(cost = tune(), rbf_sigma = tune()) \%>\%
  set_engine("kernlab") \%>\%
  set_mode("regression")

# use a space-filling design with 6 points
set.seed(3254)
svm_grid <- tune_grid(svm_mod, car_rec, folds, grid = 6)

show_best(svm_grid, metric = "rmse")

# use bayesian optimization to evaluate at 6 more points
set.seed(8241)
svm_bayes <- tune_bayes(svm_mod, car_rec, folds, initial = svm_grid, iter = 6)

# note that bayesian optimization evaluated parameterizations
# similar to those that previously decreased rmse in svm_grid
show_best(svm_bayes, metric = "rmse")

# specifying `initial` as a numeric rather than previous tuning results
# will result in `tune_bayes` initially evaluating an space-filling
# grid using `tune_grid` with `grid = initial`
set.seed(0239)
svm_init <- tune_bayes(svm_mod, car_rec, folds, initial = 6, iter = 6)

show_best(svm_init, metric = "rmse")
\dontshow{\}) # examplesIf}
}
\seealso{
\code{\link[=control_bayes]{control_bayes()}}, \code{\link[=tune]{tune()}}, \code{\link[=autoplot.tune_results]{autoplot.tune_results()}},
\code{\link[=show_best]{show_best()}}, \code{\link[=select_best]{select_best()}}, \code{\link[=collect_predictions]{collect_predictions()}},
\code{\link[=collect_metrics]{collect_metrics()}}, \code{\link[=prob_improve]{prob_improve()}}, \code{\link[=exp_improve]{exp_improve()}}, \code{\link[=conf_bound]{conf_bound()}},
\code{\link[=fit_resamples]{fit_resamples()}}
}
