# Generated by using Rcpp::compileAttributes() -> do not edit by hand
# Generator token: 10BE3573-1514-4C36-9D1C-5A225CD40393

#' Simulates species trees using constant rate birth-death process
#'
#' @description Forward simulates to a number of tips. This function does so using
#'     the general algorithm of Hartmann et al. 2010. Short for simulate species tree under
#'     birth-death process.
#' @param sbr species birth rate (i.e. speciation rate)
#' @param sdr species death rate (i.e. extinction rate)
#' @param numbsim number of species trees to simulate
#' @param n_tips number of tips to simulate to
#' @param gsa_stop_mult number of tips to simulate the GSA tip to
#' @return List of objects of the tree class (as implemented in APE)
#' @references
#' K. Hartmann, D. Wong, T. Stadler. Sampling trees from evolutionary models.
#'     Syst. Biol., 59(4): 465-476, 2010.
#'
#' T. Stadler. Simulating trees on a fixed number of extant species.
#'     Syst. Biol., 60: 676-684, 2011.
#' @examples
#' mu <- 0.5 # death rate
#' lambda <- 2.0 # birth rate
#' numb_replicates <- 10
#' numb_extant_tips <- 4
#' # simulate trees under the GSA so first simulates a tree with
#' # numb_extant_tips * 100 tips counting each time we have a tree with 10 tips
#' # then randomly picks one of those trees
#'
#' tree_list <- sim_stBD(sbr = lambda,
#'                 sdr = mu,
#'                 numbsim = numb_replicates,
#'                 n_tips = numb_extant_tips)
sim_stBD <- function(sbr, sdr, numbsim, n_tips, gsa_stop_mult = 10L) {
    .Call(`_treeducken_sim_stBD`, sbr, sdr, numbsim, n_tips, gsa_stop_mult)
}

#' Simulates species tree using constant rate birth-death process to a time
#'
#' @description Forward simulates a tree until a provided time is reached.
#' @param sbr species birth rate (i.e. speciation rate)
#' @param sdr species death rate (i.e. extinction rate)
#' @param numbsim number of species trees to simulate
#' @param t time to simulate to
#' @return List of objects of the tree class (as implemented in APE)
#' @references
#' K. Hartmann, D. Wong, T. Stadler. Sampling trees from evolutionary models.
#'     Syst. Biol., 59(4): 465-476, 2010.
#'
#' T. Stadler. Simulating trees on a fixed number of extant species.
#'     Syst. Biol., 60: 676-684, 2011.
#' @examples
#' mu <- 0.5 # death rate
#' lambda <- 2.0 # birth rate
#' numb_replicates <- 10
#' time <- 1
#'
#' tree_list <- sim_stBD_t(sbr = lambda,
#'                 sdr = mu,
#'                 numbsim = numb_replicates,
#'                 t = time)
sim_stBD_t <- function(sbr, sdr, numbsim, t) {
    .Call(`_treeducken_sim_stBD_t`, sbr, sdr, numbsim, t)
}

#' Simulates locus tree using constant rate birth-death-transfer process
#'
#' @description Given a species tree simulates a locus or gene family tree along
#'     the species tree. Short for simulates a locus tree under a birth-death-transfer
#'     process.
#' @param species_tree species tree to simulate along
#' @param gbr gene birth rate
#' @param gdr gene death rate
#' @param lgtr gene transfer rate
#' @param num_loci number of locus trees to simulate
#' @param transfer_type The type of transfer input. Acceptable options: "cladewise" or "random"
#' @return List of objects of the tree class (as implemented in APE)
#' @details Given a species tree will perform a birth-death process coupled with transfer.
#' The simulation runs along the species tree speciating and going extinct in addition to locus tree birth and deaths.
#' Thus with parameters set to 0.0 a tree identical to the species tree is returned (it is relabel however).
#'
#' Transfers are implemented as a birth-death process.
#' One descendant lineage retains its species identity the other gains a new identity.
#' At present, two types of transfers are implemented: "random" an "cladewise".
#' The random transfer mode transfers one randomly chooses a contemporaneous lineage.
#' Cladewise transfers choose lineages based on relatedness with more closely related lineages being more likely.
#' @references
#' Rasmussen MD, Kellis M. Unified modeling of gene duplication, loss, and
#'     coalescence using a locus tree. Genome Res. 2012;22(4):755–765.
#'     doi:10.1101/gr.123901.111
#' @examples
#' # first simulate a species tree
#' mu <- 0.5 # death rate
#' lambda <- 2.0 # birth rate
#' numb_replicates <- 10
#' numb_extant_tips <- 4
#' # simulate trees under the GSA so first simulates a tree with
#' # numb_extant_tips * 100 tips counting each time we have a tree with 10 tips
#' # then randomly picks one of those trees
#'
#' sp_tree <- sim_stBD(sbr = lambda,
#'                 sdr = mu,
#'                 numbsim = numb_replicates,
#'                 n_tips = numb_extant_tips)
#'
#' gene_br <- 1.0
#' gene_dr <- 0.2
#' transfer_rate <- 0.2
#' sim_ltBD(species_tree = sp_tree[[1]],
#'                   gbr = gene_br,
#'                   gdr = gene_dr,
#'                   lgtr = transfer_rate,
#'                   num_loci = 10)
sim_ltBD <- function(species_tree, gbr, gdr, lgtr, num_loci, transfer_type = "random") {
    .Call(`_treeducken_sim_ltBD`, species_tree, gbr, gdr, lgtr, num_loci, transfer_type)
}

#' Simulates a host-symbiont system using a cophylogenetic birth-death process
#'
#' @details Simulates a cophylogenetic system using birth-death processes with
#'     anagenetic processes allowing symbiont to gain or loss associations with
#'     hosts. The host tree is simulated following a constant rate birth-death process
#'     with an additional parameter - the cospeciation rate. This rate works as
#'     the speciation rate with the additional effect that if cospeciation
#'     occurs the symbiont tree also speciates. The symbiont tree is related to
#'     the host tree via an association matrix that describes which lineages
#'     are associated with which. The symbiont tree has an independent
#'     birth-death process with the addition of a host shift speciation rate
#'     that allows for the addition of more associated hosts upon symbiont
#'     speciation. The anagenetic processes are modeled using a poisson process
#'     occurring along the tree. The dispersal to hosts is at present random;
#'     there is no preferential host expansion.
#'
#'     Host expansions are similar to the more commonly found host switching. 
#'     In this model, host-expansion speciation describes events where a symbiont 
#'     speciates and at that time, both descendants retain the ancestral host 
#'     associations. Randomly one of these descendant symbionts then randomly acquires a new 
#'     host. When the option `host_switch_mode = TRUE`, the behavior of this changes to a 
#'     more traditional host switching where one descendant retains the ancestral range and
#'     the other gains a novel host association. 
#' @param hbr host tree birth rate
#' @param hdr host tree death rate
#' @param sbr symbiont tree birth rate
#' @param sdr symbiont tree death rate
#' @param s_disp_r symbiont dispersal rate to new hosts
#' @param s_extp_r symbiont exirpation rate on h
#' @param host_exp_rate host shift speciation rate
#' @param cosp_rate cospeciation rate
#' @param time_to_sim time units to simulate until
#' @param numbsim number of replicates
#' @param host_limit Maximum number of hosts for symbionts (0 implies no limit)
#' @param hs_mode Boolean turning host expansion into host switching (explained above) (default = FALSE)
#' @return A list containing the `host_tree`, the `symbiont_tree`, the
#'     association matrix in the present, with hosts as rows and smybionts as columns, and the history of events that have
#'     occurred.
#' @examples
#'
#' host_mu <- 0.5 # death rate
#' host_lambda <- 2.0 # birth rate
#' numb_replicates <- 10
#' time <- 1.0
#' symb_mu <- 0.2
#' symb_lambda <- 0.4
#' host_shift_rate <- 0.0
#' cosp_rate <- 2.0
#'
#' cophylo_pair <- sim_cophyBD_ana(hbr = host_lambda,
#'                            hdr = host_mu,
#'                            cosp_rate = cosp_rate,
#'                            s_disp_r = 1,
#'                            s_extp_r = 0.4,
#'                            host_exp_rate = host_shift_rate,
#'                            sdr = symb_mu,
#'                            sbr = symb_lambda,
#'                            numbsim = numb_replicates,
#'                            time_to_sim = time)
#'
sim_cophyBD_ana <- function(hbr, hdr, sbr, sdr, s_disp_r, s_extp_r, host_exp_rate, cosp_rate, time_to_sim, numbsim, host_limit = 0L, hs_mode = FALSE) {
    .Call(`_treeducken_sim_cophyBD_ana`, hbr, hdr, sbr, sdr, s_disp_r, s_extp_r, host_exp_rate, cosp_rate, time_to_sim, numbsim, host_limit, hs_mode)
}

#' Simulates a host-symbiont system using a cophylogenetic birth-death process
#'
#' @details Simulates a cophylogenetic system using birth-death processes. The
#'     host tree is simulated following a constant rate birth-death process
#'     with an additional parameter - the cospeciation rate. This rate works as
#'     the speciation rate with the additional effect that if cospeciation
#'     occurs the symbiont tree also speciates. The symbiont tree is related to
#'     the host tree via an association matrix that describes which lineages
#'     are associated with which. The symbiont tree has an independent
#'     birth-death process with the addition of a host shift speciation rate
#'     that allows for the addition of more associated hosts upon symbiont
#'     speciation.
#'
#'     Host expansions are similar to the more commonly found host switching. 
#'     In this model, host-expansion speciation describes events where a symbiont 
#'     speciates and at that time, both descendants retain the ancestral host 
#'     associations. Randomly one of these descendant symbionts then randomly acquires a new 
#'     host. When the option `host_switch_mode = TRUE`, the behavior of this changes to a 
#'     more traditional host switching where one descendant retains the ancestral range and
#'     the other gains a novel host association. 
#' @param hbr host tree birth rate
#' @param hdr host tree death rate
#' @param sbr symbiont tree birth rate
#' @param sdr symbiont tree death rate
#' @param host_exp_rate host shift speciation rate
#' @param cosp_rate cospeciation rate
#' @param time_to_sim time units to simulate until
#' @param numbsim number of replicates
#' @param host_limit Maximum number of hosts for symbionts (0 implies no limit)
#' @param hs_mode Boolean turning host expansion into host switching (explained above) (default = FALSE)
#' @return A list containing the `host_tree`, the `symbiont_tree`, the
#'     association matrix in the present, with hosts as rows and smybionts as columns, and the history of events that have
#'     occurred.
#' @examples
#'
#' host_mu <- 0.5 # death rate
#' host_lambda <- 2.0 # birth rate
#' numb_replicates <- 10
#' time <- 1.0
#' symb_mu <- 0.2
#' symb_lambda <- 0.4
#' host_shift_rate <- 0.0
#' cosp_rate <- 2.0
#'
#' cophylo_pair <- sim_cophyBD(hbr = host_lambda,
#'                            hdr = host_mu,
#'                            cosp_rate = cosp_rate,
#'                            host_exp_rate = host_shift_rate,
#'                            sdr = symb_mu,
#'                            sbr = symb_lambda,
#'                            numbsim = numb_replicates,
#'                            time_to_sim = time)
#'
sim_cophyBD <- function(hbr, hdr, sbr, sdr, host_exp_rate, cosp_rate, time_to_sim, numbsim, host_limit = 0L, hs_mode = FALSE) {
    .Call(`_treeducken_sim_cophyBD`, hbr, hdr, sbr, sdr, host_exp_rate, cosp_rate, time_to_sim, numbsim, host_limit, hs_mode)
}

#' Simulate multispecies coalescent on a species tree
#'
#' @description Simulates the multispecies coalescent on a species tree.
#' @param species_tree input species tree of class "phylo"
#' @param ne Effective population size
#' @param generation_time The number of time units per generation
#' @param num_sampled_individuals number of individuals sampled within each lineage
#' @param num_genes number of genes to simulate within each locus
#' @param mutation_rate The rate of mutation per generation
#' @param rescale Rescale the tree into coalescent units (otherwise assumes it is in those units)
#' @details
#' This a multispecies coalescent simulator with two usage options.
#' The function can rescale the given tree into coalescent units given the `mutation_rate`, `ne`, and the `generation_time`.
#' These result in a tree with coalescent times in units of expected number of mutations per site.
#' The generation_time parameter default is 1 time unit per generation if the units of the tree are in millions of years
#' The mutation_rate parameter is by default set to 1 mutations per site per generation (which is nonsensical).
#' Rescale is set to true by default.
#'
#' If rescale is set to false the tree is assumed to be in coalescent units and `ne` is used as the population
#' genetic parameter theta.
#' @return A list of coalescent trees
#' @seealso sim_ltBD, sim_stBD, sim_stBD_t
#'
#' @examples
#' # first simulate a species tree
#' mu <- 0.5
#' lambda <- 1.0
#' nt <- 6
#' tr <- sim_stBD(sbr = lambda, sdr = mu, numbsim = 1, n_tips = nt)
#' # for a locus tree with 100 genes sampled per locus tree
#' gentrees <- sim_msc(tr[[1]],
#'                     ne = 10000,
#'                     mutation_rate = 1e-9,
#'                     generation_time = 1e-6,
#'                     num_sampled_individuals = 1,
#'                     num_genes = 100)
#'
#' @references
#' Bruce Rannala and Ziheng Yang (2003) Bayes Estimation of Species Divergence Times and Ancestral Population Sizes Using DNA Sequences From Multiple Loci Genetics August 1, 2003 vol. 164 no. 4 1645-1656
#' Mallo D, de Oliveira Martins L, Posada D (2015) SimPhy: Phylogenomic Simulation of Gene, Locus and Species Trees. Syst. Biol. doi: http://dx.doi.org/10.1093/sysbio/syv082
sim_msc <- function(species_tree, ne, num_sampled_individuals, num_genes, rescale = TRUE, mutation_rate = 1L, generation_time = 1L) {
    .Call(`_treeducken_sim_msc`, species_tree, ne, num_sampled_individuals, num_genes, rescale, mutation_rate, generation_time)
}

