## ----setup, echo=FALSE, message=FALSE, results="hide"-------------------------
knitr::opts_chunk$set(size = "small", prompt = TRUE, comment = NA,
                      out.width=".9\\linewidth")
knitr::knit_hooks$set(
  document = function(x) {sub('\\usepackage[]{color}', '\\usepackage{xcolor}',
                              x, fixed = TRUE)}
)
oldpar <- par(no.readonly = TRUE) ## NOTE: for setting back at the end
oldopt <- options()
options(prompt = "R> ", continue = "+  ")
options(width = 80, digits = 3)

## Dependencies
library("tramME")
library("ggplot2")
library("lme4")
library("survival")

## Functions
mycolors <- function(nr, type = "line") {
  cols <- list()
  cols[[1]] <- c(red = 0, green = 84, blue = 150)
  cols[[2]] <- c(red = 202, green = 108, blue = 24)
  out <- as.list(cols[[nr]])
  out$alpha <- switch(type, line = 255L, fill = 140L)
  out$maxColorValue <- 255
  do.call("rgb", out)
}

plot_cicomp <- function(est1, est2, labels, mnames, vline = 0, ylab = "",
                        col1 = mycolors(1), col2 = mycolors(2), xlas = 0,
                        legend_pos = "topleft") {
  stopifnot(all(dim(est1) == dim(est2)))
  nr <- nrow(est1)
  xs <- matrix(rep(c(-0.2, 0.2), nr) + rep(1:nr, each = 2), nrow = 2)
  plot(0, 0, type = "n", xlab = ylab, ylab = "",
       xlim = range(est1[, -1], est2[, -1]) + c(-0.2, 0.2),
       ylim = rev(range(xs) + c(-0.3, 0.3)), yaxt = "n", yaxs = "i")
  abline(h = seq(1.5, nr-0.5, by = 1), col = "grey", lty = 2)
  abline(v = vline)
  points(c(rbind(est1[, 1], est2[, 1])), c(xs),
         col = rep(c(col1, col2), nr), pch = 20)
  segments(x0 = est1[, 2], y0 = xs[1, ], x1 = est1[, 3], col = col1, lwd = 2)
  segments(x0 = est2[, 2], y0 = xs[2, ], x1 = est2[, 3], col = col2, lwd = 2)
  axis(2, at = 1:nr, labels = labels, tick = FALSE, las = xlas)
  if (!is.null(legend_pos)) {
    legend(legend_pos, mnames, col = c(col1, col2),
           pch = 20, lwd = 2, cex = 0.95, bg = "white")
  }
}

## ggplot theme
ggth <- theme(panel.grid.major = element_blank(), panel.grid.minor = element_blank(),
              panel.background = element_blank(),
              strip.background = element_rect(colour = "black", fill = "white"),
              axis.line = element_line(colour = "black"),
              legend.title = element_blank(),
              legend.key = element_rect(fill = "transparent", colour = "transparent"))

## ----sleeptudy-data, echo=FALSE, fig.width=7, fig.height=4--------------------
data("sleepstudy", package = "lme4")
dat <- within(sleepstudy, Subject <- paste0("Subject = ", Subject))
ggplot(dat, aes(x = Days, y = Reaction)) +
  facet_wrap(~ Subject, nrow = 3) +
  geom_point(col = "black", size = 0.75) +
  geom_line(col = "black") +
  scale_x_continuous("Days of sleep deprivation",
                     breaks = c(0, 2, 4, 6, 8)) +
  ylab("Average reaction time (ms)") +
  ggth

## ----message=FALSE------------------------------------------------------------
library("tramME")
sleep_lmME <- LmME(Reaction ~ Days + (Days | Subject), data = sleepstudy)
logLik(sleep_lmME)

## ----message=FALSE------------------------------------------------------------
library("lme4")
sleep_lmer <- lmer(Reaction ~ Days + (Days | Subject), data = sleepstudy,
                   REML = FALSE)
logLik(sleep_lmer)

## -----------------------------------------------------------------------------
cbind(coef = coef(sleep_lmME, as.lm = TRUE),
      se = sqrt(diag(vcov(sleep_lmME, as.lm = TRUE, pargroup = "fixef"))))

## -----------------------------------------------------------------------------
summary(sleep_lmer)$coefficients

## -----------------------------------------------------------------------------
VarCorr(sleep_lmME, as.lm = TRUE) ## random effects
sigma(sleep_lmME) ## residual SD
VarCorr(sleep_lmer)

## -----------------------------------------------------------------------------
library("survival")
ub <- ceiling(sleepstudy$Reaction / 50) * 50
lb <- floor(sleepstudy$Reaction / 50) * 50
lb[ub == 200] <- 0
sleepstudy$Reaction_ic <- Surv(lb, ub, type = "interval2")
head(sleepstudy$Reaction_ic)

## -----------------------------------------------------------------------------
sleep_lmME2 <- LmME(Reaction_ic ~ Days + (Days | Subject), data = sleepstudy)
logLik(sleep_lmME2)

## -----------------------------------------------------------------------------
cbind(coef = coef(sleep_lmME2, as.lm = TRUE),
      se = sqrt(diag(vcov(sleep_lmME2, as.lm = TRUE, pargroup = "fixef"))))
sigma(sleep_lmME2)
VarCorr(sleep_lmME2, as.lm = TRUE)

## -----------------------------------------------------------------------------
sleep_lmME3 <- LmME(Reaction_ic ~ Days + (Days || Subject), data = sleepstudy)
logLik(sleep_lmME3)

## -----------------------------------------------------------------------------
anova(sleep_lmME2, sleep_lmME3)

## -----------------------------------------------------------------------------
sleep_bc <- BoxCoxME(Reaction ~ Days + (Days | Subject), data = sleepstudy)
logLik(sleep_bc)

## ----sleepstudy-trafo, echo=FALSE,fig.width=4.5, fig.height=4, out.width="0.7\\linewidth"----
par(mfrow = c(1, 1), cex = 0.8)
tr_bc <- trafo(sleep_bc, type = "trafo", confidence = "none", K = 100)
plot(tr_bc, lty = 1, xlab = "Average reaction time (ms)", ylab = expression(h(y)),
     lwd = 2)
tr_lm <- trafo(sleep_lmME, type = "trafo", confidence = "none", K = 100)
plot(tr_lm, lty = 2, lwd = 2, add = TRUE)
rug(sleepstudy$Reaction, col = rgb(.1, .1, .1, .1))
legend("topleft", c("BoxCoxME", "LmME"), lty = c(1, 2), lwd = 2,
       bty = "n", cex = 0.8)

## ----eval=FALSE---------------------------------------------------------------
#  ndraws <- 1000
#  nd <- expand.grid(
#    Reaction = seq(min(sleepstudy$Reaction), max(sleepstudy$Reaction), length.out = 100),
#    Days = 0:9,
#    Subject = 1)
#  re <- simulate(sleep_bc, newdata = nd, nsim = ndraws, what = "ranef", seed = 100)
#  cp <- parallel::mclapply(re, function(x) {
#    predict(sleep_bc, newdata = nd, ranef = x, type = "distribution")
#  }, mc.cores = 8)
#  cp <- array(unlist(cp), dim = c(100, 10, ndraws))
#  mp_bc <- apply(cp, c(1, 2), mean)

## ----margdist-bc, echo=FALSE--------------------------------------------------
## Saved for faster execution
nd <- expand.grid(
  Reaction = seq(min(sleepstudy$Reaction), max(sleepstudy$Reaction), length.out = 100),
  Days = 0:9,
  Subject = 1)
mp_bc <-
structure(c(0.00507337214818551, 0.00748195337173772, 0.0108389920611311,
0.0154350012192178, 0.0216104600646775, 0.0297462033753768, 0.0402465865545393,
0.0535169349593186, 0.0699382024923084, 0.0898422503812447, 0.113490403379527,
0.141056237506205, 0.172611745120405, 0.208115115757537, 0.247398974287144,
0.289991822437949, 0.333661370643938, 0.377256310387937, 0.420143595182202,
0.461796566690828, 0.501799910466699, 0.539847226873397, 0.575732715566085,
0.609338901958667, 0.640622335337018, 0.669598888767934, 0.696329851268785,
0.720909550839167, 0.743454860578696, 0.764096652250291, 0.782973074298604,
0.800224429165699, 0.815989385717214, 0.830402264837359, 0.843591161695045,
0.855676703755764, 0.866771280990579, 0.876978619367263, 0.886393598676183,
0.895102240714991, 0.903181814309776, 0.910701020347085, 0.917720233666784,
0.924291789875043, 0.930460314200619, 0.936263096526565, 0.941730521584695,
0.946886565779734, 0.951749371956347, 0.956331910450944, 0.960642728990594,
0.964686785734663, 0.968466349717005, 0.971981942262986, 0.975233283067737,
0.978220197076674, 0.980943434455337, 0.983405356607831, 0.985610446504959,
0.987565610863179, 0.989280253752025, 0.990766114485292, 0.99203687551887,
0.993107556593936, 0.993995036852786, 0.994763931733651, 0.99544892654972,
0.99605731196674, 0.996595936509909, 0.997071220980373, 0.99748917228903,
0.997855396723406, 0.998175112751548, 0.99845316352777, 0.998694029298885,
0.998901839918073, 0.999080387660142, 0.999233140500887, 0.99936325597941,
0.999473595710708, 0.999566740561406, 0.999645006448549, 0.999710460673532,
0.999764938663256, 0.999810060960294, 0.999847250284153, 0.999877748476613,
0.9999026331451, 0.999922833827792, 0.99993914752112, 0.999952253432804,
0.999962726849571, 0.999971052036568, 0.99997763411361, 0.999982809880321,
0.999986857587043, 0.999990005670147, 0.999992440488747, 0.999994313114474,
0.999995745236944, 0.00227055154169292, 0.00348297063003836,
0.00523671079736867, 0.00772935969987355, 0.0112103484026782,
0.0159829585154181, 0.0224002450039267, 0.0308538019125036, 0.0417558368990524,
0.0555165997564748, 0.0725201544136056, 0.0931013245840025, 0.117525414393337,
0.14597060437619, 0.178511663603017, 0.214957508074452, 0.253510669763063,
0.29315466502159, 0.333263972548163, 0.373266755549299, 0.412660343219481,
0.451021307865512, 0.488009899223386, 0.523369220079851, 0.556920086587466,
0.588552832238536, 0.618217362218856, 0.645912608839078, 0.671676271354782,
0.695575427755496, 0.717698339044043, 0.738147557280317, 0.757034304351957,
0.774474002402597, 0.790582796083948, 0.805474897197193, 0.819260591639127,
0.832044767720976, 0.843925847819646, 0.854995028412202, 0.865335755078472,
0.875023378374262, 0.884124953477791, 0.892699161279528, 0.900796341161271,
0.908458635968295, 0.915720257286527, 0.922607883580898, 0.92914120442178,
0.935333620367559, 0.941193099790207, 0.946723181252039, 0.951924093960868,
0.9567939512821, 0.961329956117864, 0.965529545598177, 0.969391399258955,
0.972916241861603, 0.976107389217222, 0.978971009972052, 0.981516102831221,
0.983754210346203, 0.985698901070651, 0.98736504813724, 0.988770022343159,
0.990008878526737, 0.991133227446038, 0.992151747850834, 0.993072529619154,
0.993903117933382, 0.994650557029371, 0.995321432071435, 0.995921908013395,
0.996457764627785, 0.996934427204177, 0.997356992717648, 0.997730251536223,
0.998058704961711, 0.998346579074873, 0.998597835480201, 0.998816179617865,
0.999005067333754, 0.99916771037847, 0.999307081450135, 0.999425919312248,
0.999526734415724, 0.999611815342259, 0.999683236272498, 0.999742865574073,
0.999792375506955, 0.999833252960932, 0.99986681107467, 0.999894201539024,
0.999916427358552, 0.999934355833413, 0.999948731526775, 0.999960188997953,
0.999969265105812, 0.999976410717611, 0.999982001692637, 0.00107290664857875,
0.00171953290110646, 0.00268808173111371, 0.0041069272860542,
0.00614240340934729, 0.00900418859469746, 0.0129485186225571,
0.0182779311523613, 0.0253365751892532, 0.0345007405739366, 0.0461649690917698,
0.060724629295636, 0.0785559796446323, 0.0999945122064154, 0.125311986679867,
0.154573300309126, 0.186480908147388, 0.22024604345813, 0.255344822201904,
0.291260176420223, 0.327502342383367, 0.363624937874709, 0.399236324118473,
0.434006268552361, 0.467668292351108, 0.50001838832556, 0.530910969669648,
0.560252948765978, 0.587996773342066, 0.614133105255532, 0.638683655601946,
0.661694520514032, 0.683230215140635, 0.703368488250056, 0.722195918008888,
0.739804237107041, 0.756287306626469, 0.771738646606577, 0.786249431591582,
0.799906867195076, 0.812792875762385, 0.824983033478078, 0.836545716446762,
0.847541428539402, 0.858022298536461, 0.868031747723879, 0.877604340891106,
0.886765842707565, 0.895533506541754, 0.903916622587371, 0.911917345324767,
0.9195318058331, 0.926751491991892, 0.933564850173358, 0.93995902843061,
0.945921648296644, 0.951442466791724, 0.956514779509282, 0.961136425954465,
0.965310292209177, 0.969044260075944, 0.972350615740381, 0.975244988734612,
0.977744925936585, 0.979871407386077, 0.981764261055758, 0.983500482159961,
0.985092366644044, 0.986551272875908, 0.987887645245048, 0.989111052031678,
0.990230234933178, 0.991253167477982, 0.992187119519434, 0.993038725078151,
0.993814050979635, 0.994518663998661, 0.995157694554484, 0.995735895380413,
0.996257693996076, 0.99672723821992, 0.997148434353452, 0.997524978030739,
0.997860378042881, 0.998157973707794, 0.998420946554452, 0.998652327225881,
0.998854998577797, 0.999031695964741, 0.999185005669772, 0.999317362356217,
0.99943104631069, 0.999528181115849, 0.999610732249263, 0.999680506960542,
0.999739155640478, 0.999788174769737, 0.999828911425432, 0.999862569234536,
0.99989021559512, 0.000547385324706129, 0.000918436515290171,
0.00149536447073125, 0.00236701247496648, 0.00364909586004772,
0.00548819944747592, 0.00806510676018091, 0.0115970133545241,
0.0163381583165019, 0.0225784436498318, 0.0306396927426217, 0.0408692781037559,
0.0536308475876134, 0.0692917456041395, 0.088206519007313, 0.110603549581545,
0.135647548197205, 0.16282314060508, 0.191777060676106, 0.222123628932546,
0.253464096934917, 0.285404881567167, 0.317573318192416, 0.349629982132489,
0.381277171584021, 0.412263648029469, 0.442386096087514, 0.471487966192346,
0.499456420215717, 0.526218053599028, 0.551733961865529, 0.57599459082071,
0.599014683195854, 0.620828524059062, 0.641485598893936, 0.661046712356743,
0.679580569973388, 0.697160795845044, 0.713863343093819, 0.729764247029745,
0.744937671231311, 0.759454201920645, 0.773379354737128, 0.786772269217021,
0.79968457915243, 0.812159460793132, 0.824230874773887, 0.835923030721368,
0.847250114462434, 0.858216325024357, 0.8688162702633, 0.879035763843374,
0.888853050344077, 0.898240457978085, 0.907166439489738, 0.915597913127913,
0.923502761920822, 0.930852298905607, 0.9376234693176, 0.943800549666561,
0.949376127141598, 0.95435120300079, 0.958734352036755, 0.962539966675337,
0.965790599486066, 0.968693997102407, 0.97136512863168, 0.973821097925986,
0.976078310932872, 0.978152379944321, 0.98005804761009, 0.981809131303286,
0.983418487797873, 0.984897997618812, 0.98625856788093, 0.987510151962792,
0.988661783982111, 0.989721625759165, 0.990697023778301, 0.991594573583804,
0.992420189069329, 0.993179174230367, 0.993876295134238, 0.994515850107588,
0.995101736431806, 0.995637512156336, 0.996126451973355, 0.996571596430432,
0.99697579407789, 0.997341736443881, 0.99767198599403, 0.997968997457417,
0.998235133082524, 0.998472672523646, 0.99868381815015, 0.9988706966198,
0.999035357566596, 0.999179770227833, 0.999305818780055, 0.999415297075337,
0.000319039129290769, 0.000555807499087579, 0.000935161190511173,
0.001522561846493, 0.0024035504257071, 0.00368632426066466, 0.00550383786546207,
0.00801539398491258, 0.011407738648519, 0.0158956594718574, 0.0217220007012212,
0.0291568592033344, 0.0384955229293722, 0.0500544607511198, 0.0641643856555498,
0.0810889408903429, 0.100297559508401, 0.121484203375806, 0.144452922392502,
0.168966163962331, 0.194755653470937, 0.22153481036367, 0.24901153195328,
0.276900172434861, 0.304931786047875, 0.332862068430151, 0.360476802979466,
0.387594922532474, 0.414069498280946, 0.439787068885219, 0.464665743475043,
0.488652479082514, 0.511719870506758, 0.533862717441064, 0.555094562275867,
0.575444329226161, 0.594953144245604, 0.613671375992051, 0.631655909947798,
0.648967649145592, 0.665669224182309, 0.681822890833962, 0.697488594369251,
0.712722184574521, 0.727573773704258, 0.742086240301265, 0.756293894325382,
0.770221332417765, 0.783882525334959, 0.797280191254602, 0.81040551708751,
0.823238293094277, 0.835747521698055, 0.847892547031297, 0.85962472534538,
0.870889616659607, 0.881629625225249, 0.891786953321977, 0.901306665707825,
0.910139600463219, 0.9182448185342, 0.925591271668058, 0.932158394945668,
0.937935394055276, 0.942926660934698, 0.947429218648329, 0.951608093017989,
0.955480060427795, 0.9590621153224, 0.962371326824865, 0.965424697658872,
0.968239028093578, 0.970830787536717, 0.973215996194645, 0.975410118910526,
0.977427972903467, 0.979283650685317, 0.980990458953263, 0.982560873770512,
0.984006511877747, 0.985338117544984, 0.98656556399335, 0.987697868100626,
0.988743216860006, 0.989709003890668, 0.990601874199701, 0.991427775362618,
0.992192013316927, 0.992899311040919, 0.993553868508652, 0.994159422462249,
0.994719304715107, 0.995236497886127, 0.99571368765834, 0.996153310849295,
0.996557598770105, 0.996928615531073, 0.997268291121038, 0.997578449242355,
0.997860830021762, 0.000227315054826847, 0.000401659291905247,
0.000684350656322734, 0.00112645493481311, 0.00179476683974265,
0.00277346514831529, 0.00416516213663913, 0.00609142767270037,
0.0086930619564405, 0.012130509093734, 0.0165847487649818, 0.0222587182941001,
0.029378840334097, 0.0381957111705148, 0.048982649245593, 0.0619767059654308,
0.0768161728694607, 0.0933124724937194, 0.111361743411551, 0.130825916289831,
0.151538421442512, 0.173311261784063, 0.195942813197779, 0.219225748172467,
0.242954552660887, 0.266932210996142, 0.290975761336742, 0.314920560219498,
0.338623222136108, 0.361963304270949, 0.384843879798569, 0.407191184456794,
0.428953534594126, 0.450099707301039, 0.470616951887805, 0.490508773373366,
0.509792597871518, 0.528497400336261, 0.546661349249655, 0.564329501664907,
0.581551566002093, 0.598379739131845, 0.614866618316141, 0.631063187117477,
0.647016876984758, 0.662769712371655, 0.67835655636086, 0.69380348514858,
0.709126332485096, 0.724329458100673, 0.739404805779224, 0.754331325231868,
0.769074835098491, 0.783588399861969, 0.797813278739419, 0.811680477507402,
0.825112893201367, 0.838027986470781, 0.850340848852906, 0.861967456857891,
0.872827829131422, 0.882848737244993, 0.891965575591109, 0.900122978860217,
0.907284744519346, 0.913843955815689, 0.920022204411069, 0.925830338826042,
0.931280046087704, 0.936383765912775, 0.941154598243973, 0.945606204949429,
0.949752706755373, 0.953608576704893, 0.957188531609678, 0.960507423079784,
0.963580129774553, 0.966421452515337, 0.969046013840406, 0.971468163470078,
0.97370189099371, 0.975760746899177, 0.977657772850237, 0.979405441887863,
0.981015608997825, 0.982499472256929, 0.983867544551137, 0.985129635655594,
0.986294844282989, 0.987371559544897, 0.988367471131993, 0.989289587403604,
0.990144260484907, 0.990937217400827, 0.991673596228975, 0.99235798622938,
0.992994470905867, 0.993586672972255, 0.994137800235222, 0.99465069146357,
0.000204601496266591, 0.000354393573096201, 0.000594375836201332,
0.000966545330788949, 0.00152585668251166, 0.00234132290122036,
0.00349625156028667, 0.00508768126768495, 0.00722545743930086,
0.0100316998942108, 0.0136414582922012, 0.0182049774341912, 0.0238912408191306,
0.030891597485258, 0.0394217112099143, 0.0496774675065418, 0.0613919742165509,
0.0744405048742936, 0.0887668235682859, 0.104288124747828, 0.120898838259118,
0.138475480845681, 0.156881846900582, 0.175974121638235, 0.195605706849829,
0.215631651266667, 0.235912609481886, 0.256318258538456, 0.276730108199646,
0.297043660187041, 0.317169901607228, 0.337036151452831, 0.356586309373604,
0.375780578136337, 0.394594743463849, 0.413019097713882, 0.431057088991763,
0.448723767229951, 0.466044085945425, 0.483051104889245, 0.499784126243968,
0.516286786543621, 0.532605118825264, 0.548785595095617, 0.564873158203515,
0.580909254661374, 0.596929885707707, 0.612963702622287, 0.629030183450673,
0.645137941040845, 0.661283225475572, 0.677448696045944, 0.693602546896558,
0.70969807405178, 0.725673767092563, 0.741453993562215, 0.756950315646975,
0.772063434757954, 0.786685699416288, 0.800704036266323, 0.814003076680817,
0.826468158811799, 0.837987795932773, 0.848455125414742, 0.857782170636027,
0.866448146084617, 0.874729228492404, 0.88262890625598, 0.890151694366242,
0.897303095051277, 0.904089552655349, 0.910518402785076, 0.916597815934041,
0.922336735961295, 0.927744813940399, 0.932832338009048, 0.937610159932669,
0.9420896191484, 0.946282465080595, 0.950200778518679, 0.953856892827484,
0.957263315724158, 0.960432652309783, 0.963377529992918, 0.966110525890908,
0.968644097246366, 0.970990515353066, 0.973161803448693, 0.975169679001079,
0.977025500788408, 0.978740221149783, 0.980324343757334, 0.981787887231025,
0.983140354878884, 0.984390710795263, 0.985547362485571, 0.986618150106496,
0.987610342316223, 0.988530638621427, 0.989385177990069, 0.000223607860154018,
0.000370394285711246, 0.000598199590753242, 0.000943083757601369,
0.00145228154673816, 0.0021850012917356, 0.00321210212204412,
0.00461465087275888, 0.00648187658478898, 0.00890951410979191,
0.0119996737455978, 0.0158629876501742, 0.0206229001496803, 0.0264209099002245,
0.033420827345726, 0.0417754317124787, 0.0512705831140687, 0.0618177250113083,
0.0733874773760698, 0.0859303701900581, 0.0993789648411676, 0.113651164276831,
0.128653914277316, 0.144286837005229, 0.160445611131317, 0.177025062415499,
0.193921976213373, 0.211037635329482, 0.22828006280754, 0.245565932604406,
0.262822109285324, 0.279986789041024, 0.297010232705827, 0.313855101132032,
0.330496419958571, 0.346921212251265, 0.363127843162086, 0.379125121283092,
0.394931197988374, 0.410572300187122, 0.426081324910701, 0.441496317194,
0.456858846725956, 0.472212294461141, 0.487600058344834, 0.503063687893402,
0.518640960828403, 0.534363921361574, 0.550256908936631, 0.566334617847949,
0.582600241422189, 0.599043768195224, 0.615640510109761, 0.632349952103852,
0.649115016055344, 0.665861827097223, 0.682500053908362, 0.698923863921897,
0.715013487162106, 0.730637317275834, 0.745654395539187, 0.759917025728894,
0.773273159822049, 0.785568083481023, 0.796662029654409, 0.807096354808749,
0.817190690848229, 0.826941925481266, 0.836347891561702, 0.845407358898842,
0.854120022748866, 0.862486488677329, 0.870508253590541, 0.878187682844648,
0.885527983446831, 0.89253317345639, 0.899208047769819, 0.905558140530729,
0.911589684442222, 0.917309567277639, 0.92272528588846, 0.927844897999831,
0.932676972069222, 0.937230535466793, 0.941515021221398, 0.945540213566965,
0.949316192522784, 0.952853277748887, 0.956161971934446, 0.959252904001778,
0.962136772439395, 0.964824289112273, 0.967326123933373, 0.969652850814791,
0.971814895347124, 0.973822484679225, 0.975685600085533, 0.977413932712712,
0.979016842990044, 0.980503324167733, 0.000274060682519859, 0.000431310596328588,
0.000665647254332248, 0.00100918636271191, 0.00150398519716681,
0.00220259750823464, 0.00316734622872324, 0.00446830413910582,
0.0061805307844138, 0.00838161089976541, 0.0111507042555709,
0.0145699806645372, 0.0187285256657652, 0.0237278377255109, 0.0296872867918861,
0.0367186725247939, 0.0446345787269989, 0.0533659940100487, 0.0628992092414103,
0.0732066536492028, 0.084246961590455, 0.0959665713386132, 0.108302138885576,
0.121183225828725, 0.134534943760567, 0.148280413666628, 0.162343001530923,
0.176648332484112, 0.191126089943462, 0.205711595671454, 0.22034715600706,
0.234983155180381, 0.249578879426419, 0.264103063442278, 0.278534160613503,
0.292860347790071, 0.307079282541122, 0.32119763501762, 0.335230417804256,
0.349200135927697, 0.363135776237068, 0.377071651492181, 0.391046110470576,
0.405100121950435, 0.419275738172545, 0.433614442874552, 0.448155390678638,
0.4629335488434, 0.477977759353021, 0.493308748981484, 0.508937126998449,
0.524861423836633, 0.541066238097675, 0.557520571990229, 0.574176444417666,
0.59096787378466, 0.60781031621859, 0.624600626280162, 0.641217573536165,
0.65752289727476, 0.673362811805219, 0.688569786338337, 0.702964318537904,
0.716356303800426, 0.728564509641104, 0.740161986598567, 0.75149514496571,
0.762556627153711, 0.773339805889676, 0.783838786467082, 0.794048407250679,
0.803964238243324, 0.813582577605302, 0.822900446113696, 0.831915579647924,
0.840626419875969, 0.849032103382988, 0.857132449521818, 0.864927947268311,
0.872419741332245, 0.879609617709622, 0.886499988770701, 0.893093877869324,
0.899394903344058, 0.9054072616719, 0.91113570944214, 0.916585543750686,
0.921762580580861, 0.926673130739099, 0.931323972953667, 0.935722323818753,
0.939875804369521, 0.943792403198425, 0.947480436160424, 0.950948502855432,
0.954205440211765, 0.957260273616766, 0.960122166144459, 0.962800366510856,
0.965304156443602, 0.000353046417379221, 0.000530020165364884,
0.000783375752619142, 0.00114269490175009, 0.00164656297210864,
0.00234281240955566, 0.00328742299272243, 0.0045421330897538,
0.00617133599002771, 0.00823925235724772, 0.0108084798094694,
0.0139407321531629, 0.0176999664907544, 0.0221573600838993, 0.027396982828632,
0.0334956904122277, 0.040278538520544, 0.0476858741599018, 0.0557121554515604,
0.0643437824804351, 0.0735572970630591, 0.0833192478258806, 0.0935872807723591,
0.104311966259044, 0.115438950979464, 0.126911146094784, 0.138670779814478,
0.15066123177584, 0.162828623148739, 0.175123165794555, 0.187500284644032,
0.199921527958096, 0.212355276498593, 0.224777258681263, 0.237170876289599,
0.249527344644042, 0.261845651788864, 0.274132342521169, 0.286401134223473,
0.29867237197478, 0.310972330087066, 0.323332366091417, 0.3357879315384,
0.348377442186934, 0.361141008732737, 0.374119028712335, 0.387350641151689,
0.400872048400632, 0.414714714786854, 0.42890345943122, 0.443454470708097,
0.458373281958796, 0.473652761275375, 0.489271181083755, 0.50519044401866,
0.521354547956215, 0.537688372539944, 0.554096859464129, 0.57046463655734,
0.586656098949133, 0.602515907403413, 0.617869793250926, 0.632525471634349,
0.646273362234968, 0.658906473952209, 0.671001524909634, 0.682914491748928,
0.694636408414933, 0.706158821698448, 0.717473791629695, 0.728573890973489,
0.739452203751758, 0.750102322754125, 0.760518346053006, 0.770694872610423,
0.780626997142655, 0.790310304488028, 0.799740863794163, 0.808915222895296,
0.817830403280162, 0.826483896049874, 0.834873659229071, 0.842998116720793,
0.85085615908768, 0.85844714620436, 0.865770911666197, 0.872827768669007,
0.879618516905329, 0.886144449868867, 0.892407361832724, 0.898409553680088,
0.904153836726722, 0.909643533687348, 0.914882476003562, 0.919874996865221,
0.924625919412679, 0.929140539792722, 0.93342460494341, 0.937484285188245,
0.941326141914427), .Dim = c(100L, 10L))

## ----sleepstudy-margdist,echo=FALSE,message=FALSE,fig.width=7,fig.height=4,cache=TRUE----
## --- Lm
## re <- simulate(sleep_lmME, newdata = nd, nsim = ndraws, what = "ranef", seed = 100)
## cp <- parallel::mclapply(re, function(x) {
##   predict(sleep_lmME, newdata = nd, ranef = x, type = "distribution")
## }, mc.cores = 8)
## cp <- array(unlist(cp), dim = c(100, 10, ndraws))
## mp_lmME <- apply(cp, c(1, 2), mean)

## Saved for faster execution
mp_lmME <-
structure(c(0.0478858079664213, 0.0565943025983427, 0.0664902339037705,
0.0776589376654088, 0.0901786676639275, 0.104118163899217, 0.119534212868745,
0.136469280170928, 0.154949300485965, 0.174981711479834, 0.196553816033614,
0.219631551221881, 0.244158732623244, 0.270056829006745, 0.297225305568471,
0.325542554272193, 0.354867408257527, 0.385041214679317, 0.415890417830286,
0.447229583150475, 0.478864773919519, 0.510597177157136, 0.542226864450771,
0.573556567766864, 0.60439535015356, 0.634562056629916, 0.663888441146892,
0.692221880634779, 0.719427605879579, 0.74539040015767, 0.770015738957683,
0.793230366476574, 0.814982325722207, 0.835240477978103, 0.853993563296726,
0.871248866047498, 0.887030558077364, 0.901377796716824, 0.914342655872917,
0.925987966160365, 0.936385134929931, 0.945612009737255, 0.953750839869843,
0.960886380623727, 0.967104174667186, 0.972489034557371, 0.977123740718369,
0.98108796028852, 0.984457384450174, 0.987303075324396, 0.989691008319366,
0.991681791962081, 0.993330544653484, 0.994686906354482, 0.995795162787704,
0.996694460161874, 0.997419089520102, 0.997998821412577, 0.998459273542474,
0.998822296193471, 0.999106362500473, 0.999326952875619, 0.999496925073063,
0.999626863410784, 0.999725402524083, 0.99979952267582, 0.999854815076726,
0.999895716869146, 0.999925716401289, 0.999947530175658, 0.999963253409451,
0.99997448651582, 0.999982440025574, 0.999988020544069, 0.999991900303342,
0.999994572750238, 0.999996396431397, 0.999997629216967, 0.999998454665596,
0.999999002088857, 0.999999361635813, 0.999999595496512, 0.999999746122443,
0.999999842185449, 0.999999902845246, 0.999999940768866, 0.999999964241379,
0.99999997862368, 0.99999998734727, 0.999999992584965, 0.999999995697715,
0.999999997528718, 0.999999998594719, 0.99999999920895, 0.999999999559216,
0.999999999756885, 0.999999999867277, 0.999999999928285, 0.999999999961648,
0.999999999979701, 0.026741920031586, 0.0320945706469255, 0.0382939259474623,
0.0454269001394536, 0.0535805746628029, 0.0628404866974539, 0.0732887300287162,
0.0850019054099541, 0.0980489676437497, 0.112489025881344, 0.12836916161259,
0.145722334980164, 0.164565453916477, 0.184897681728933, 0.206699056785083,
0.229929492622949, 0.254528218032747, 0.280413704508688, 0.307484113248927,
0.335618276106411, 0.364677205298159, 0.39450610618687, 0.424936847113318,
0.455790821201118, 0.486882118376131, 0.518020912529717, 0.549016959602661,
0.579683097907635, 0.609838642469877, 0.639312570456352, 0.667946404488209,
0.69559671413633, 0.722137172338139, 0.747460121881845, 0.771477626480366,
0.794122000333758, 0.815345828579316, 0.835121507907055, 0.853440351280786,
0.870311312714603, 0.885759397125169, 0.899823826264215, 0.912556034611073,
0.924017568962438, 0.934277962491345, 0.943412648552161, 0.951500971866553,
0.958624345400412, 0.964864590761455, 0.97030248888036, 0.975016556664142,
0.979082054783395, 0.982570222267936, 0.985547725535873, 0.988076303145299,
0.990212583075271, 0.99200804671431, 0.993509112833989, 0.994757315421444,
0.995789551033934, 0.996638373979228, 0.99733232077356, 0.997896248667925,
0.998351676299099, 0.998717117514905, 0.999008402015464, 0.999238978581845,
0.999420198325998, 0.999561576629265, 0.999671033306657, 0.999755111118407,
0.999819173127342, 0.999867579639473, 0.999903845621024, 0.999930779596564,
0.999950605123491, 0.999965066018285, 0.999975516581283, 0.999982998125126,
0.999988303151529, 0.999992028535946, 0.999994619066574, 0.999996402642222,
0.999997618364878, 0.999998438671733, 0.999998986543742, 0.999999348710064,
0.999999585646211, 0.999999739044207, 0.99999983731996, 0.999999899619788,
0.99999993869655, 0.999999962947059, 0.999999977836373, 0.999999986880417,
0.999999992315029, 0.999999995545588, 0.999999997445248, 0.99999999855022,
0.999999999185978, 0.0167635768353816, 0.020216968839925, 0.0242469275160786,
0.0289219541372177, 0.0343135843926038, 0.0404955890375045, 0.0475429970848638,
0.0555309415414434, 0.0645333341017522, 0.0746213825898809, 0.085861973085496,
0.0983159472831431, 0.112036314304428, 0.127066444405355, 0.143438299218899,
0.161170758734246, 0.180268108542632, 0.200718751451326, 0.222494204990099,
0.245548440403036, 0.269817609456699, 0.295220193079594, 0.321657591005615,
0.349015154978447, 0.377163650609452, 0.405961115692214, 0.435255066698379,
0.464884991269353, 0.494685053572798, 0.524486931979331, 0.554122704930564,
0.583427701128686, 0.612243234027881, 0.640419147574154, 0.667816109588979,
0.694307600405323, 0.719781556614536, 0.744141642398709, 0.767308133341223,
0.789218409398383, 0.809827064587486, 0.829105650735363, 0.847042081264738,
0.863639728456554, 0.878916253913788, 0.892902217049495, 0.905639510277094,
0.91717967210426, 0.927582130407106, 0.9369124276716, 0.945240477848892,
0.952638900660693, 0.95918147378372, 0.964941736523395, 0.969991770654249,
0.974401175454599, 0.978236245063255, 0.981559347633667, 0.984428497838386,
0.986897107506335, 0.98901389388091, 0.990822921366709, 0.992363750746799,
0.993671669625547, 0.994777979090117, 0.995710314002045, 0.996492977588155,
0.997147274738223, 0.997691832291867, 0.998142898310863, 0.9985146156523,
0.998819267925612, 0.999067498052389, 0.99926850114233, 0.999430194299246,
0.999559366366366, 0.999661810622539, 0.999742443169869, 0.999805409322287,
0.999854179810923, 0.999891638141491, 0.999920160022682, 0.999941685461568,
0.999957783901276, 0.999969712651654, 0.999978468819645, 0.999984834961268,
0.999989418729571, 0.999992686862129, 0.999994993921174, 0.999996606257371,
0.999997721707406, 0.999998485553022, 0.999999003265369, 0.999999350536174,
0.999999581060494, 0.999999732489171, 0.999999830917163, 0.999999894220453,
0.99999993450243, 0.012175339654655, 0.0146345118538205, 0.0174980913732582,
0.020815069080104, 0.0246374865772057, 0.0290201646734871, 0.0340203328341099,
0.0396971490863234, 0.0461111012577979, 0.0533232827643063, 0.0613945395482218,
0.0703844892365661, 0.0803504190879397, 0.0913460756798403, 0.10342036629043,
0.116615999184743, 0.130968097067272, 0.146502824291195, 0.163236073476008,
0.181172260460792, 0.200303277565063, 0.22060765362297, 0.242049965045386,
0.26458053528942, 0.288135450824168, 0.312636910421285, 0.337993911984979,
0.364103267906207, 0.390850926879725, 0.418113568053625, 0.445760423007868,
0.473655272965728, 0.501658563232849, 0.529629574326816, 0.557428589583578,
0.584919001981752, 0.611969308118854, 0.638454944189945, 0.664259926877812,
0.689278270674512, 0.713415161771876, 0.73658787682693, 0.758726442290697,
0.779774036383242, 0.799687141135036, 0.818435456242524, 0.836001589940538,
0.852380544864738, 0.867579019166138, 0.881614545126264, 0.8945144893317,
0.906314940153952, 0.917059509818346, 0.926798079633881, 0.935585517839113,
0.943480399813065, 0.950543759924062, 0.956837902898261, 0.962425300201452,
0.967367593543831, 0.971724723330431, 0.975554194870961, 0.978910489686803,
0.981844623615114, 0.984403847936286, 0.986631484756892, 0.988566883641172,
0.990245483211527, 0.99169895926362, 0.992955439908309, 0.99403976831906,
0.994973794712696, 0.995776681047246, 0.996465204368576, 0.997054047545577,
0.99755606907398, 0.997982546494878, 0.998343390594785, 0.998647329802215,
0.99890206599212, 0.999114404220864, 0.99929035974969, 0.999435246116199,
0.999553748048498, 0.999649982766711, 0.999727552768367, 0.999789592631263,
0.999838811763904, 0.999877534449878, 0.999907738013037, 0.999931089503032,
0.99994898097881, 0.999962563251348, 0.999972777826872, 0.999980386752316,
0.999986000086653, 0.999990100785239, 0.999993066870919, 0.999995190859896,
0.999996696500049, 0.0101168586082937, 0.0120554515401813, 0.0142946335759639,
0.0168689190225098, 0.0198151119786215, 0.0231722474415729, 0.0269814876853815,
0.0312859644957301, 0.0361305574954584, 0.0415615990414689, 0.0476264971197689,
0.0543732693699338, 0.0618499838445377, 0.070104105311926, 0.079181749749195,
0.0891268540053516, 0.0999802722593169, 0.111778815636651, 0.124554255940577,
0.138332318640058, 0.153131693779547, 0.168963096079322, 0.185828406952473,
0.203719931283478, 0.222619800457852, 0.242499550238077, 0.263319897669801,
0.285030735391834, 0.307571354731437, 0.330870901104815, 0.354849056903392,
0.379416938677198, 0.404478187499178, 0.429930224367079, 0.455665636780366,
0.481573658539333, 0.507541702558207, 0.533456906136725, 0.559207649630323,
0.584685012602125, 0.609784136025334, 0.634405464548184, 0.658455848810163,
0.681849493881247, 0.704508745696707, 0.726364712559826, 0.747357723158559,
0.767437625970478, 0.786563937400389, 0.804705847588712, 0.821842093703023,
0.837960710884598, 0.853058671088826, 0.867141420044268, 0.880222322634817,
0.892322027302713, 0.903467760632109, 0.913692564091696, 0.923034485918977,
0.931535742200216, 0.939241862191958, 0.946200833686886, 0.952462264603415,
0.958076576857331, 0.963094247877212, 0.967565113823854, 0.971537746687962,
0.975058915038574, 0.978173135386714, 0.980922318055483, 0.983345508269554,
0.98547872006067, 0.987354857692176, 0.989003716778289, 0.990452055229308,
0.991723722674673, 0.99283983614593, 0.993818989545904, 0.994677484755632,
0.995429573070559, 0.996087696919267, 0.996662723390623, 0.997164162858583,
0.99760036782835, 0.99797870892169, 0.998305726577705, 0.99858725849301,
0.998828544010838, 0.99903430756476, 0.999208823885104, 0.999355968000248,
0.999479253142025, 0.999581859536876, 0.99966665678055, 0.999736222104332,
0.999792856392839, 0.999838599349974, 0.999875244765792, 0.999904356439427,
0.99992728497954, 0.00932620079849635, 0.0109857813886373, 0.0128813778274409,
0.0150375040970606, 0.017480237326346, 0.0202372444049372, 0.0233377913220371,
0.026812727138182, 0.0306944338744415, 0.0350167335369288, 0.0398147440281856,
0.0451246768185087, 0.0509835708956128, 0.057428959575526, 0.0644984691122479,
0.0722293505536182, 0.0806579488328349, 0.0898191155607497, 0.0997455743262692,
0.11046724948353, 0.122010571388848, 0.134397772839001, 0.147646193038082,
0.161767606743539, 0.176767597242648, 0.192644992386131, 0.209391382929714,
0.226990741773081, 0.245419161219535, 0.264644723028151, 0.284627512774265,
0.30531978593304, 0.326666288299215, 0.348604728080484, 0.371066391545778,
0.3939768888085, 0.417257011524911, 0.440823680311959, 0.464590956803344,
0.488471093650059, 0.512375595512843, 0.536216265159883, 0.559906211039737,
0.583360795922512, 0.606498510101672, 0.629241756898641, 0.65151754248368,
0.673258066018796, 0.69440120959635, 0.714890930219765, 0.734677558063937,
0.753718006461735, 0.771975899564054, 0.789421623549171, 0.806032306785529,
0.821791733668202, 0.836690196132862, 0.85072428625479, 0.863896632976388,
0.876215585939309, 0.887694849643489, 0.898353071688683, 0.908213389615673,
0.91730294177307, 0.925652348601437, 0.933295171659957, 0.940267358541928,
0.946606682468004, 0.952352185760622, 0.95754363655572, 0.962221007979581,
0.966423988602183, 0.970191532277543, 0.973561454508989, 0.976570081255844,
0.979251954660539, 0.981639598566154, 0.983763344969856, 0.985651220785389,
0.987328892543708, 0.988819665026769, 0.990144528385785, 0.991322247115621,
0.992369483400927, 0.993300946856819, 0.994129562572971, 0.994866649624379,
0.99552210279921, 0.996104571156596, 0.996621628091743, 0.99707992877016,
0.997485352014092, 0.99784312490551, 0.998157929446791, 0.998433991544094,
0.998675153319727, 0.998884930306292, 0.999066555431389, 0.99922301188371,
0.999357056985043, 0.0092301573618731, 0.0107379179109002, 0.0124391690347219,
0.0143514375577615, 0.0164932619433046, 0.0188842600830559, 0.0215451937457382,
0.0244980220315495, 0.0277659355746689, 0.0313733632268641, 0.0353459435502482,
0.039710454589972, 0.0444946969748513, 0.0497273272518415, 0.0554376403268464,
0.0616553017940037, 0.0684100326523342, 0.0757312503440144, 0.0836476711712007,
0.0921868799838051, 0.101374873651218, 0.111235585337494, 0.121790397098141,
0.133057648894695, 0.145052152828433, 0.15778472222086, 0.171261726052062,
0.185484680092426, 0.200449886675629, 0.216148135293852, 0.232564475894134,
0.249678075798878, 0.26746216950282, 0.28588410822374, 0.304905513090472,
0.324482532394688, 0.344566199623015, 0.365102885267874, 0.386034831943426,
0.407300759345676, 0.428836523293666, 0.450575811615456, 0.472450859074184,
0.494393163868851, 0.516334189424236, 0.538206037075878, 0.559942077683181,
0.581477532960668, 0.60275000018828, 0.623699916739887, 0.644270963375107,
0.664410407334624, 0.68406938787472, 0.703203147938081, 0.721771216201532,
0.739737543827748, 0.757070599969785, 0.773743429545535, 0.789733676128535,
0.805023572097686, 0.819599897538049, 0.833453908850095, 0.846581237641756,
0.85898176025915, 0.870659438252705, 0.88162213015907, 0.891881375184617,
0.901452149682945, 0.910352597709693, 0.918603737400515, 0.926229145442113,
0.933254622481295, 0.939707842928775, 0.945617993241796, 0.951015403383026,
0.955931176714891, 0.960396824054606, 0.964443907938979, 0.968103703285223,
0.971406880546906, 0.974383217127292, 0.97706134221565, 0.979468519363697,
0.981630470046313, 0.983571240198069, 0.985313110345598, 0.986876548537574,
0.988280203887653, 0.989540937270206, 0.990673884617516, 0.99169254742198,
0.992608904492772, 0.993433538777655, 0.99417577313861, 0.994843809343502,
0.995444865162976, 0.995985305283612, 0.996470762695338, 0.996906248209148,
0.997296246738763, 0.00954678319050827, 0.0109715985209313, 0.0125600286463309,
0.0143245935276414, 0.0162784387529925, 0.0184354258667758, 0.0208102262715916,
0.0234184113679411, 0.0262765310535083, 0.0294021727283352, 0.0328139935449171,
0.0365317197478467, 0.0405761084605675, 0.0449688690534927, 0.0497325431031736,
0.0548903437592755, 0.0604659569252367, 0.0664833079151107, 0.0729662981067737,
0.0799385165589976, 0.0874229316401502, 0.0954415675198323, 0.104015170024872,
0.113162865995185, 0.122901820023644, 0.133246892430731, 0.144210302569217,
0.155801302081961, 0.168025863495556, 0.180886390420887, 0.194381456506416,
0.208505580989509, 0.223249049055169, 0.238597785104717, 0.254533286367307,
0.271032623020423, 0.288068509151735, 0.305609446586977, 0.323619940973838,
0.342060786735966, 0.36088941479882, 0.38006029454281, 0.399525379438979,
0.419234584408337, 0.439136282205771, 0.45917780609229, 0.479305946695598,
0.49946743218569, 0.519609382583755, 0.539679731024819, 0.55962760693982,
0.579403678246982, 0.59896045159874, 0.618252531402447, 0.637236839641971,
0.655872799436782, 0.674122485790153, 0.691950747140221, 0.709325301205534,
0.726216808294841, 0.742598924817136, 0.758448339260594, 0.773744792468303,
0.788471083659428, 0.802613063335908, 0.816159613963244, 0.829102619090465,
0.841436921344651, 0.853160269470584, 0.864273254271621, 0.874779232949969,
0.884684240971216, 0.893996890236191, 0.902728252091453, 0.910891723607723,
0.918502875653615, 0.925579281621318, 0.932140326226578, 0.938206994584274,
0.943801642703215, 0.948947751578722, 0.953669668105823, 0.957992337003405,
0.961941028750872, 0.965541069128408, 0.968817576273636, 0.971795211197146,
0.974497947434051, 0.976948864965207, 0.979169972752621, 0.981182063244082,
0.98300460106588, 0.98465564689877, 0.986151816282773, 0.987508271882493,
0.988738746625516, 0.989855594156114, 0.990869862271087, 0.991791384460168,
0.992628884382578, 0.0101255146821511, 0.0115047657889316, 0.0130251412947553,
0.0146954986912924, 0.0165250733253271, 0.0185235806527015, 0.0207013244946456,
0.0230693045792453, 0.0256393162132802, 0.0284240349485071, 0.0314370795838294,
0.0346930477479132, 0.0382075195705983, 0.0419970264833451, 0.0460789838759043,
0.0504715880535022, 0.0551936795592414, 0.0602645763317886, 0.0657038812593043,
0.0715312693955861, 0.0777662603876731, 0.0844279815296182, 0.0915349263494587,
0.0991047128379065, 0.107153844450723, 0.115697475994738, 0.124749185578942,
0.134320753107192, 0.144421945414073, 0.155060308168973, 0.166240965115982,
0.177966426045126, 0.190236406015964, 0.203047659641669, 0.21639383551941,
0.230265356972412, 0.244649335967213, 0.259529527230697, 0.274886329110411,
0.290696836560491, 0.306934949833345, 0.323571540130562, 0.340574670799246,
0.357909869883922, 0.375540447212737, 0.393427846954515, 0.411532024936532,
0.429811839104717, 0.448225441401954, 0.466730660014551, 0.485285362290859,
0.503847790504803, 0.52237686481594, 0.540832450048748, 0.559175585072684,
0.577368675443046, 0.595375651444748, 0.613162094709964, 0.630695337157884,
0.64794453618197, 0.664880729874434, 0.681476875735739, 0.697707875877416,
0.71355059128468, 0.728983847331994, 0.74398843247832, 0.758547091914285,
0.772644517864654, 0.786267338216542, 0.799404105083757, 0.812045284766338,
0.824183250269409, 0.835812277076444, 0.846928542228769, 0.857530125978519,
0.867617014420039, 0.877191100652722, 0.886256181286114, 0.894817944563581,
0.902883946134938, 0.910463568602771, 0.917567961415829, 0.924209958459159,
0.930403971730944, 0.936165860708402, 0.941512778282298, 0.946462995373533,
0.951035707440583, 0.955250826971337, 0.959128766684417, 0.962690218530609,
0.965955933697655, 0.968946508712678, 0.971682182446572, 0.974182648394618,
0.976466886072531, 0.97855301475218, 0.980458172082199, 0.982198419404392,
0.983788674794035, 0.0108783786890413, 0.0122314859440017, 0.013707615748003,
0.0153129223192771, 0.0170538018547876, 0.0189369995308057, 0.0209697215229002,
0.0231597460809308, 0.0255155274768542, 0.0280462867750196, 0.0307620838457813,
0.0336738658219726, 0.0367934882338241, 0.0401337062864842, 0.0437081350907959,
0.0475311790431197, 0.0516179318948366, 0.0559840502826715, 0.060645604541573,
0.0656189114379857, 0.0709203540008563, 0.0765661938617683, 0.0825723814292065,
0.0889543688149505, 0.0957269297194796, 0.102903989503786, 0.110498467485186,
0.118522132177202, 0.126985468855606, 0.135897557602797, 0.14526595900154,
0.15509660405661, 0.165393684838651, 0.17615954284797, 0.187394553205878,
0.199097004440919, 0.21126297570902, 0.223886215556549, 0.23695802853324,
0.25046717779132, 0.264399812984763, 0.278739433079742, 0.293466892968404,
0.308560461027094, 0.323995932092203, 0.339746796979945, 0.355784465984881,
0.37207854014423, 0.388597120842097, 0.40530714589117, 0.422174738815255,
0.439165557783775, 0.456245131498967, 0.473379171156491, 0.490533850133499,
0.507676045982452, 0.524773542283534, 0.541795190620739, 0.558711035151011,
0.575492403779777, 0.592111970790955, 0.608543795954025, 0.624763344773737,
0.64074749383897, 0.656474524365354, 0.671924106198363, 0.687077273899718,
0.701916396175093, 0.71642513984833, 0.730588429819249, 0.744392406880329,
0.757824385798619, 0.770872816562455, 0.783527252019751, 0.795778325187143,
0.807617739213226, 0.819038272305107, 0.830033798895792, 0.840599327009873,
0.850731050288645, 0.860426411606185, 0.869684173803649, 0.878504491944963,
0.886888980785823, 0.894840770941914, 0.902364547580935, 0.909466566325997,
0.916154642366149, 0.922438110394279, 0.928327754771196, 0.933835711071997,
0.938975341741797, 0.94376108983816, 0.948208315679823, 0.952333121623999,
0.956152170182022, 0.959682500327686, 0.962941346259516, 0.965945962167005,
0.968713455835743), .Dim = c(100L, 10L))

dat <- nd
dat$Days <- paste0("Days = ", dat$Days)
dat$mp_lm <- c(mp_lmME)
dat$mp_bc <- c(mp_bc)
dat2 <- sleepstudy
dat2$Days <- paste0("Days = ", dat2$Days)

ggplot(dat, aes(x = Reaction)) +
  facet_wrap(~ Days, nrow = 2) +
  geom_line(aes(y = mp_bc, colour = "BoxCoxME")) +
  geom_line(aes(y = mp_lm, colour = "LmME")) +
  stat_ecdf(aes(x = Reaction, colour = "ECDF"), data = dat2, geom = "step") +
  xlab("Average reaction time (ms)") +
  ylab("Marginal distribution") +
  ggth +
  theme(legend.position = "bottom") +
  scale_color_manual(
    values = c(rgb(0, 84, 150, maxColorValue = 255),
               rgb(202, 108, 24, maxColorValue = 255),
               rgb(.5, .5, .5, .5)),
    breaks = c("BoxCoxME", "LmME", "ECDF"))

## ----neckpain-data, echo=FALSE, fig.width=6, fig.height=3---------------------
data("neck_pain", package = "ordinalCont")
dat <- within(neck_pain, {
  trt <- ifelse(laser == 1, "Active", "Placebo")
  t   <- ifelse(time == 1, "Baseline",
         ifelse(time == 2, "Week 7", "Week 12"))
})
ggplot(dat, aes(t, vas, group = id)) +
  facet_wrap(~ trt) +
  geom_line(col = "grey", alpha = 0.5) +
  geom_point(col = "grey", alpha = 0.5, size = 0.75) +
  scale_x_discrete(limits = c("Baseline", "Week 7", "Week 12")) +
  xlab("Follow-up times") +
  ylab("Pain level (VAS)") +
  ggth

## -----------------------------------------------------------------------------
neck_tr <- ColrME(vas ~ laser * time + (1 | id), data = neck_pain,
                  bounds = c(0, 1), support = c(0, 1))

## ----results="hide", message=FALSE, cache=TRUE--------------------------------
library("ordinalCont")
neck_ocm <- ocm(vas ~ laser * time + (1 | id), data = neck_pain, scale = c(0, 1))

## ----neckpain-compestim, echo=FALSE,fig.width=7, fig.height=3.5---------------
par(mfrow = c(1, 2), cex = 0.8)
## -- baseline
h_tr <- trafo(neck_tr, type = "trafo", confidence = "interval", K = 100)
h_ocm <- ordinalCont:::get_gfun.ocm(neck_ocm)
plot(h_tr, col = mycolors(1), fill = mycolors(1, "fill"),
     xlab = "Pain level (VAS)", ylab = expression(h(y)), lwd = 2)
polygon(c(h_ocm[, 1], rev(h_ocm[, 1])), c(h_ocm[, 3], rev(h_ocm[, 4])), border = NA,
        col = mycolors(2, "fill"))
lines(h_ocm[, 1], h_ocm[, 2], col = mycolors(2), lwd = 2)
legend("topleft", c("tramME", "ordinalCont"),
       col = c(mycolors(1), mycolors(2)),
       lty = 1, lwd = 2, bty = "n")
rug(neck_pain$vas, col = rgb(.1, .1, .1, .1))
## --- lORs
idx_ocm <- 2:6
ci_ocm <- cbind(coef(neck_ocm)[idx_ocm], confint(neck_ocm)[idx_ocm, ])
ci_tr <- cbind(coef(neck_tr), confint(neck_tr, pargroup = "shift"))
labs <- c(expression(beta[Active]), expression(beta["7w"]), expression(beta["12w"]),
          expression(beta["7w, Active"]), expression(beta["12w, Active"]))
#par(cex = 0.8, mar = c(4, 10, 2, 2))
plot_cicomp(ci_tr, ci_ocm, labels = labs, xlas = 1,
            mnames = c("tramME", "ordinalCont"), ylab = "Log-odds ratios",
            legend_pos = NULL)

## -----------------------------------------------------------------------------
exp(coef(neck_tr))

## ----eval=FALSE---------------------------------------------------------------
#  jointCDF <- function(re, nd, mod) {
#    nd <- nd[rep(1, length(re)), ]
#    nd$id <- seq(nrow(nd)) ## to take vector-valued REs
#    pr <- predict(mod, newdata = nd, ranef = re, type = "distribution") *
#      dnorm(re, 0, sd = sqrt(varcov(mod)[[1]][1, 1]))
#    c(pr)
#  }
#  marginalCDF <- function(nd, mod) {
#    nd$cdf <- integrate(jointCDF, lower = -Inf, upper = Inf, nd = nd, mod = mod)$value
#    nd
#  }
#  nd <- expand.grid(vas = seq(0, 1, length.out = 100),
#                    time = unique(neck_pain$time),
#                    laser = unique(neck_pain$laser))
#  mp_colr <- parallel::mclapply(split(nd, seq(nrow(nd))),
#                                marginalCDF, mod = neck_tr, mc.cores = 8)
#  mp_colr <- do.call("rbind", mp_colr)

## ----margdist-colr, echo=FALSE------------------------------------------------
## Saved for faster execution
mp_colr <-
structure(list(vas = c(0, 0.010101010101010102, 0.020202020202020204,
0.030303030303030304, 0.040404040404040407, 0.050505050505050511,
0.060606060606060608, 0.070707070707070718, 0.080808080808080815,
0.090909090909090912, 0.10101010101010102, 0.11111111111111112,
0.12121212121212122, 0.13131313131313133, 0.14141414141414144,
0.15151515151515152, 0.16161616161616163, 0.17171717171717174,
0.18181818181818182, 0.19191919191919193, 0.20202020202020204,
0.21212121212121213, 0.22222222222222224, 0.23232323232323235,
0.24242424242424243, 0.25252525252525254, 0.26262626262626265,
0.27272727272727276, 0.28282828282828287, 0.29292929292929293,
0.30303030303030304, 0.31313131313131315, 0.32323232323232326,
0.33333333333333337, 0.34343434343434348, 0.35353535353535359,
0.36363636363636365, 0.37373737373737376, 0.38383838383838387,
0.39393939393939398, 0.40404040404040409, 0.4141414141414142,
0.42424242424242425, 0.43434343434343436, 0.44444444444444448,
0.45454545454545459, 0.4646464646464647, 0.47474747474747481,
0.48484848484848486, 0.49494949494949497, 0.50505050505050508,
0.51515151515151525, 0.5252525252525253, 0.53535353535353536,
0.54545454545454553, 0.55555555555555558, 0.56565656565656575,
0.5757575757575758, 0.58585858585858586, 0.59595959595959602,
0.60606060606060608, 0.61616161616161624, 0.6262626262626263,
0.63636363636363646, 0.64646464646464652, 0.65656565656565657,
0.66666666666666674, 0.6767676767676768, 0.68686868686868696,
0.69696969696969702, 0.70707070707070718, 0.71717171717171724,
0.72727272727272729, 0.73737373737373746, 0.74747474747474751,
0.75757575757575768, 0.76767676767676774, 0.77777777777777779,
0.78787878787878796, 0.79797979797979801, 0.80808080808080818,
0.81818181818181823, 0.8282828282828284, 0.83838383838383845,
0.84848484848484851, 0.85858585858585867, 0.86868686868686873,
0.8787878787878789, 0.88888888888888895, 0.89898989898989912,
0.90909090909090917, 0.91919191919191923, 0.92929292929292939,
0.93939393939393945, 0.94949494949494961, 0.95959595959595967,
0.96969696969696972, 0.97979797979797989, 0.98989898989898994,
1, 0, 0.010101010101010102, 0.020202020202020204, 0.030303030303030304,
0.040404040404040407, 0.050505050505050511, 0.060606060606060608,
0.070707070707070718, 0.080808080808080815, 0.090909090909090912,
0.10101010101010102, 0.11111111111111112, 0.12121212121212122,
0.13131313131313133, 0.14141414141414144, 0.15151515151515152,
0.16161616161616163, 0.17171717171717174, 0.18181818181818182,
0.19191919191919193, 0.20202020202020204, 0.21212121212121213,
0.22222222222222224, 0.23232323232323235, 0.24242424242424243,
0.25252525252525254, 0.26262626262626265, 0.27272727272727276,
0.28282828282828287, 0.29292929292929293, 0.30303030303030304,
0.31313131313131315, 0.32323232323232326, 0.33333333333333337,
0.34343434343434348, 0.35353535353535359, 0.36363636363636365,
0.37373737373737376, 0.38383838383838387, 0.39393939393939398,
0.40404040404040409, 0.4141414141414142, 0.42424242424242425,
0.43434343434343436, 0.44444444444444448, 0.45454545454545459,
0.4646464646464647, 0.47474747474747481, 0.48484848484848486,
0.49494949494949497, 0.50505050505050508, 0.51515151515151525,
0.5252525252525253, 0.53535353535353536, 0.54545454545454553,
0.55555555555555558, 0.56565656565656575, 0.5757575757575758,
0.58585858585858586, 0.59595959595959602, 0.60606060606060608,
0.61616161616161624, 0.6262626262626263, 0.63636363636363646,
0.64646464646464652, 0.65656565656565657, 0.66666666666666674,
0.6767676767676768, 0.68686868686868696, 0.69696969696969702,
0.70707070707070718, 0.71717171717171724, 0.72727272727272729,
0.73737373737373746, 0.74747474747474751, 0.75757575757575768,
0.76767676767676774, 0.77777777777777779, 0.78787878787878796,
0.79797979797979801, 0.80808080808080818, 0.81818181818181823,
0.8282828282828284, 0.83838383838383845, 0.84848484848484851,
0.85858585858585867, 0.86868686868686873, 0.8787878787878789,
0.88888888888888895, 0.89898989898989912, 0.90909090909090917,
0.91919191919191923, 0.92929292929292939, 0.93939393939393945,
0.94949494949494961, 0.95959595959595967, 0.96969696969696972,
0.97979797979797989, 0.98989898989898994, 1, 0, 0.010101010101010102,
0.020202020202020204, 0.030303030303030304, 0.040404040404040407,
0.050505050505050511, 0.060606060606060608, 0.070707070707070718,
0.080808080808080815, 0.090909090909090912, 0.10101010101010102,
0.11111111111111112, 0.12121212121212122, 0.13131313131313133,
0.14141414141414144, 0.15151515151515152, 0.16161616161616163,
0.17171717171717174, 0.18181818181818182, 0.19191919191919193,
0.20202020202020204, 0.21212121212121213, 0.22222222222222224,
0.23232323232323235, 0.24242424242424243, 0.25252525252525254,
0.26262626262626265, 0.27272727272727276, 0.28282828282828287,
0.29292929292929293, 0.30303030303030304, 0.31313131313131315,
0.32323232323232326, 0.33333333333333337, 0.34343434343434348,
0.35353535353535359, 0.36363636363636365, 0.37373737373737376,
0.38383838383838387, 0.39393939393939398, 0.40404040404040409,
0.4141414141414142, 0.42424242424242425, 0.43434343434343436,
0.44444444444444448, 0.45454545454545459, 0.4646464646464647,
0.47474747474747481, 0.48484848484848486, 0.49494949494949497,
0.50505050505050508, 0.51515151515151525, 0.5252525252525253,
0.53535353535353536, 0.54545454545454553, 0.55555555555555558,
0.56565656565656575, 0.5757575757575758, 0.58585858585858586,
0.59595959595959602, 0.60606060606060608, 0.61616161616161624,
0.6262626262626263, 0.63636363636363646, 0.64646464646464652,
0.65656565656565657, 0.66666666666666674, 0.6767676767676768,
0.68686868686868696, 0.69696969696969702, 0.70707070707070718,
0.71717171717171724, 0.72727272727272729, 0.73737373737373746,
0.74747474747474751, 0.75757575757575768, 0.76767676767676774,
0.77777777777777779, 0.78787878787878796, 0.79797979797979801,
0.80808080808080818, 0.81818181818181823, 0.8282828282828284,
0.83838383838383845, 0.84848484848484851, 0.85858585858585867,
0.86868686868686873, 0.8787878787878789, 0.88888888888888895,
0.89898989898989912, 0.90909090909090917, 0.91919191919191923,
0.92929292929292939, 0.93939393939393945, 0.94949494949494961,
0.95959595959595967, 0.96969696969696972, 0.97979797979797989,
0.98989898989898994, 1, 0, 0.010101010101010102, 0.020202020202020204,
0.030303030303030304, 0.040404040404040407, 0.050505050505050511,
0.060606060606060608, 0.070707070707070718, 0.080808080808080815,
0.090909090909090912, 0.10101010101010102, 0.11111111111111112,
0.12121212121212122, 0.13131313131313133, 0.14141414141414144,
0.15151515151515152, 0.16161616161616163, 0.17171717171717174,
0.18181818181818182, 0.19191919191919193, 0.20202020202020204,
0.21212121212121213, 0.22222222222222224, 0.23232323232323235,
0.24242424242424243, 0.25252525252525254, 0.26262626262626265,
0.27272727272727276, 0.28282828282828287, 0.29292929292929293,
0.30303030303030304, 0.31313131313131315, 0.32323232323232326,
0.33333333333333337, 0.34343434343434348, 0.35353535353535359,
0.36363636363636365, 0.37373737373737376, 0.38383838383838387,
0.39393939393939398, 0.40404040404040409, 0.4141414141414142,
0.42424242424242425, 0.43434343434343436, 0.44444444444444448,
0.45454545454545459, 0.4646464646464647, 0.47474747474747481,
0.48484848484848486, 0.49494949494949497, 0.50505050505050508,
0.51515151515151525, 0.5252525252525253, 0.53535353535353536,
0.54545454545454553, 0.55555555555555558, 0.56565656565656575,
0.5757575757575758, 0.58585858585858586, 0.59595959595959602,
0.60606060606060608, 0.61616161616161624, 0.6262626262626263,
0.63636363636363646, 0.64646464646464652, 0.65656565656565657,
0.66666666666666674, 0.6767676767676768, 0.68686868686868696,
0.69696969696969702, 0.70707070707070718, 0.71717171717171724,
0.72727272727272729, 0.73737373737373746, 0.74747474747474751,
0.75757575757575768, 0.76767676767676774, 0.77777777777777779,
0.78787878787878796, 0.79797979797979801, 0.80808080808080818,
0.81818181818181823, 0.8282828282828284, 0.83838383838383845,
0.84848484848484851, 0.85858585858585867, 0.86868686868686873,
0.8787878787878789, 0.88888888888888895, 0.89898989898989912,
0.90909090909090917, 0.91919191919191923, 0.92929292929292939,
0.93939393939393945, 0.94949494949494961, 0.95959595959595967,
0.96969696969696972, 0.97979797979797989, 0.98989898989898994,
1, 0, 0.010101010101010102, 0.020202020202020204, 0.030303030303030304,
0.040404040404040407, 0.050505050505050511, 0.060606060606060608,
0.070707070707070718, 0.080808080808080815, 0.090909090909090912,
0.10101010101010102, 0.11111111111111112, 0.12121212121212122,
0.13131313131313133, 0.14141414141414144, 0.15151515151515152,
0.16161616161616163, 0.17171717171717174, 0.18181818181818182,
0.19191919191919193, 0.20202020202020204, 0.21212121212121213,
0.22222222222222224, 0.23232323232323235, 0.24242424242424243,
0.25252525252525254, 0.26262626262626265, 0.27272727272727276,
0.28282828282828287, 0.29292929292929293, 0.30303030303030304,
0.31313131313131315, 0.32323232323232326, 0.33333333333333337,
0.34343434343434348, 0.35353535353535359, 0.36363636363636365,
0.37373737373737376, 0.38383838383838387, 0.39393939393939398,
0.40404040404040409, 0.4141414141414142, 0.42424242424242425,
0.43434343434343436, 0.44444444444444448, 0.45454545454545459,
0.4646464646464647, 0.47474747474747481, 0.48484848484848486,
0.49494949494949497, 0.50505050505050508, 0.51515151515151525,
0.5252525252525253, 0.53535353535353536, 0.54545454545454553,
0.55555555555555558, 0.56565656565656575, 0.5757575757575758,
0.58585858585858586, 0.59595959595959602, 0.60606060606060608,
0.61616161616161624, 0.6262626262626263, 0.63636363636363646,
0.64646464646464652, 0.65656565656565657, 0.66666666666666674,
0.6767676767676768, 0.68686868686868696, 0.69696969696969702,
0.70707070707070718, 0.71717171717171724, 0.72727272727272729,
0.73737373737373746, 0.74747474747474751, 0.75757575757575768,
0.76767676767676774, 0.77777777777777779, 0.78787878787878796,
0.79797979797979801, 0.80808080808080818, 0.81818181818181823,
0.8282828282828284, 0.83838383838383845, 0.84848484848484851,
0.85858585858585867, 0.86868686868686873, 0.8787878787878789,
0.88888888888888895, 0.89898989898989912, 0.90909090909090917,
0.91919191919191923, 0.92929292929292939, 0.93939393939393945,
0.94949494949494961, 0.95959595959595967, 0.96969696969696972,
0.97979797979797989, 0.98989898989898994, 1, 0, 0.010101010101010102,
0.020202020202020204, 0.030303030303030304, 0.040404040404040407,
0.050505050505050511, 0.060606060606060608, 0.070707070707070718,
0.080808080808080815, 0.090909090909090912, 0.10101010101010102,
0.11111111111111112, 0.12121212121212122, 0.13131313131313133,
0.14141414141414144, 0.15151515151515152, 0.16161616161616163,
0.17171717171717174, 0.18181818181818182, 0.19191919191919193,
0.20202020202020204, 0.21212121212121213, 0.22222222222222224,
0.23232323232323235, 0.24242424242424243, 0.25252525252525254,
0.26262626262626265, 0.27272727272727276, 0.28282828282828287,
0.29292929292929293, 0.30303030303030304, 0.31313131313131315,
0.32323232323232326, 0.33333333333333337, 0.34343434343434348,
0.35353535353535359, 0.36363636363636365, 0.37373737373737376,
0.38383838383838387, 0.39393939393939398, 0.40404040404040409,
0.4141414141414142, 0.42424242424242425, 0.43434343434343436,
0.44444444444444448, 0.45454545454545459, 0.4646464646464647,
0.47474747474747481, 0.48484848484848486, 0.49494949494949497,
0.50505050505050508, 0.51515151515151525, 0.5252525252525253,
0.53535353535353536, 0.54545454545454553, 0.55555555555555558,
0.56565656565656575, 0.5757575757575758, 0.58585858585858586,
0.59595959595959602, 0.60606060606060608, 0.61616161616161624,
0.6262626262626263, 0.63636363636363646, 0.64646464646464652,
0.65656565656565657, 0.66666666666666674, 0.6767676767676768,
0.68686868686868696, 0.69696969696969702, 0.70707070707070718,
0.71717171717171724, 0.72727272727272729, 0.73737373737373746,
0.74747474747474751, 0.75757575757575768, 0.76767676767676774,
0.77777777777777779, 0.78787878787878796, 0.79797979797979801,
0.80808080808080818, 0.81818181818181823, 0.8282828282828284,
0.83838383838383845, 0.84848484848484851, 0.85858585858585867,
0.86868686868686873, 0.8787878787878789, 0.88888888888888895,
0.89898989898989912, 0.90909090909090917, 0.91919191919191923,
0.92929292929292939, 0.93939393939393945, 0.94949494949494961,
0.95959595959595967, 0.96969696969696972, 0.97979797979797989,
0.98989898989898994, 1), time = structure(c(1L, 1L, 1L, 1L, 1L,
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L,
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L,
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L,
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L,
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L,
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 2L,
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L,
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L,
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L,
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L,
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L,
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L,
2L, 2L, 2L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L,
3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L,
3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L,
3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L,
3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L,
3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L,
3L, 3L, 3L, 3L, 3L, 3L, 3L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L,
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L,
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L,
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L,
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L,
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L,
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 2L, 2L, 2L, 2L, 2L,
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L,
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L,
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L,
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L,
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L,
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 3L,
3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L,
3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L,
3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L,
3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L,
3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L,
3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L,
3L, 3L, 3L), .Label = c("1", "2", "3"), class = "factor"), laser = structure(c(2L,
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L,
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L,
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L,
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L,
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L,
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L,
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L,
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L,
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L,
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L,
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L,
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L,
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L,
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L,
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L,
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L,
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L,
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L,
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 1L, 1L, 1L, 1L, 1L,
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L,
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L,
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L,
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L,
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L,
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L,
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L,
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L,
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L,
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L,
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L,
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L,
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L,
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L,
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L,
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L,
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L,
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L,
1L, 1L, 1L, 1L, 1L, 1L, 1L), .Label = c("2", "1"), class = "factor"),
    cdf = c(0, 0.003030359534579561, 0.0040492276645409586, 0.0053053014388041667,
    0.0068250590024800332, 0.008632459463022395, 0.010748293464295048,
    0.013189740009803715, 0.015970135079829501, 0.019098939000216024,
    0.022581875958355705, 0.026421210880159762, 0.030616125628561568,
    0.035163157221150369, 0.040056664355892384, 0.045289293862711445,
    0.05085242480337996, 0.056736574081309159, 0.062931753086015604,
    0.069427769781407342, 0.076214474617028488, 0.083281951683987235,
    0.090620658720045752, 0.098221521004544954, 0.1060759850060382,
    0.11417603798718574, 0.1225141997555199, 0.13108349248129225,
    0.1398773940718554, 0.14888978006415285, 0.15811485842397338,
    0.16754710105908446, 0.17718117528763949, 0.18701187796847848,
    0.19703407450427735, 0.20724264447517626, 0.21763243524945128,
    0.22819822454690339, 0.23893469259677713, 0.24983640423167899,
    0.26089780098872434, 0.27211320304596975, 0.28347682060354668,
    0.29498277412283891, 0.30662512266205622, 0.31839789939161239,
    0.33029515323705538, 0.34231099548033722, 0.3544396500514076,
    0.36667550616084865, 0.37901317185973682, 0.39144752706412134,
    0.40397377454712668, 0.41658748738018903, 0.42928465129452359,
    0.44206170043266058, 0.45491554496579212, 0.46784358906382983,
    0.48084373771987132, 0.49391439094799167, 0.50705442389244482,
    0.52026315140790036, 0.53354027569588813, 0.54688581561536809,
    0.56030001633025706, 0.57378323802104281, 0.5873358224810229,
    0.60095793655287122, 0.61464939155402964, 0.62840943810923655,
    0.64223653617842935, 0.65612810056500193, 0.67008022284311841,
    0.68408737148616949, 0.69814207304565357, 0.71223457855399686,
    0.72635252093541791, 0.74048057112666932, 0.75460010284044476,
    0.76868887843193878, 0.78272077110429605, 0.79666554161795333,
    0.81048869060089501, 0.82415141026917205, 0.83761066155371255,
    0.85081940389313437, 0.86372700480421283, 0.87627985421510168,
    0.88842220382926229, 0.90009724388151824, 0.91124841805602008,
    0.92182096179068107, 0.93176362980711414, 0.94103055614740361,
    0.94958316565782142, 0.95739203196177636, 0.96443855658938726,
    0.97071633085790976, 0.97623204042484257, 1.0000000000000002,
    0, 0.067410773768318624, 0.08208278102783545, 0.098224049444370853,
    0.11570185576020316, 0.13436053825701333, 0.15403025311564092,
    0.17453502530264858, 0.1956996706530223, 0.21735537480995809,
    0.23934388334981047, 0.26152037905060305, 0.28375519904793672,
    0.30593458374287208, 0.32796065988825812, 0.34975085127298811,
    0.37123688958800588, 0.39236357142719269, 0.41308737925747557,
    0.43337505738385085, 0.45320221004009648, 0.47255196848598685,
    0.49141375752934224, 0.50978217899631806, 0.52765601993115607,
    0.5450373862228769, 0.56193095743142807, 0.57834335535755121,
    0.59428261696612661, 0.60975776129536696, 0.62477843969223124,
    0.63935465889487897, 0.65349656697346747, 0.66721429282042088,
    0.68051783066324267, 0.69341696189580004, 0.70592120734642438,
    0.71803980389639299, 0.729781700114294, 0.74115556627169488,
    0.75216981474976174, 0.76283262743454816, 0.77315198723211864,
    0.78313571131605741, 0.79279148415235168, 0.80212688873328963,
    0.81114943479600909, 0.81986658310558569, 0.82828576514979013,
    0.83641439782540217, 0.84425989289669434, 0.8518296611777193,
    0.85913111153373756, 0.8661716449158714, 0.87295864373937093,
    0.87949945699226573, 0.88580138152032828, 0.89180456046888823,
    0.89771735597520919, 0.90334552694817027, 0.90876299535159943,
    0.91397641871935187, 0.91899223920497208, 0.92381665320156447,
    0.92845558166412234, 0.93291464177670158, 0.9371991206315361,
    0.94131395161628217, 0.94526369423818846, 0.9490525181481555,
    0.95268419216035449, 0.95616207909002515, 0.95948913724766938,
    0.96266792942507629, 0.9657006401791598, 0.96858910215391048,
    0.97133483206847981, 0.97393907682976499, 0.97640286999050707,
    0.97872709775385003, 0.98091257827209333, 0.9829601426747322,
    0.9848707304270149, 0.98664548626940596, 0.98828586016628905,
    0.98979370608342587, 0.9911713755048237, 0.9924218017611024,
    0.99354856559633009, 0.99455594922914747, 0.99544895760923946,
    0.99623331544336524, 0.99691543361687696, 0.99750232868402278,
    0.99800156595775691, 0.99842106133957931, 0.99876899223710247,
    0.99905360077167038, 0.9992830247163762, 1.0000000000000002,
    0, 0.042713552641678344, 0.052964381658605876, 0.064495922609518752,
    0.07725385300418447, 0.091159178244056296, 0.10611368854841526,
    0.1220056314649951, 0.13871515558691225, 0.15611920857187617,
    0.17409569669614627, 0.19252682021409231, 0.21130158218689474,
    0.23031752704925523, 0.24948180131120412, 0.26871164652447405,
    0.28793443859435242, 0.30708738204833697, 0.326116956670014,
    0.34497819983565392, 0.36363389301272431, 0.38205370655740678,
    0.40021334399141745, 0.41809371574524001, 0.43568016303953871,
    0.45296174507153986, 0.46993059680674537, 0.48658136022724235,
    0.50291068862211297, 0.51891682119894234, 0.53459922374652602,
    0.54995829011397235, 0.56499509874605114, 0.57971121831201622,
    0.59410855649303018, 0.60818924618061476, 0.62195556363150262,
    0.63540987348317246, 0.64855459593045395, 0.66139219177658182,
    0.67392516148789361, 0.68615605479054198, 0.69808748774392215,
    0.70972216460498871, 0.72106290215777702, 0.73211265452169294,
    0.74287453676956261, 0.75335184598143357, 0.76354807863237617,
    0.77346694346188094, 0.78311236919888672, 0.7924885067202474,
    0.80159972540196378, 0.81045060358251031, 0.81904591319707265,
    0.82739059876186483, 0.83548975099057032, 0.84334857541240171,
    0.85097235643563729, 0.85836641736436292, 0.86553607693235379,
    0.87248660296953706, 0.87922316386624622, 0.88575077855148121,
    0.89200706148426467, 0.89819819339479368, 0.90412682896412788,
    0.90986409194442175, 0.91541350926635234, 0.9207781750112769,
    0.92596071561016757, 0.93096326191075363, 0.935787429579786,
    0.94043430939036854, 0.94490446900108083, 0.94919796784962152,
    0.95331438674161106, 0.95725287359528977, 0.96101220658358122,
    0.96459087557425049, 0.96798718228570046, 0.97119935893298626,
    0.97422570432537492, 0.97706473539358862, 0.97971535028139345,
    0.98217700282089915, 0.98444987207051549, 0.9865350340266229,
    0.98843461676572297, 0.99015193426714743, 0.99169158882312192,
    0.99305953326178054, 0.99426308097042493, 0.99531086819407799,
    0.99621274969465989, 0.99697964055437616, 0.99762328847708792,
    0.99815606253479772, 0.9985905705829432, 1.0000000000000002,
    0, 0.019025837679973311, 0.024257815802023273, 0.030346273333595214,
    0.037307227046430781, 0.045139384181833914, 0.05382544553734387,
    0.063334094051084797, 0.073622413901794664, 0.084638502697833143,
    0.096324076609565795, 0.10861691509289675, 0.1214530400449061,
    0.13476856826661504, 0.14850121282856954, 0.16259143712343632,
    0.17698328519147072, 0.19162492429311503, 0.20646894204888924,
    0.22147244223132628, 0.23659698183614955, 0.25180838853609228,
    0.26707649294707675, 0.28237480499463996, 0.29768015853349916,
    0.31297234355069475, 0.32823374094806163, 0.34344897113303885,
    0.35860456446290745, 0.37368865895949333, 0.38869072858819803,
    0.40360134371441442, 0.41841196404658521, 0.43311476338552468,
    0.44770248476697821, 0.46216832405877639, 0.47650583971295207,
    0.49070888614178809, 0.50477156805658063, 0.51868821305631052,
    0.53245335976269281, 0.54606175885442954, 0.55950838444621998,
    0.57278845337893003, 0.58589745012973549, 0.5988311552098361,
    0.61158567508813544, 0.62415747185839965, 0.63654339105156821,
    0.64874068618114, 0.66074703879516961, 0.67256057299098515,
    0.68417986352606119, 0.69560393682877497, 0.70683226437457736,
    0.71786474804550393, 0.72870169723350919, 0.73934379758101298,
    0.74979207137618531, 0.7600478297372969, 0.7701126168322191,
    0.7799881464886409, 0.78967623166133294, 0.79917870733883867,
    0.80849734759781167, 0.81763377765362399, 0.8265893819157295,
    0.83536520924027158, 0.84396187678479095, 0.85237947411382242,
    0.86061746948145501, 0.86867462052708999, 0.87654889196033814,
    0.8842373831728928, 0.89173626908709791, 0.89904075791443128,
    0.90614506982669563, 0.91304244080454688, 0.91972515607931227,
    0.92618461757293857, 0.93241144950697785, 0.93839564582783008,
    0.94412676221231129, 0.94959415410822501, 0.95478726047378448,
    0.9596959305749323, 0.96431078838513107, 0.96862362586502027,
    0.97262781280580235, 0.97631870721063385, 0.97969404596009069,
    0.98275429750834098, 0.98550294164407215, 0.98794666575790446,
    0.99009544409267858, 0.99196248549534594, 0.99356403704378093,
    0.99491903797664627, 0.99604864202109511, 1.0000000000000002,
    0, 0.011762649762339096, 0.015213009769148859, 0.019298645632529143,
    0.02404967973369828, 0.029484275388970697, 0.035608711505727676,
    0.042418067585587568, 0.049897377354700012, 0.058023097475761815,
    0.066764744360848616, 0.076086571144038204, 0.085949182679565125,
    0.096311014283288254, 0.10712962641883149, 0.11836279052832188,
    0.12996935970089321, 0.14190993165755147, 0.15414732094426989,
    0.16664686290173233, 0.17937657467466758, 0.19230719898774223,
    0.20541215532403059, 0.21866742106026449, 0.2320513624837891,
    0.24554453276859833, 0.25912945115214508, 0.27279037488025526,
    0.28651307306158319, 0.30028460943708046, 0.31409313923357773,
    0.32792772372275164, 0.34177816482415019, 0.35563486104404712,
    0.36948868519924422, 0.38333088370585627, 0.39715299668939363,
    0.41094679776917736, 0.42470425206598578, 0.4384174907590373,
    0.45207880036243636, 0.46568062479032324, 0.47921557822481831,
    0.49267646678397686, 0.50605631700236919, 0.51934840917965941,
    0.53254631371851857, 0.5456439286587349, 0.55863551671615452,
    0.57151574024999652, 0.58427969270716351, 0.59692292522457557,
    0.60944146720756365, 0.62183183984152268, 0.63409106163308548,
    0.64621664521424549, 0.65820658477677141, 0.67005933363415737,
    0.68177377153419538, 0.69334916146773273, 0.70478509583987514,
    0.71608143199128504, 0.72723821718275106, 0.73825560329026463,
    0.74913375160574425, 0.75987272830649533, 0.77047239135141043,
    0.78093226979120833, 0.79125143675141207, 0.80142837766769714,
    0.81146085573044868, 0.82134577693400357, 0.83107905762876744,
    0.84065549804000217, 0.85006866583914609, 0.85931079451834835,
    0.86837270200361516, 0.87724373561213564, 0.88591175006674616,
    0.89436312576101484, 0.90258283474073198, 0.91055456183277583,
    0.91826088789572069, 0.92568354116199569, 0.93280372095956876,
    0.93960249562225018, 0.94606127303011567, 0.95216233792558613,
    0.95788944496799833, 0.96322845057518081, 0.96816796025446927,
    0.97269996181672314, 0.97682040924180136, 0.98052971712339754,
    0.9838331288261255, 0.9867409082591525, 0.98926833560112981,
    0.99143547290725187, 0.99326669538805912, 1.0000000000000002,
    0, 0.016379425830546385, 0.020980288046168489, 0.026364467211639812,
    0.03255397315895777, 0.03955522888153544, 0.047359904103933113,
    0.055946426154104376, 0.065281951411722397, 0.075324586430012302,
    0.086025673652045553, 0.097331993624080307, 0.10918777629570292,
    0.12153645294381524, 0.13432211406987024, 0.14749066565548785,
    0.16099069609050418, 0.17477407943845935, 0.18879634847802063,
    0.20301687433515839, 0.21739888966441723, 0.23190939028273175,
    0.24651894675768007, 0.26120145336181161, 0.27593383750387274,
    0.29069574856121072, 0.30546924116768698, 0.32023846456783611,
    0.33498936667423657, 0.3497094189599706, 0.36438736625261287,
    0.37901300382600606, 0.39357698286012616, 0.40807064430584028,
    0.42248588040144025, 0.43681502249757476, 0.45105075341849743,
    0.46518604228829297, 0.47921409955451566, 0.4931283498270938,
    0.50692242010043476, 0.52059014092803269, 0.5341255581610328,
    0.54752295293694941, 0.5607768677055216, 0.57388213620024786,
    0.58683391540205887, 0.5996277176920396, 0.61225944154970968,
    0.62472539931898363, 0.63702234073267894, 0.64914747105566306,
    0.66109846287400253, 0.67287346072069276, 0.68447107788598938,
    0.69589038491071553, 0.70713088940350433, 0.71819250695754966,
    0.72907552306968981, 0.73978054608576349, 0.75030845131317281,
    0.76066031655724342, 0.77083734945576976, 0.78084080711026382,
    0.79067190864767944, 0.80033174149796626, 0.80982116234629076,
    0.81914069391961952, 0.82829041900088152, 0.83726987333449399,
    0.84607793939784437, 0.85471274336536418, 0.86317155798333367,
    0.87145071449861977, 0.87954552723219881, 0.88745023484095753,
    0.89515796274335557, 0.90266071156118666, 0.90994937670540865,
    0.91701380435222402, 0.92384288894852018, 0.93042471697607865,
    0.9367467609079323, 0.94279612602349061, 0.94855985093521733,
    0.95402526026250123, 0.95918036484746083, 0.96401430127681287,
    0.96851779836259699, 0.97268365384621935, 0.97650720023864135,
    0.97998673411765247, 0.98312388525535555, 0.98592388584140789,
    0.98839572573274659, 0.99056007546419245, 0.99240954665374481,
    0.99398752890244946, 0.99530853299313904, 1.0000000000000002
    )), out.attrs = list(dim = c(vas = 100L, time = 3L, laser = 2L
), dimnames = list(vas = c("vas=0.0000", "vas=0.0101", "vas=0.0202",
"vas=0.0303", "vas=0.0404", "vas=0.0505", "vas=0.0606", "vas=0.0707",
"vas=0.0808", "vas=0.0909", "vas=0.1010", "vas=0.1111", "vas=0.1212",
"vas=0.1313", "vas=0.1414", "vas=0.1515", "vas=0.1616", "vas=0.1717",
"vas=0.1818", "vas=0.1919", "vas=0.2020", "vas=0.2121", "vas=0.2222",
"vas=0.2323", "vas=0.2424", "vas=0.2525", "vas=0.2626", "vas=0.2727",
"vas=0.2828", "vas=0.2929", "vas=0.3030", "vas=0.3131", "vas=0.3232",
"vas=0.3333", "vas=0.3434", "vas=0.3535", "vas=0.3636", "vas=0.3737",
"vas=0.3838", "vas=0.3939", "vas=0.4040", "vas=0.4141", "vas=0.4242",
"vas=0.4343", "vas=0.4444", "vas=0.4545", "vas=0.4646", "vas=0.4747",
"vas=0.4848", "vas=0.4949", "vas=0.5051", "vas=0.5152", "vas=0.5253",
"vas=0.5354", "vas=0.5455", "vas=0.5556", "vas=0.5657", "vas=0.5758",
"vas=0.5859", "vas=0.5960", "vas=0.6061", "vas=0.6162", "vas=0.6263",
"vas=0.6364", "vas=0.6465", "vas=0.6566", "vas=0.6667", "vas=0.6768",
"vas=0.6869", "vas=0.6970", "vas=0.7071", "vas=0.7172", "vas=0.7273",
"vas=0.7374", "vas=0.7475", "vas=0.7576", "vas=0.7677", "vas=0.7778",
"vas=0.7879", "vas=0.7980", "vas=0.8081", "vas=0.8182", "vas=0.8283",
"vas=0.8384", "vas=0.8485", "vas=0.8586", "vas=0.8687", "vas=0.8788",
"vas=0.8889", "vas=0.8990", "vas=0.9091", "vas=0.9192", "vas=0.9293",
"vas=0.9394", "vas=0.9495", "vas=0.9596", "vas=0.9697", "vas=0.9798",
"vas=0.9899", "vas=1.0000"), time = c("time=1", "time=2", "time=3"
), laser = c("laser=1", "laser=2"))), row.names = c("1", "2",
"3", "4", "5", "6", "7", "8", "9", "10", "11", "12", "13", "14",
"15", "16", "17", "18", "19", "20", "21", "22", "23", "24", "25",
"26", "27", "28", "29", "30", "31", "32", "33", "34", "35", "36",
"37", "38", "39", "40", "41", "42", "43", "44", "45", "46", "47",
"48", "49", "50", "51", "52", "53", "54", "55", "56", "57", "58",
"59", "60", "61", "62", "63", "64", "65", "66", "67", "68", "69",
"70", "71", "72", "73", "74", "75", "76", "77", "78", "79", "80",
"81", "82", "83", "84", "85", "86", "87", "88", "89", "90", "91",
"92", "93", "94", "95", "96", "97", "98", "99", "100", "101",
"102", "103", "104", "105", "106", "107", "108", "109", "110",
"111", "112", "113", "114", "115", "116", "117", "118", "119",
"120", "121", "122", "123", "124", "125", "126", "127", "128",
"129", "130", "131", "132", "133", "134", "135", "136", "137",
"138", "139", "140", "141", "142", "143", "144", "145", "146",
"147", "148", "149", "150", "151", "152", "153", "154", "155",
"156", "157", "158", "159", "160", "161", "162", "163", "164",
"165", "166", "167", "168", "169", "170", "171", "172", "173",
"174", "175", "176", "177", "178", "179", "180", "181", "182",
"183", "184", "185", "186", "187", "188", "189", "190", "191",
"192", "193", "194", "195", "196", "197", "198", "199", "200",
"201", "202", "203", "204", "205", "206", "207", "208", "209",
"210", "211", "212", "213", "214", "215", "216", "217", "218",
"219", "220", "221", "222", "223", "224", "225", "226", "227",
"228", "229", "230", "231", "232", "233", "234", "235", "236",
"237", "238", "239", "240", "241", "242", "243", "244", "245",
"246", "247", "248", "249", "250", "251", "252", "253", "254",
"255", "256", "257", "258", "259", "260", "261", "262", "263",
"264", "265", "266", "267", "268", "269", "270", "271", "272",
"273", "274", "275", "276", "277", "278", "279", "280", "281",
"282", "283", "284", "285", "286", "287", "288", "289", "290",
"291", "292", "293", "294", "295", "296", "297", "298", "299",
"300", "301", "302", "303", "304", "305", "306", "307", "308",
"309", "310", "311", "312", "313", "314", "315", "316", "317",
"318", "319", "320", "321", "322", "323", "324", "325", "326",
"327", "328", "329", "330", "331", "332", "333", "334", "335",
"336", "337", "338", "339", "340", "341", "342", "343", "344",
"345", "346", "347", "348", "349", "350", "351", "352", "353",
"354", "355", "356", "357", "358", "359", "360", "361", "362",
"363", "364", "365", "366", "367", "368", "369", "370", "371",
"372", "373", "374", "375", "376", "377", "378", "379", "380",
"381", "382", "383", "384", "385", "386", "387", "388", "389",
"390", "391", "392", "393", "394", "395", "396", "397", "398",
"399", "400", "401", "402", "403", "404", "405", "406", "407",
"408", "409", "410", "411", "412", "413", "414", "415", "416",
"417", "418", "419", "420", "421", "422", "423", "424", "425",
"426", "427", "428", "429", "430", "431", "432", "433", "434",
"435", "436", "437", "438", "439", "440", "441", "442", "443",
"444", "445", "446", "447", "448", "449", "450", "451", "452",
"453", "454", "455", "456", "457", "458", "459", "460", "461",
"462", "463", "464", "465", "466", "467", "468", "469", "470",
"471", "472", "473", "474", "475", "476", "477", "478", "479",
"480", "481", "482", "483", "484", "485", "486", "487", "488",
"489", "490", "491", "492", "493", "494", "495", "496", "497",
"498", "499", "500", "501", "502", "503", "504", "505", "506",
"507", "508", "509", "510", "511", "512", "513", "514", "515",
"516", "517", "518", "519", "520", "521", "522", "523", "524",
"525", "526", "527", "528", "529", "530", "531", "532", "533",
"534", "535", "536", "537", "538", "539", "540", "541", "542",
"543", "544", "545", "546", "547", "548", "549", "550", "551",
"552", "553", "554", "555", "556", "557", "558", "559", "560",
"561", "562", "563", "564", "565", "566", "567", "568", "569",
"570", "571", "572", "573", "574", "575", "576", "577", "578",
"579", "580", "581", "582", "583", "584", "585", "586", "587",
"588", "589", "590", "591", "592", "593", "594", "595", "596",
"597", "598", "599", "600"), class = "data.frame")

## ----neckpain-margdist,echo=FALSE,fig.width=7,fig.height=3--------------------
dat <- within(mp_colr, {
  trt <- factor(ifelse(laser == 1, "Active", "Placebo"), levels = c("Active", "Placebo"))
  t   <- factor(ifelse(time == 1, "Baseline",
                ifelse(time == 2, "Week 7", "Week 12")),
                levels = c("Baseline", "Week 7", "Week 12"))
})
dat2 <- within(neck_pain, {
  trt <- factor(ifelse(laser == 1, "Active", "Placebo"), levels = c("Active", "Placebo"))
  t   <- factor(ifelse(time == 1, "Baseline",
                ifelse(time == 2, "Week 7", "Week 12")),
                levels = c("Baseline", "Week 7", "Week 12"))
})
ggplot() +
  facet_wrap(~ t) +
  geom_line(data = dat, aes(vas, cdf, color = trt)) +
  stat_ecdf(aes(x = vas, color = laser, group = laser), data = dat2, geom = "step",
            size = 0.3, alpha = 0.5) +
  xlab("Pain level (VAS)") +
  ylab("Marginal distribution") +
  ggth +
  theme(legend.position = "bottom") +
  scale_color_manual(
    values = c(rgb(0, 84, 150, maxColorValue = 255),
               rgb(202, 108, 24, maxColorValue = 255),
               rgb(0, 84, 150, maxColorValue = 255),
               rgb(202, 108, 24, maxColorValue = 255)),
    breaks = c("Active", "Placebo"))

## ----cache=TRUE---------------------------------------------------------------
data("eortc", package = "coxme")
eortc$trt <- factor(eortc$trt, levels = c(0, 1))
eortc_cp <- CoxphME(Surv(y, uncens) ~ trt + (1 | center/trt), data = eortc,
                    log_first = TRUE, order = 10)

## -----------------------------------------------------------------------------
exp(confint(eortc_cp, parm = "trt1", estimate = TRUE))

## ----cache=TRUE---------------------------------------------------------------
exp(confint(eortc_cp, pargroup = "ranef", type = "profile", estimate = TRUE,
            ncpus = 2, parallel = "multicore"))

## ----cache=TRUE, warning=FALSE------------------------------------------------
eortc_cp2 <- CoxphME(Surv(y, uncens) | 0 + trt ~ 0 + (1 | center/trt), data = eortc,
                     log_first = TRUE, order = 10)
tr <- trafo(eortc_cp2, confidence = "interval")

## ----eortc-trafo, echo=FALSE, fig.width=4.5, fig.height=4, warning=FALSE, out.width="0.75\\linewidth"----
x <- log(as.numeric(rownames(tr[[1]]))[-1])
par(cex = 0.8, mfrow = c(1, 1))
plot(0, type = "n", xlim = range(x), ylim = range(tr[[1]], tr[[2]], na.rm = TRUE),
     xlab = "log-time", ylab = expression(h(y)), cex.axis = 0.8, cex.lab = 0.8)
polygon(c(x, rev(x)), c(tr[[1]][-1, 2], rev(tr[[1]][-1, 3])), border = NA,
        col = mycolors(1, "fill"))
lines(x, tr[[1]][-1, 1], lwd = 2, col = mycolors(1))
polygon(c(x, rev(x)), c(tr[[2]][-1, 2], rev(tr[[2]][-1, 3])), border = NA,
        col = mycolors(2, "fill"))
lines(x, tr[[2]][-1, 1], lwd = 2, col = mycolors(2))
rug(log(eortc$y), col = rgb(.1, .1, .1, .1))
legend("topleft", c("treatment", "control"),
       col = c(mycolors(1), mycolors(2)), lwd = 2, bty = "n", cex = 0.8)

## -----------------------------------------------------------------------------
eortc_w <- SurvregME(Surv(y, uncens) ~ trt + (1 | center/trt), data = eortc,
                     dist = "weibull")

## ----cache=TRUE---------------------------------------------------------------
## --- CoxphME
c(coef = coef(eortc_cp), se = sqrt(diag(vcov(eortc_cp, pargroup = "shift"))))
VarCorr(eortc_cp)
## --- SurvregME
c(coef = -coef(eortc_w), se = sqrt(diag(vcov(eortc_w, pargroup = "shift"))))
VarCorr(eortc_w)

## -----------------------------------------------------------------------------
c(logLik(eortc_cp), logLik(eortc_w))

## ----message=FALSE, cache=TRUE------------------------------------------------
library("coxme")
eortc_cm <- coxme(Surv(y, uncens) ~ trt + (1 | center/trt), data = eortc)
summary(eortc_cm)

## ----soup-data, echo=FALSE, fig.width = 6, fig.height=3.5---------------------
data("soup", package = "ordinal")
dat <- within(soup, {
  prod <- factor(ifelse(PROD == "Ref", "Reference", "Test"), levels = c("Reference", "Test"))
})
ggplot(data = dat, aes(x = prod, fill = SURENESS)) +
  facet_wrap(~ SOUPFREQ) +
  geom_bar(position = "fill") +
  scale_fill_grey() +
  xlab("Product") +
  ylab("Proportion") +
  ggth +
  theme(axis.ticks.x = element_blank(), legend.position = "bottom",
        legend.title = element_text()) +
  guides(fill = guide_legend(nrow = 1, title = "Sureness"))

## ----cache=TRUE---------------------------------------------------------------
soup_pr <- PolrME(SURENESS ~ PROD + SOUPFREQ + (1 | RESP/PROD),
                  data = soup, method = "probit")
logLik(soup_pr)

## ----cache=TRUE, message=FALSE------------------------------------------------
library("ordinal")
soup_or <- clmm(SURENESS ~ PROD + SOUPFREQ + (1 | RESP/PROD), data = soup,
                link = "probit")
logLik(soup_or)

## -----------------------------------------------------------------------------
max(abs(coef(soup_or) - coef(soup_pr, with_baseline = TRUE)))

## ----cache=TRUE---------------------------------------------------------------
soup_pr2 <- PolrME(SURENESS | SOUPFREQ ~ PROD + (1 | RESP/PROD),
                   data = soup, method = "probit")
logLik(soup_pr2)

## -----------------------------------------------------------------------------
anova(soup_pr, soup_pr2)

## ----include=FALSE------------------------------------------------------------
## NOTE: reset user's options and pars (as per CRAN submission request)
options(oldopt)
par(oldpar)

