% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/style_tt.R, R/zzz-aliases.R
\name{style_tt}
\alias{style_tt}
\alias{tt_style}
\title{Style a Tiny Table}
\usage{
style_tt(
  x,
  i = NULL,
  j = NULL,
  bold = FALSE,
  italic = FALSE,
  monospace = FALSE,
  smallcap = FALSE,
  underline = FALSE,
  strikeout = FALSE,
  color = NULL,
  background = NULL,
  fontsize = NULL,
  align = NULL,
  alignv = NULL,
  colspan = NULL,
  rowspan = NULL,
  indent = NULL,
  line = NULL,
  line_color = "black",
  line_width = 0.1,
  line_trim = NULL,
  finalize = NULL,
  ...
)

tt_style(
  x,
  i = NULL,
  j = NULL,
  bold = FALSE,
  italic = FALSE,
  monospace = FALSE,
  smallcap = FALSE,
  underline = FALSE,
  strikeout = FALSE,
  color = NULL,
  background = NULL,
  fontsize = NULL,
  align = NULL,
  alignv = NULL,
  colspan = NULL,
  rowspan = NULL,
  indent = NULL,
  line = NULL,
  line_color = "black",
  line_width = 0.1,
  line_trim = NULL,
  finalize = NULL,
  ...
)
}
\arguments{
\item{x}{A table object created by \code{tt()}.}

\item{i}{Numeric vector, logical matrix, string, or unquoted expression.
\itemize{
\item Numeric vector: Row indices where the styling should be applied. Can be a single value or a vector.
\item Logical matrix: A matrix with the same number of rows and columns as \code{x}. \code{i=0} is the header, and negative values are higher level headers. Row indices refer to rows \emph{after} the insertion of row labels by \code{group_tt()}, when applicable.
\item String: Table components "caption", "colnames", "groupi" (row group labels), "~groupi" (non-group rows), "groupj" (column group labels), "notes".
\item Unquoted expression: When supplying an unquoted expression, it is first evaluated in the calling environment, then in the data frame passed to \code{tt()}.
}}

\item{j}{Column indices where the styling should be applied. Can be:
\itemize{
\item Integer vectors indicating column positions.
\item Character vector indicating column names.
\item A single string specifying a Perl-style regular expression used to match column names.
\item Unquoted expression: Non-standard evaluation is supported. When supplying an unquoted expression, it is first evaluated in the calling environment, then in an environment that includes the columns of the original data passed to \code{tt()}, and \code{groupi} indices. See examples below.
}}

\item{bold}{Logical; if \code{TRUE}, text is styled in bold.}

\item{italic}{Logical; if \code{TRUE}, text is styled in italic.}

\item{monospace}{Logical; if \code{TRUE}, text is styled in monospace font.}

\item{smallcap}{Logical; if \code{TRUE}, text is styled in small caps. In Markdown output format, text is converted to uppercase.}

\item{underline}{Logical; if \code{TRUE}, text is underlined.}

\item{strikeout}{Logical; if \code{TRUE}, text has a strike through line.}

\item{color}{Text color. Colors are standardized across output formats and can be specified as:
\itemize{
\item Hex codes: "#CC79A7", "#FF0000", "#123ABC"
\item R color names: Any color recognized by R, such as "red", "blue", "forestgreen", "lightblue"
\item Extended color names: 749+ named colors from the LaTeX xcolor package (see \code{tinytable:::latex_colors} for the full list)
\item LaTeX color blending (LaTeX output only): "white!80!blue", "red!50", "green!20!red"
}}

\item{background}{Background color. Same color specification options as the \code{color} parameter. Can be \code{NULL} for default color.}

\item{fontsize}{Font size in em units. Can be \code{NULL} for default size.}

\item{align}{A single character or a string with a number of characters equal to the number of columns in \code{j}. Valid characters include 'c' (center), 'l' (left), 'r' (right), 'd' (decimal). Decimal alignment is only available in LaTeX via the \code{siunitx} package. The width of columns is determined by the maximum number of digits to the left and to the right in all cells specified by \code{i} and \code{j}.}

\item{alignv}{A single character specifying vertical alignment. Valid characters include 't' (top), 'm' (middle), 'b' (bottom).}

\item{colspan}{Number of columns a cell should span. \code{i} and \code{j} must be of length 1.}

\item{rowspan}{Number of rows a cell should span. \code{i} and \code{j} must be of length 1.}

\item{indent}{Text indentation in em units. Positive values only.}

\item{line}{String determines if solid lines (rules or borders) should be drawn around the cell, row, or column.
\itemize{
\item "t": top
\item "b": bottom
\item "l": left
\item "r": right
\item Can be combined such as: "lbt" to draw borders at the left, bottom, and top.
}}

\item{line_color}{Color of the line. See the \code{color} argument for details.}

\item{line_width}{Width of the line in em units (default: 0.1).}

\item{line_trim}{String specifying line trimming. Acceptable values: "l" (left), "r" (right), "lr" (both sides). When specified, shortens the lines by 0.8pt on the specified side(s). Default: NULL (no trimming).}

\item{finalize}{A function applied to the table object at the very end of table-building, for post-processing. For example, the function could use regular expressions to add LaTeX commands to the text version of the table hosted in \code{x@table_string}, or it could programmatically change the caption in \code{x@caption}.}

\item{...}{extra arguments are ignored}
}
\value{
An object of class \code{tt} representing the table.
}
\description{
Style a Tiny Table

Alias for \code{style_tt()}
}
\details{
This function applies styling to a table created by \code{tt()}. It allows customization of text style (bold, italic, monospace), text and background colors, font size, cell width, text alignment, column span, and indentation. The function also supports passing native instructions to LaTeX (tabularray) and HTML (bootstrap) formats.
}
\section{Markdown limitations}{


Markdown is a text-only format that only supports these styles: italic, bold, strikeout. The \code{width} argument is also unavailable.
These limitations exist because there is no standard markdown syntax for the other styling options.

However, in terminals (consoles) that support it, \code{tinytable} can display colors and text styles using
ANSI escape codes by setting \code{theme_markdown(ansi = TRUE)}. This allows for rich formatting in
compatible terminal environments.
}

\section{Word limitations}{


Word tables only support these styles: italic, bold, strikeout. The \code{width} argument is also unavailable.
Moreover, the \code{style_tt()} function cannot be used to style headers inserted by the \code{group_tt()} function;
instead, you should style the headers directly in the header definition using markdown syntax:
\code{group_tt(i = list("*italic header*" = 2))}. These limitations are due to the fact that we create
Word documents by converting a markdown table to .docx via the Pandoc software, which requires
going through a text-only intermediate format.
}

\examples{
\dontshow{if (knitr::is_html_output()) (if (getRversion() >= "3.4") withAutoprint else force)(\{ # examplesIf}
\dontshow{\}) # examplesIf}
if (knitr::is_html_output()) options(tinytable_print_output = "html")

library(tinytable)

tt(mtcars[1:5, 1:6])

# Alignment
tt(mtcars[1:5, 1:6]) |>
  style_tt(j = 1:5, align = "lcccr")

# Colors and styles
tt(mtcars[1:5, 1:6]) |>
  style_tt(i = 2:3, background = "black", color = "orange", bold = TRUE)

# column selection with `j``
tt(mtcars[1:5, 1:6]) |>
  style_tt(j = 5:6, background = "pink")

tt(mtcars[1:5, 1:6]) |>
  style_tt(j = "drat|wt", background = "pink")

tt(mtcars[1:5, 1:6]) |>
  style_tt(j = c("drat", "wt"), background = "pink")

tt(mtcars[1:5, 1:6], theme = "empty") |>
  style_tt(
    i = 2, j = 2,
    colspan = 3,
    rowspan = 2,
    align = "c",
    alignv = "m",
    color = "white",
    background = "black",
    bold = TRUE)

tt(mtcars[1:5, 1:6], theme = "empty") |>
  style_tt(
    i = 0:3,
    j = 1:3,
    line = "tblr",
    line_width = 0.4,
    line_color = "teal")

tt(mtcars[1:5, 1:6], theme = "striped") |>
  style_tt(
    i = c(2, 5),
    j = 3,
    strikeout = TRUE,
    fontsize = 0.7)

# Non-standard evaluation (NSE)
dat <- data.frame(
  w = c(143002.2092, 201399.181, 100188.3883),
  x = c(1.43402, 201.399, 0.134588),
  y = as.Date(c(897, 232, 198), origin = "1970-01-01"),
  z = c(TRUE, TRUE, FALSE)
)
tt(dat) |>
  style_tt(i = w > 150000, j = c("w", "x"), 
           color = "white", background = "black")

tt(mtcars[1:5, 1:6]) |>
  theme_html(class = "table table-dark table-hover")


inner <- "
column{1-4}={halign=c},
hlines = {fg=white},
vlines = {fg=white},
cell{1,6}{odd} = {bg=teal7},
cell{1,6}{even} = {bg=green7},
cell{2,4}{1,4} = {bg=red7},
cell{3,5}{1,4} = {bg=purple7},
cell{2}{2} = {r=4,c=2}{bg=azure7},
"
tt(mtcars[1:5, 1:4], theme = "empty") |>
  theme_latex(inner = inner)

# Style group rows and non-group rows
dat <- data.frame(x = 1:6, y = letters[1:6])
dat |>
  tt() |>
  group_tt(i = list("Group A" = 3)) |>
  style_tt(i = "groupi", background = "lightblue") |>
  style_tt(i = "~groupi", background = "lightgray")

# unquote expressions
dat <- mtcars[1:10,]
dat <- dat[order(dat$am),]
tt(dat) |>
   subset(mpg > 20) |>
   group_tt(am)

# style elements: captions and colnames
notes <- list(
  "*" = "Hello world", 
  "a" = "Bacon ipsum dolor amet kevin t-bone porchetta.")
tt(head(iris), 
   width = .5,
   caption = "This is a Caption Example.", 
   notes = notes) |>
  style_tt(2, 2, background = "pink", rowspan = 2, colspan = 2, 
    alignv = "m", align = "c", line = "tblr") |>
  style_tt("colnames", italic = TRUE) |>
  style_tt("caption", smallcap = TRUE)

}
