# A geom_pointrange but with sensible defaults for displaying multiple intervals
#
# Author: mjskay
###############################################################################


# Names that should be suppressed from global variable check by codetools
# Names used broadly should be put in _global_variables.R
globalVariables(c(".lower", ".upper", ".width"))


#' Point + multiple probability interval plots (ggplot geom)
#'
#' Modified versions of \code{\link{geom_pointrange}} and \code{\link{geom_pointrangeh}} with default aesthetics
#' designed for use with output from \code{\link{point_interval}}.
#'
#' \code{geom_pointinterval} is a modified version of \code{\link{geom_pointrange}}, and \code{geom_pointintervalh} is
#' a modified version of \code{\link{geom_pointrangeh}}. These geoms set some default aesthetics equal
#' to the \code{.lower}, \code{.upper}, and \code{.width} columns generated by the \code{point_interval} family
#' of functions, making them often more convenient than vanilla \code{\link{geom_pointrange}} or
#' \code{\link{geom_pointrangeh}} when used with functions like \code{\link{median_qi}}, \code{\link{mean_qi}},
#' \code{\link{mode_hdi}}, etc.
#'
#' Specifically, \code{geom_pointinterval} acts as if its default aesthetics are
#' \code{aes(ymin = .lower, ymax = .upper, size = -.width)}. \code{geom_pointintervalh} acts as if its default
#' aesthetics are \code{aes(xmin = .lower, xmax = .upper, size = -.width)}.
#'
#' Both geoms provides a scaling factor for line width as well as point size through the \code{fatten.interval} and
#' \code{fatten.point} arguments; this scaling factor is designed to give multiple probability intervals reasonable
#' scaling at the default settings for \code{\link{scale_size_continuous}}. Finally, these geoms default to not
#' displaying the legend, though this can be overridden through setting \code{show.legend = NA} (the setting for most
#' geoms) or \code{show.legend = TRUE}.
#'
#' @param mapping The aesthetic mapping, usually constructed with
#' \code{\link{aes}} or \code{\link{aes_string}}. Only needs to be set at the
#' layer level if you are overriding the plot defaults.
#' @param data A layer specific dataset - only needed if you want to override
#' the plot defaults.
#' @param stat The statistical transformation to use on the data for this layer.
#' @param position The position adjustment to use for overlapping points on this layer.
#' @param ...  Other arguments passed to \code{\link{layer}}.
#' @param size_domain The minimum and maximum of the values of the size aesthetic that will be translated into actual
#' sizes drawn according to \code{size_range} (see the documentation for that argument, below.)
#' @param size_range This geom scales the raw size aesthetic values, as they tend to be too thick when using the default
#' settings of \code{\link{scale_size_continuous}}, which give sizes with a range of \code{c(1, 6)}. The
#' \code{size_domain} value indicates the input domain of raw size values (typically this should be equal to the value
#' of the \code{range} argument of the \code{\link{scale_size_continuous}} function), and \code{size_range} indicates
#' the desired output range of the size values (the min and max of the actual sizes used to draw intervals).
#' @param fatten_point A multiplicative factor used to adjust the size of the point relative to the size of the
#' thickest line.
#' @param na.rm	If \code{FALSE}, the default, missing values are removed with a warning. If \code{TRUE}, missing
#' values are silently removed.
#' @param show.legend Should this layer be included in the legends? Default is \code{c(size = FALSE)}, unlike most geoms,
#' to match its common use cases. \code{FALSE} hides all legends, \code{TRUE} shows all legends, and \code{NA} shows only
#' those that are mapped (the default for most geoms).
#' @param inherit.aes If \code{FALSE}, overrides the default aesthetics, rather than combining with them. This is
#' most useful for helper functions that define both data and aesthetics and shouldn't inherit behavior from the
#' default plot specification, e.g. borders.
#' @author Matthew Kay
#' @seealso See \code{\link{geom_lineribbon}} for a similar geom designed for curves plus probability bands. See
#' \code{\link{geom_pointrange}} and \code{\link{geom_pointrangeh}} for the geoms these are based on.
#' @keywords manip
#' @examples
#'
#' library(magrittr)
#' library(ggplot2)
#'
#' data(RankCorr, package = "tidybayes")
#'
#' RankCorr %>%
#'   spread_draws(u_tau[i]) %>%
#'   median_qi(.width = c(.8, .95)) %>%
#'   ggplot(aes(y = i, x = u_tau)) +
#'   geom_pointintervalh()
#'
#' RankCorr %>%
#'   spread_draws(u_tau[i]) %>%
#'   median_qi(.width = c(.8, .95)) %>%
#'   ggplot(aes(x = i, y = u_tau)) +
#'   geom_pointinterval()
#'
#' @import ggplot2
#' @export
geom_pointinterval <- function(mapping = NULL, data = NULL,
  stat = "identity", position = "identity",
  ...,
  size_domain = c(1, 6),
  size_range = c(0.6, 1.4),
  fatten_point = 1.8,
  na.rm = FALSE,
  show.legend = c(size = FALSE),
  inherit.aes = TRUE) {

  l = layer(
    data = data,
    mapping = mapping,
    stat = stat,
    geom = GeomPointinterval,
    position = position,
    show.legend = show.legend,
    inherit.aes = inherit.aes,
    params = list(
      size_domain = size_domain,
      size_range = size_range,
      fatten_point = fatten_point,
      na.rm = na.rm,
      ...
    )
  )

  #provide some default computed aesthetics
  default_computed_aesthetics = aes(ymin = .lower, ymax = .upper, size = -.width)

  compute_aesthetics = l$compute_aesthetics
  l$compute_aesthetics = function(self, data, plot) {
    apply_default_computed_aesthetics(self, plot, default_computed_aesthetics)
    compute_aesthetics(data, plot)
  }

  map_statistic = l$map_statistic
  l$map_statistic = function(self, data, plot) {
    apply_default_computed_aesthetics(self, plot, default_computed_aesthetics)
    map_statistic(data, plot)
  }

  l
}

#' @importFrom grid grobTree
draw_key_pointinterval <- function(data, params, size) {
  grobTree(
    draw_key_path(transform(data, size = (3 + data$size) * 0.15), params, size)
  )
}

#' @rdname tidybayes-ggproto
#' @format NULL
#' @usage NULL
#' @importFrom grid grobName gTree gList
#' @import ggplot2
#' @export
GeomPointinterval <- ggproto("GeomPointinterval", Geom,
  default_aes = aes(colour = "black", size = 1.35, linetype = 1, shape = 19,
    fill = NA, alpha = NA, stroke = 1),

  draw_key = draw_key_pointinterval,

  required_aes = c("x", "y", "ymin", "ymax"),

  draw_panel = function(
      data, panel_scales, coord, size_domain = c(1, 6), size_range = c(0.6, 1.4), fatten_point = 1.8
    ) {

    line_data = transform(data,
      size = pmax(
        (size - size_domain[[1]]) / (size_domain[[2]] - size_domain[[1]]) *
        (size_range[[2]] - size_range[[1]]) + size_range[[1]],
        0)
    )

    if (is.null(data$y)) {
      return(GeomInterval$draw_panel(line_data, panel_scales, coord))
    }

    ggname("geom_pointinterval",
      gTree(children = gList(
        GeomInterval$draw_panel(line_data, panel_scales, coord),
        GeomPoint$draw_panel(transform(line_data, size = size * fatten_point), panel_scales, coord)
      ))
    )
  }
)
