% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/statGraph.R
\name{fast.graph.param.estimator}
\alias{fast.graph.param.estimator}
\title{Degree-based graph parameter estimator}
\usage{
fast.graph.param.estimator(
  G,
  model,
  lo = NULL,
  hi = NULL,
  eps = 0.001,
  from = NULL,
  to = NULL,
  npoints = 2000,
  numCores = 1
)
}
\arguments{
\item{G}{The undirected unweighted graph (igraph type).}

\item{model}{Either a string or a function:

A string that indicates one of the following models: "ER" (Erdos-Renyi random
graph model), "GRG" (geometric random graph model), "WS" (Watts-Strogatz
model), and "BA" (Barabasi-Albert model).

A function that returns a Graph generated by a graph model. It must contain
two arguments: the first one corresponds to the graph size and the second to
the parameter of the model.}

\item{lo}{Smallest parameter value that the graph model can take.

If ``model'' is an string, then the default value of 0 is used for the
predefined models ("ER", "GRG", "WS", and "BA").}

\item{hi}{Largest parameter value that the graph model can take.

If ``model'' is an string, then the default values are used for the
predefined models 1 for "ER", sqrt(2) for "GRG", 1 for "WS", and 5 for "BA").}

\item{eps}{Desired precision of the parameter estimate.}

\item{from}{Lower end of the interval that contain the eigenvalues to
generate the degree-based spectral densities. The smallest eigenvalue of the
adjacency matrix corresponding to ``graph'' is used if the value is not given.}

\item{to}{Upper end of the interval that contain the eigenvalues to generate
the degree-based spectral densities. The largest eigenvalue of the adjacency
matrix corresponding to ``graph'' is used if the value is not given.}

\item{npoints}{Number of points to discretize the interval <\code{from},\code{to}>.}

\item{numCores}{Number of cores to use for parallelization.}
}
\value{
Returns a list containing:
\item{param}{The degree-based parameter estimate. For the "ER", "GRG", "WS",
and "BA" models, the parameter corresponds to the probability to connect a
pair of vertices, the radius used to construct the geometric graph in a unit
square, the probability to reconnect a vertex, and the scaling exponent
respectively.}
\item{L1_dist}{The L1 distance between the observed graph and the graph model
with the estimated value.}
}
\description{
\code{fast.graph.param.estimator} estimates the parameter of the complex
network model using the degree-based spectral density and ternary search.
}
\examples{
set.seed(42)

### Example giving only the name of the model to use
G <- igraph::sample_smallworld(dim = 1, size = 15, nei = 2, p = 0.2)

# Obtain the parameter of the WS model
estimated.parameter1 <- fast.graph.param.estimator(G, "WS", lo = 0.1, hi = 0.5,
                                                  eps = 1e-1, npoints = 10,
                                                  numCores = 1)
estimated.parameter1

\dontrun{
### Example giving a function instead of a model

# Defining the model to use
G <- igraph::sample_smallworld(dim = 1, size = 5000, nei = 2, p = 0.2)
K <- as.integer(igraph::ecount(G)/igraph::vcount(G))
fun_WS <- function(n, param, nei = K){
 return (igraph::sample_smallworld(dim = 1,size = n, nei = nei, p = param))
}

# Obtain the parameter of the WS model
estimated.parameter2 <- fast.graph.param.estimator(G, fun_WS, lo = 0.0, hi = 1.0,
                                                   npoints = 100, numCores = 2)
estimated.parameter2
}

}
\keyword{degree_based_parameter_estimation}
