% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/statGraph.R
\name{GIC}
\alias{GIC}
\title{Graph Information Criterion (GIC)}
\usage{
GIC(A, model, p = NULL, bandwidth = "Silverman", eigenvalues = NULL)
}
\arguments{
\item{A}{the adjacency matrix of the graph. For an unweighted graph it
contains only 0s and 1s. For a weighted graph, it may contain nonnegative
real values that correspond to the weights of the edges.}

\item{model}{either a list, a string, a function or a matrix describing a
graph model:

A list that represents the spectral density of a model. It contains the
components "x" and "y", which are numeric vectors of the same size. The "x"
component contains the points at which the density was computed and the "y"
component contains the observed density.

A string that indicates one of the following models: "ER" (Erdos-Renyi random
graph), "GRG" (geometric random graph), "KR" (k regular random graph), "WS"
(Watts-Strogatz model), and "BA" (Barabási-Albert model). When the argument
'model' is a string, the user must also provide the model parameter by using
the argument 'p'.

A function that returns a graph (represented by its adjacency matrix)
generated by a graph model. It must contain two arguments: the first one
correponds to the graph size and the second to the parameter of the model.
The model parameter will be provided by the argument 'p' of the 'GIC'
function.

A matrix containing the spectrum of the model. Each column contains the
eigenvalues of a graph generated by the model. To estimate the spectral
density of the model, the method will estimate the density of the values of
each column, and then will take the average density.}

\item{p}{the model parameter. The user must provide a valid parameter if the
argument 'model' is a string or a function.
For the predefined models ("ER", "GRG", "KR", "WS", and "BA"), the parameter
the probability to connect a pair of vertices, for the "ER" model
(Erdos-Renyi random graph);

the radius used to contruct the geometric graph in a unit square, for the
"GRG" model (geometric random graph);

the degree 'k' of a regular graph, for the "KR" model (k regular random
graph);

the probability to reconnect a vertex, for the "WS" model (Watts-Strogatz
model);

and the scaling exponent, for the "BA" model (Barabási-Albert model).}

\item{bandwidth}{string indicating which criterion will be used to choose
the bandwidth for the spectral density estimation. The available criteria are
"Silverman" (default) and "Sturges".}

\item{eigenvalues}{optional parameter. It contains the eigenvalues of matrix
A. Then, it can be used when the eigenvalues of A were previously computed.
If no value is passed, then the eigenvalues of A will be computed by 'GIC'.}
}
\value{
A real number corresponding to the Kullback-Leibler divergence
between the observed graph and the graph model.
}
\description{
'GIC' returns the Kullback-Leibler divergence between an undirected graph and
a given graph model.
}
\examples{
require(igraph)
A <- as.matrix(get.adjacency(erdos.renyi.game(100, p=0.5)))
# Using a string to indicate the graph model
result1 <- GIC(A, "ER", 0.5)
result1

# Using a function to describe the graph model
# Erdos-Renyi graph
model <- function(n, p) {
   return(as.matrix(get.adjacency(erdos.renyi.game(n, p))))
}
result2 <- GIC(A, model, 0.5)
result2
}
\references{
Takahashi, D. Y., Sato, J. R., Ferreira, C. E. and Fujita A. (2012)
Discriminating Different Classes of Biological Networks by Analyzing the
Graph Spectra  Distribution. _PLoS ONE_, *7*, e49949.
doi:10.1371/journal.pone.0049949.

Silverman, B. W. (1986) _Density Estimation_.  London: Chapman and Hall.

Sturges, H. A. The Choice of a Class Interval. _J. Am. Statist. Assoc._,
*21*, 65-66.
}
\keyword{graph_information_criterion}
