% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/statGraph.R
\name{graph.param.estimator}
\alias{graph.param.estimator}
\title{Graph parameter estimator}
\usage{
graph.param.estimator(A, model, parameters = NULL, eps = 0.01,
  bandwidth = "Silverman", eigenvalues = NULL)
}
\arguments{
\item{A}{the adjacency matrix of the graph. For an unweighted graph it 
contains only 0s and 1s. For a weighted graph, it may contain nonnegative 
real values that correspond to the weights of the edges.}

\item{model}{either a string, a function or an array describing a graph 
model: 

A string that indicates one of the following models: "ER" (Erdos-Renyi random 
graph), "GRG" (geometric random graph), "KR" (k regular random graph), "WS" 
(Watts-Strogatz model), and "BA" (Barabási-Albert model). 

A function that returns a graph (represented by its adjacency matrix) 
generated by a graph model. It must contain two arguments: the first one
correponds to the graph size and the second to the parameter of the model.

A three-dimensional array containing the spectrum of the model. For each 
parameter p that will be considered, the array contains the eigenvalues of N 
graphs that were generated by the model with parameter p. The position 
(i,j,k) contains the j-th eigenvalue of the k-th graph that were generated 
with the i-th parameter. The attribute 'rownames' of the array corresponds to 
the values of the parameters converted to string. To obtain the spectral 
density for each parameter p, the method will estimate the density of the 
eigenvalues of each graph that were generated with parameter p, and then will 
take the average density.}

\item{parameters}{numeric vector containing the values that that will be 
considerated for the parameter estimation. The 'graph.param.estimator' will
return the element of 'parameter' that minimizes the Graph Information 
Criterion (GIC). 
If the user does not provide the argument 'parameters', and 'model' is an 
array, then the values considered for the parameter estimation are the 
rownames converted to a numeric vector. If 'model' is a string, then
default values are used for the predefined models ("ER", "GRG", "KR", "WS", 
and "BA"). The default 'parameter' argument corresponds to a sequence from 

0 to 1 with step 'eps' for the "ER" model (Erdos-Renyi random graph), in 
which the parameter corresponds to the probability to connect a pair of 
vertices;

0 to sqrt(2) with step 'eps' for the "GRG" model (geometric random graph), in 
which the parameter corresponds to the radius used to contruct the geometric 
graph in a unit square;

0 to 'n' with step 'n*eps' for the "KR" model (k regular random graph), in 
which the parameter of the model corresponds to the degree 'k' of a regular 
graph;

0 to 1 with step 'eps' for the "WS" model (Watts-Strogatz model), in which 
the parameter corresponds to the probability to reconnect a vertex;

and 0 to 3 with step 'eps' for the "BA" model (Barabási-Albert model), in  
which the parameter corresponds to the scaling exponent.}

\item{eps}{precision of the grid (default is 0.01).}

\item{bandwidth}{string indicating which criterion will be used to choose the 
bandwidth for the spectral density estimation. The available criteria are 
"Silverman" (default) and "Sturges".}

\item{eigenvalues}{optional parameter. It contains the eigenvalues of matrix
A. Then, it can be used when the eigenvalues of A were previously computed. 
If no value is passed, then the eigenvalues of A will be computed by 
'graph.param.estimator'.}
}
\value{
A list containing:
\item{p}{the parameter estimate. For the "ER", "GRG", "KR", "WS", and "BA"
models, the parameter corresponds to the probability to connect a pair of 
vertices, the radius used to contruct the geometric graph in a unit square,
the degree k of a regular graph, the probability to reconnect a vertex, and 
the scaling exponent, respectively.}  
\item{KL}{the Graph Information Criterion (GIC), i. e. the Kullback-Leibler 
divergence between the observed graph and the graph model with the estimated
parameter.}
}
\description{
'graph.param.estimator' estimates the parameter that best approximates the 
model to the observed graph according to the Graph Information Criterion 
(GIC).
}
\examples{
require(igraph)
A <- as.matrix(get.adjacency(erdos.renyi.game(50, p=0.5)))

# Using a string to indicate the graph model
result1 <- graph.param.estimator(A, "ER", eps=0.25)
result1

# Using a function to describe the graph model
# Erdos-Renyi graph
model <- function(n, p) {
   return(as.matrix(get.adjacency(erdos.renyi.game(n, p))))
}
result2 <- graph.param.estimator(A, model,  seq(0.2, 0.8, 0.1))
result2

}
\references{
Takahashi, D. Y., Sato, J. R., Ferreira, C. E. and Fujita A. (2012) 
Discriminating Different Classes of Biological Networks by Analyzing the 
Graph Spectra  Distribution. _PLoS ONE_, *7*, e49949. 
doi:10.1371/journal.pone.0049949.

Silverman, B. W. (1986) _Density Estimation_.  London: Chapman and Hall.

Sturges, H. A. The Choice of a Class Interval. _J. Am. Statist. Assoc._, 
*21*, 65-66.
}
\keyword{parameter_estimation}

