% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/jointSpectralRadius.R
\name{bound_JSR}
\alias{bound_JSR}
\title{Calculate upper bound for the joint spectral radius of the "companion form AR matrices" of the regimes}
\usage{
bound_JSR(
  stvar,
  epsilon = 0.01,
  adaptive_eps = FALSE,
  ncores = 2,
  print_progress = TRUE
)
}
\arguments{
\item{stvar}{object of class \code{"stvar"}}

\item{epsilon}{a strictly positive real number that approximately defines the goal of length of the interval between the lower
and upper bounds. A smaller epsilon value results in a narrower interval, thus providing better accuracy for the bounds,
but at the cost of increased computational effort. Note that the bounds are always wider than \code{epsilon} and it is not
obvious what \code{epsilon} should be chosen obtain bounds of specific tightness.}

\item{adaptive_eps}{logical: if \code{TRUE}, starts with a large epsilon and then decreases it gradually whenever the progress
of the algorithm requires, until the value given in the argument \code{epsilon} is reached. Usually speeds up the algorithm
substantially but is an unconventional approach, and there is no guarantee that the algorithm converges appropriately towards
bounds with the tightness given by the argument \code{epsilon}.}

\item{ncores}{the number of cores to be used in parallel computing.}

\item{print_progress}{logical: should the progress of the algorithm be printed?}
}
\value{
Returns lower and upper bounds for the joint spectral radius of the "companion form AR matrices" of the regimes.
}
\description{
\code{bound_JSR} calculates an bounds for the joint spectral radius of the
 "companion form AR matrices" matrices of the regimes to assess the validity of the stationarity condition.
}
\details{
A sufficient condition for ergodic stationarity of the STVAR processes implemented in \code{sstvars} is that the joint
 spectral radius of the "companion form AR matrices" of the regimes is smaller than one (Kheifets and Saikkonen, 2020).
 This function calculates an upper (and lower) bound for the JSR and is implemented to assess the validity of this condition
 in practice. If the bound is smaller than one, the model is deemed ergodic stationary.

 Implements the branch-and-bound method by Gripenberg (1996) in the conventional form (\code{adaptive_eps=FALSE}) and in a form
 incorporating "adaptive tightness" (\code{adaptive_eps=FALSE}). The latter approach is unconventional and does not guarantee
 appropriate convergence of the bounds close to the desired tightness given in the argument \code{epsilon}, but it usually
 substantially speeds up the algorithm. When \code{print_progress==TRUE}, the tightest bounds found so-far are printed in each
 iteration of the algorithm, so you can also just terminate the algorithm when the bounds are tight enough for your purposes.
 Consider also adjusting the argument \code{epsilon}, in particular when \code{adaptive_eps=FALSE}, as larger epsilon does not
 just make the bounds less tight but also speeds up the algorithm significantly. See Chang and Blondel (2013) for a discussion
 on variuous methods for bounding the JSR.
}
\examples{
\donttest{
## Below examples take approximately 5 seconds to run.

# Gaussian STVAR p=1, M=2 model with weighted relative stationary densities
# of the regimes as the transition weight function:
theta_122relg <- c(0.734054, 0.225598, 0.705744, 0.187897, 0.259626, -0.000863,
  -0.3124, 0.505251, 0.298483, 0.030096, -0.176925, 0.838898, 0.310863, 0.007512,
  0.018244, 0.949533, -0.016941, 0.121403, 0.573269)
mod122 <- STVAR(data=gdpdef, p=1, M=2, params=theta_122relg)

# Absolute values of the eigenvalues of the "companion form AR matrices":
summary(mod122)$abs_boldA_eigens
# It is a necessary (but not sufficient!) condition for ergodic stationary that
# the spectral radius of the "companion form AR matrices" are smaller than one
# for all of the regimes. A sufficient (but not necessary) condition for
# ergodic stationary is that the joint spectral radius of the companion form
# AR matrices" of the regimes is smaller than one. Therefore, we calculate
# bounds for the joint spectral radius.

## Bounds by Gripenberg's (1996) branch-and-bound method:
# Since the largest modulus of the companion form AR matrices is not very close
# to one, we likely won't need very thight bounds to verify the JSR is smaller
# than one. Thus, using a small epsilon would make the algorithm unnecessarily slow,
# so we use the (still quite small) epsilon=0.01:
bound_JSR(mod122, epsilon=0.01, adaptive_eps=FALSE)
# The upper bound is smaller than one, so the model is ergodic stationary.

# If we want tighter bounds, we can set smaller epsilon, e.g., epsilon=0.001:
bound_JSR(mod122, epsilon=0.001, adaptive_eps=FALSE)

# Using adaptive_eps=TRUE usually speeds up the algorithm when the model
# is large, but with the small model here, the speed-difference is small:
bound_JSR(mod122, epsilon=0.001, adaptive_eps=TRUE)
}
}
\references{
\itemize{
 \item C-T Chang and V.D. Blondel. 2013 . An experimental study of approximation algorithms for the joint spectral radius.
     \emph{Numerical algorithms}, \strong{64}, 181-202.
   \item Gripenberg, G. 1996. Computing the joint spectral radius. \emph{Linear Algebra and its Applications},
     234, 43–60.
   \item I.L. Kheifets, P.J. Saikkonen. 2020. Stationarity and ergodicity of Vector STAR models.
     \emph{Econometric Reviews}, \strong{39}:4, 407-414.
   \item Virolainen S. 2024. Identification by non-Gaussianity in structural threshold and
      smooth transition vector autoregressive models. Unpublished working
      paper, available in ArXiv.
 }
}
\seealso{
\code{\link{bound_jsr_G}}
}
