% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/sample_agecomp.R
\name{sample_agecomp}
\alias{sample_agecomp}
\title{Sample age compositions from expected values}
\usage{
sample_agecomp(dat_list, outfile, fleets = c(1, 2), Nsamp, years, cpar = 1,
  ESS = NULL, write_file = TRUE, keep_conditional = TRUE)
}
\arguments{
\item{dat_list}{An SS data list object as read in from
\code{\link[r4ss]{SS_readdat}} in the \pkg{r4ss} package. Make sure
you select option \code{section=2}.}

\item{outfile}{A character string of the new file name to be created.
Must end in \code{.dat} or equal \code{wtatage.ss}.}

\item{fleets}{*A numeric vector giving the fleets to be used. This order also
pertains to other arguments. A missing value excludes that fleet from
\code{outfile} (i.e. it turns it off so no samples are written). If none of
the fleet collected samples, keep the value to \code{fleets=NULL}.}

\item{Nsamp}{*A numeric list of the same length as \code{fleets}.
Either single values or vectors of the same length as the number of
years can be passed through. Single values are repeated for all
years. If no fleet collected samples, keep the value to
\code{Nsamp=NULL}.}

\item{years}{*A list the same length as \code{fleets} giving the years as
numeric vectors. If no fleet collected samples, keep the value to
\code{years=NULL}.}

\item{cpar}{*A numeric value or vector the same length as
\code{fleets} controlling the variance of the Dirichlet
distribution used for sampling. A value of \code{1} indicates the
same standard deviation as a multinomial of the given \code{Nsamp},
\code{2} indicates twice, etc. Values greater than one indicate
overdispersion, and less underdispersion.}

\item{ESS}{An effective sample size to write to file (but not use in
sampling). The default of NULL means to use the true (internally
calculated) ESS, which is \code{Nsamp} for the multinomial case or given
by the formula under \code{cpar} for the Dirichlet case. Must match the
structure of the \code{years} arguments.}

\item{write_file}{A logical switch for whether to write \code{outfile} to
disk. Can be turned off to speed up testing or exploration of the
function. The new data are returned invisibly, as in the examples
below.}

\item{keep_conditional}{A logical if conditional age-at-length data
should be kept or removed entirely from the \code{.dat} file.
\code{sample_agecomp} only works on the age composition data
and not on the conditional age-at-length data. To sample the
conditional data set \code{keep_conditional} to \code{TRUE}
and use \code{\link{sample_calcomp}}.}
}
\value{
A modified \code{.dat} file if \code{write_file=TRUE}. A list object
containing the modified \code{.dat} file is returned invisibly.
}
\description{
Take a \code{data.SS_new} file containing expected values and sample to
create observed age compositions which are then written to file for use by
the estimation model.
If used with \code{\link{run_ss3sim}} the case file should be named
\code{agecomp}. A suggested (default) case letter is \code{D} for data.
}
\section{Which arguments to specifiy in case files}{

All function argument descriptions that start with an asterisk (*) will be passed
through the case files to \code{\link{run_ss3sim}}. If one of these arguments
is not specified in a case file, then a value of \code{NULL} will be passed,
which may or may not be an appropriate value. Other arguments will be ignored
if specified.
}

\examples{
d <- system.file("extdata", package = "ss3sim")
f_in <- paste0(d, "/models/cod-om/codOM.dat")
dat_list <- r4ss::SS_readdat(f_in, verbose = FALSE)
dat_list <- change_fltname(dat_list)

## Turn off age comps by specifying fleets=NULL
sample_agecomp(dat_list=dat_list, outfile="test1.dat",
               fleets=NULL, cpar=c(5,NA), Nsamp=list(100,100),
               years=list(1995, 1995), write_file=FALSE)

## Generate with a smaller number of fleet taking samples
ex1 <- sample_agecomp(dat_list=dat_list, outfile="test1.dat", fleets=c(2),
                      Nsamp=list(c(10,50)), years=list(c(26,27)),
                      write_file=FALSE)

## Generate with varying Nsamp by year for first fleet
ex2 <- sample_agecomp(dat_list=dat_list, outfile="test2.dat", fleets=c(1,2),
                      Nsamp=list(c(rep(50, 5), rep(100, 5)), 50),
                      years=list(seq(26, 44, 2),
                          c(26:100)), write_file=FALSE)

## Run three  cases showing Multinomial, Dirichlet(1) and over-dispersed
## Dirichlet for different levels of sample sizes
op <- par(mfrow = c(1,3))
set.seed(1)
for(samplesize in c(30, 100, 1000)){
    ex4 <- sample_agecomp(dat_list=dat_list, outfile="test4.dat", fleets=c(1,2),
                          Nsamp=list(samplesize, samplesize),
                          write_file = FALSE,
                          years=list(50,50), cpar=c(NA, 1))
    ex5 <- sample_agecomp(dat_list=dat_list, outfile="test5.dat", fleets=c(1,2),
                          Nsamp=list(samplesize, samplesize),
                          write_file = FALSE,
                          years=list(50,50), cpar=c(1, 1))
    ex6 <- sample_agecomp(dat_list=dat_list, outfile="test6.dat", fleets=c(1,2),
                          Nsamp=list(samplesize, samplesize),
                          write_file = FALSE,
                          years=list(50,50), cpar=c(5, 1))
    true <- subset(dat_list$agecomp, FltSvy==1 & Yr == 50)[-(1:9)]
    true <- true/sum(true)
    mult <- subset(ex4$agecomp, FltSvy==1)[1,-(1:9)]
    mult <- mult/sum(mult)
    plot(1:15, mult, type="b", ylim=c(0,1),
         col=1, xlab="Age", ylab="Proportion", main=paste("Sample size=",
         samplesize))
    legend("topright", legend=c("Multinomial", "Dirichlet(1)",
                                "Dirichlet(5)", "Truth"),
           lty=1, col=1:4)
    lines((1:15), subset(ex5$agecomp, FltSvy==1)[1,-(1:9)], type="b", col=2)
    lines((1:15), subset(ex6$agecomp, FltSvy==1)[1,-(1:9)], type="b", col=3)
    lines((1:15), true, col=4, lwd=2)
}
par(op)
}
\seealso{
Other sampling functions: \code{\link{clean_data}},
  \code{\link{sample_calcomp}}, \code{\link{sample_index}},
  \code{\link{sample_lcomp}}, \code{\link{sample_mlacomp}},
  \code{\link{sample_wtatage}}
}
\author{
Cole Monnahan and Kotaro Ono; modified from a version by Roberto
Licandeo and Felipe Hurtado-Ferro
}
