% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/mSpline.R
\name{mSpline}
\alias{mSpline}
\title{M-Spline Basis for Polynomial Splines}
\usage{
mSpline(
  x,
  df = NULL,
  knots = NULL,
  degree = 3L,
  intercept = FALSE,
  Boundary.knots = NULL,
  derivs = 0L,
  ...
)
}
\arguments{
\item{x}{The predictor variable.  Missing values are allowed and will be
returned as they are.}

\item{df}{Degree of freedom that equals to the column number of returned
matrix.  One can specify \code{df} rather than \code{knots}, then the
function chooses \code{df - degree - as.integer(intercept)} internal
knots at suitable quantiles of \code{x} ignoring missing values and
those \code{x} outside of the boundary.  If internal knots are specified
via \code{knots}, the specified \code{df} will be ignored.}

\item{knots}{The internal breakpoints that define the spline.  The default
is \code{NULL}, which results in a basis for ordinary polynomial
regression.  Typical values are the mean or median for one knot,
quantiles for more knots.}

\item{degree}{A non-negative integer specifying the degree of the piecewise
polynomial. The default value is 3 for cubic splines. Zero degree is
allowed for piece-wise constant bases.}

\item{intercept}{If \code{TRUE}, the complete basis matrix will be returned.
Otherwise, the first basis will be excluded from the output.}

\item{Boundary.knots}{Boundary points at which to anchor the spline basis.
By default, they are the range of the non-\code{NA} data.  If both
\code{knots} and \code{Boundary.knots} are supplied, the basis
parameters do not depend on \code{x}. Data can extend beyond
\code{Boundary.knots}.}

\item{derivs}{A non-negative integer specifying the order of derivatives of
M-splines. The default value is \code{0L} for M-spline bases.}

\item{...}{Optional arguments that are not used.}
}
\value{
A numeric matrix with \code{length(x)} rows and \code{df} columns if
    \code{df} is specified or \code{length(knots) + degree +
    as.integer(intercept)} columns if \code{knots} are specified instead.
    Attributes that correspond to the arguments specified are returned for
    usage of other functions in this package.
}
\description{
Generates the basis matrix of the regression spline called M-spline or the
corresponding derivatives of given order.  For monotone regression,
\code{\link{iSpline}} should be used instead of M-splines.
}
\details{
It is an implementation of the close form M-spline basis based on the
recursion formula given by Ramsay (1988).
}
\examples{
library(splines2)

## Example given in the reference paper by Ramsay (1988)
x <- seq.int(0, 1, 0.01)
knots <- c(0.3, 0.5, 0.6)
msMat <- mSpline(x, knots = knots, degree = 2, intercept = TRUE)

par(mar = c(2.5, 2.5, 0.2, 0.1), mgp = c(1.5, 0.5, 0))
matplot(x, msMat, type = "l", ylab = "y")
abline(v = knots, lty = 2, col = "gray")

## derivatives of M-splines
dmsMat <- mSpline(x, knots = knots, degree = 2,
                  intercept = TRUE, derivs = 1)

## or using the deriv method
dmsMat1 <- deriv(msMat)
stopifnot(all.equal(dmsMat, dmsMat1, check.attributes = FALSE))
}
\references{
Ramsay, J. O. (1988). Monotone regression splines in action.
\emph{Statistical science}, 3(4), 425--441.
}
\seealso{
\code{\link{bSpline}} for B-splines;
\code{\link{iSpline}} for I-splines;
\code{\link{cSpline}} for C-splines.
}
