# Author: Darren Colby
# Date: 9/6/2021
# Purpose: Provide a function to compare two simulated networks


#' Compare statistics from two simulated networks
#'
#' @description Generates summary statistics for two simulated networks of class
#' NetSim.
#'
#' @usage compare_networks(net1, net2)
#'
#' @details This function enables a quick comparison of two simulated networks
#' of class NetSim. This allows the user to see how different a network
#' simulated from a Halton sequence is from network simulated from a Poisson
#' point process with the same number of points in the same geographical window.
#' Similarly, it enables comparison of networks generated from different
#' spatial interaction functions. The density, mean degree, mean closeness, mean
#' betweenness, and the size of the largest connected component are computed for
#' each network.
#'
#' @param net1 a NetSim object
#' @param net2 a NetSim object
#'
#' @return A dataframe with two rows and five columns. The first row displays
#' statistics from the first network and the second row displays statistics from
#' the second network. The values in each column are as follows. Density: the
#' density of the network. Mean Degree: the average degree of all nodes in the
#' network. Mean Closeness: the average closeness centrality of all nodes in
#' each network. Mean Betweenness: the average betweenness centrality for the
#' nodes in a network. Largest Component Size: The number of nodes in the
#' largest connected component of a network.
#'
#' @examples
#' # load data
#' data("RI")
#'
#' # Simulate point processes
#' points1 <- PointSim(10, RI)
#' points2 <- PointSim(20, RI)
#'
#' # Create two networks
#' net1 <- NetSim(points1)
#' net2 <- NetSim(points2)
#'
#' compare_networks(net1, net2)
#'
#' @author Darren Colby \cr
#' Email: dscolby17@@gmail.com
#' @export
compare_networks <- function(net1, net2) {

    stopifnot(methods::is(net1, c("NetSim", "igraph"))&
                  methods::is(net2, c("NetSim", "igraph")))

    # Compute basic network statistics
    density <- c(igraph::edge_density(net1), igraph::edge_density(net1))
    degree <- c(mean(igraph::degree(net1)), mean(igraph::degree(net2)))
    closeness <- c(mean(igraph::closeness(net1)), mean(igraph::closeness(net2)))
    betweenness <- c(mean(igraph::betweenness(net1)),
                     mean(igraph::betweenness(net2)))
    component <- c(max(igraph::clusters(net1)$csize),
                   max(igraph::clusters(net2)$csize))

    # Coerce into a dataframe
    df <- data.frame(density, degree, closeness, betweenness, component)

    colnames(df) <- c("Density", "Mean Degree", "Mean Closeness",
                      "Mean Betweenness", "Largest Component Size")

    rownames(df) <- c(deparse(substitute(net1)), deparse(substitute(net2)))

    print(df)

    return(df)

}

