% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/spLMstack.R
\name{spLMstack}
\alias{spLMstack}
\title{Bayesian spatial linear model using predictive stacking}
\usage{
spLMstack(
  formula,
  data = parent.frame(),
  coords,
  cor.fn,
  priors,
  params.list,
  n.samples,
  loopd.method,
  parallel = FALSE,
  solver = "ECOS",
  verbose = TRUE,
  ...
)
}
\arguments{
\item{formula}{a symbolic description of the regression model to be fit.
See example below.}

\item{data}{an optional data frame containing the variables in the model.
If not found in \code{data}, the variables are taken from
\code{environment(formula)}, typically the environment from which
\code{spLMstack} is called.}

\item{coords}{an \eqn{n \times 2}{n x 2} matrix of the observation
coordinates in \eqn{\mathbb{R}^2} (e.g., easting and northing).}

\item{cor.fn}{a quoted keyword that specifies the correlation function used
to model the spatial dependence structure among the observations. Supported
covariance model key words are: \code{'exponential'} and \code{'matern'}.
See below for details.}

\item{priors}{a list with each tag corresponding to a parameter name and
containing prior details. If not supplied, uses defaults.}

\item{params.list}{a list containing candidate values of spatial process
parameters for the \code{cor.fn} used, and, noise-to-spatial variance ratio.}

\item{n.samples}{number of posterior samples to be generated.}

\item{loopd.method}{character. Valid inputs are \code{'exact'} and \code{'PSIS'}. The
option \code{'exact'} corresponds to exact leave-one-out predictive densities.
The option \code{'PSIS'} is faster, as it finds approximate leave-one-out
predictive densities using Pareto-smoothed importance sampling
(Gelman \emph{et al.} 2024).}

\item{parallel}{logical. If \code{parallel=FALSE}, the parallelization plan,
if set up by the user, is ignored. If \code{parallel=TRUE}, the function
inherits the parallelization plan that is set by the user via the function
\code{\link[future:plan]{future::plan()}} only. Depending on the parallel backend available, users
may choose their own plan. More details are available at
\url{https://cran.R-project.org/package=future}.}

\item{solver}{(optional) Specifies the name of the solver that will be used
to obtain optimal stacking weights for each candidate model. Default is
\code{"ECOS"}. Users can use other solvers supported by the
\link[CVXR]{CVXR-package} package.}

\item{verbose}{logical. If \code{TRUE}, prints model-specific optimal
stacking weights.}

\item{...}{currently no additional argument.}
}
\value{
An object of class \code{spLMstack}, which is a list including the
following tags -
\describe{
\item{\code{samples}}{a list of length equal to total number of candidate models
with each entry corresponding to a list of length 3, containing posterior
samples of fixed effects (\code{beta}), variance parameter
(\code{sigmaSq}), spatial effects (\code{z}) for that model.}
\item{\code{loopd}}{a list of length equal to total number of candidate models with
each entry containing leave-one-out predictive densities under that
particular model.}
\item{\code{n.models}}{number of candidate models that are fit.}
\item{\code{candidate.models}}{a matrix with \code{n_model} rows with each row
containing details of the model parameters and its optimal weight.}
\item{\code{stacking.weights}}{a numeric vector of length equal to the number of
candidate models storing the optimal stacking weights.}
\item{\code{run.time}}{a \code{proc_time} object with runtime details.}
\item{\code{solver.status}}{solver status as returned by the optimization
routine.}
}
The return object might include additional data that is useful for subsequent
prediction, model fit evaluation and other utilities.
}
\description{
Fits Bayesian spatial linear model on a collection of candidate
models constructed based on some candidate values of some model parameters
specified by the user and subsequently combines inference by stacking
predictive densities. See Zhang, Tang and Banerjee (2024) for more details.
}
\details{
Instead of assigning a prior on the process parameters \eqn{\phi}
and \eqn{\nu}, noise-to-spatial variance ratio \eqn{\delta^2}, we consider
a set of candidate models based on some candidate values of these parameters
supplied by the user. Suppose the set of candidate models is
\eqn{\mathcal{M} = \{M_1, \ldots, M_G\}}. Then for each
\eqn{g = 1, \ldots, G}, we sample from the posterior distribution
\eqn{p(\sigma^2, \beta, z \mid y, M_g)} under the model \eqn{M_g} and find
leave-one-out predictive densities \eqn{p(y_i \mid y_{-i}, M_g)}. Then we
solve the optimization problem
\deqn{
 \begin{aligned}
 \max_{w_1, \ldots, w_G}& \, \frac{1}{n} \sum_{i = 1}^n \log \sum_{g = 1}^G
 w_g p(y_i \mid y_{-i}, M_g) \\
 \text{subject to} & \quad w_g \geq 0, \sum_{g = 1}^G w_g = 1
 \end{aligned}
 }
to find the optimal stacking weights \eqn{\hat{w}_1, \ldots, \hat{w}_G}.
}
\examples{
# load data and work with first 100 rows
data(simGaussian)
dat <- simGaussian[1:100, ]

# setup prior list
muBeta <- c(0, 0)
VBeta <- cbind(c(1.0, 0.0), c(0.0, 1.0))
sigmaSqIGa <- 2
sigmaSqIGb <- 2
prior_list <- list(beta.norm = list(muBeta, VBeta),
                   sigma.sq.ig = c(sigmaSqIGa, sigmaSqIGb))

mod1 <- spLMstack(y ~ x1, data = dat,
                  coords = as.matrix(dat[, c("s1", "s2")]),
                  cor.fn = "matern",
                  priors = prior_list,
                  params.list = list(phi = c(1.5, 3),
                                     nu = c(0.5, 1),
                                     noise_sp_ratio = c(1)),
                  n.samples = 1000, loopd.method = "exact",
                  parallel = FALSE, solver = "ECOS", verbose = TRUE)

post_samps <- stackedSampler(mod1)
post_beta <- post_samps$beta
print(t(apply(post_beta, 1, function(x) quantile(x, c(0.025, 0.5, 0.975)))))

post_z <- post_samps$z
post_z_summ <- t(apply(post_z, 1,
                       function(x) quantile(x, c(0.025, 0.5, 0.975))))

z_combn <- data.frame(z = dat$z_true,
                      zL = post_z_summ[, 1],
                      zM = post_z_summ[, 2],
                      zU = post_z_summ[, 3])

library(ggplot2)
plot1 <- ggplot(data = z_combn, aes(x = z)) +
  geom_point(aes(y = zM), size = 0.25,
             color = "darkblue", alpha = 0.5) +
  geom_errorbar(aes(ymin = zL, ymax = zU),
                width = 0.05, alpha = 0.15) +
  geom_abline(slope = 1, intercept = 0,
              color = "red", linetype = "solid") +
  xlab("True z") + ylab("Stacked posterior of z") +
  theme_bw() +
  theme(panel.background = element_blank(),
        aspect.ratio = 1)
}
\references{
Vehtari A, Simpson D, Gelman A, Yao Y, Gabry J (2024). "Pareto
Smoothed Importance Sampling." \emph{Journal of Machine Learning Research},
\strong{25}(72), 1-58. URL \url{https://jmlr.org/papers/v25/19-556.html}.

Zhang L, Tang W, Banerjee S (2024). "Bayesian Geostatistics Using
Predictive Stacking." \cr
\doi{10.48550/arXiv.2304.12414}.
}
\seealso{
\code{\link[=spLMexact]{spLMexact()}}, \code{\link[=spGLMstack]{spGLMstack()}}
}
\author{
Soumyakanti Pan \href{mailto:span18@ucla.edu}{span18@ucla.edu},\cr
Sudipto Banerjee \href{mailto:sudipto@ucla.edu}{sudipto@ucla.edu}
}
