% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/sourceSlurm.R
\name{sourceSlurm}
\alias{sourceSlurm}
\alias{slurmr_cmd}
\title{Source an R script as a Slurm job}
\usage{
sourceSlurm(
  file,
  job_name = NULL,
  tmp_path = opts_slurmR$get_tmp_path(),
  rscript_opt = list(vanilla = TRUE),
  plan = "submit",
  ...
)

slurmr_cmd(
  cmd_path,
  cmd_name = "slurmr",
  add_alias = TRUE,
  bashrc_path = "~/.bashrc"
)
}
\arguments{
\item{file}{Character. Path to the R script to source using Slurm.}

\item{job_name}{Character. Name of the job to be passed to \code{Slurm}.}

\item{tmp_path}{Character. Path to the directory where all the data (including
scripts) will be stored. Notice that this path must be accessible by all the
nodes in the network (See \link{opts_slurmR}).}

\item{rscript_opt}{List. Options to be passed to \code{Rscript}.}

\item{plan}{A character scalar. (See \link{the_plan}).}

\item{...}{Further options passed to \link{sbatch}.}

\item{cmd_path}{Character scalar. Path (directory) where to put the command function.
This is usually your home directory.}

\item{cmd_name}{Character scalar. Name of the command (of the file).}

\item{add_alias, bashrc_path}{Logical scalar and character scalar. When
\code{add_alias=TRUE} it will modify (or create, if non-existent) the \code{.bashrc}
file to add an alias of the same name of \code{cmd_name}. The path to \code{.bashrc} can be
specified via the \code{bashrc_path} option.}
}
\value{
In the case of \code{sourceSlurm}, Whatever \link{sbatch} returns.

The function \code{slurmr_cmd} returns \code{invisible()}.
}
\description{
This function sources R scripts using Slurm by creating a batch script file
and submitting it via \link{sbatch}.
}
\details{
\code{sourceSlurm} checks for flags that may be included in the Slurm job file. If
the R script starts with \verb{#!/bin/} or similar, then \verb{#SBATCH} flags will be
read from the R script and added to the Slurm job file.

The function \code{slurmr_cmd} writes a simple command that works as a wrapper
of \code{sourceSlurm}. In particular, from command line, if the user wants to source an
R script using \code{sourceSlurm}, we can either:

\if{html}{\out{<div class="sourceCode">}}\preformatted{$ Rscript -e "slurmR::sourceSlurm('path/to/the/script.R', plan = 'submit')"
}\if{html}{\out{</div>}}

Or, after calling \code{slurmr_cmd} from within R, do the following instead

\if{html}{\out{<div class="sourceCode">}}\preformatted{$ ./slurmr path/to/the/script.R
}\if{html}{\out{</div>}}

And, if you used the option \code{add_alias = TRUE}, then, after restarting bash,
you can run R scripts with Slurm as follows:

\if{html}{\out{<div class="sourceCode">}}\preformatted{$ slurmr path/to/the/script.R
}\if{html}{\out{</div>}}

The main side effect of this function is that it creates a file named \code{cmd_name}
in the directory specified by \code{cmd_path}, and, if \code{add_alias = TRUE}. it will
create (if not found) or modify (if found) the \code{.bashrc} file adding a line
with an alias. For more information on \code{.bashrc} see \href{https://superuser.com/questions/49289}{here}.
}
\examples{
# In this example we will be sourcing an R script that also has #SBATCH
# flags. Here are the contents
file <- system.file("example.R", package="slurmR")

cat(readLines(file), sep="\n")
# #!/bin/sh
# #SBATCH --account=lc_ggv
# #SBATCH --time=01:00:00
# #SBATCH --mem-per-cpu=4G
# #SBATCH --job-name=Waiting
# Sys.sleep(10)
# message("done.")

# We can directly submit this R script as a job by calling `sourceSlurm`.
# (of course you need Slurm to do this!)
\dontrun{
sourceSlurm(file)
}

# The function will create a bash script that is used later to be submitted to
# the queue using `sbatch`. The resulting file looks something like this
# #!/bin/sh
# #SBATCH --job-name=Waiting
# #SBATCH --output=/home/vegayon/Documents/slurmR/Waiting.out
# #SBATCH --account=lc_ggv
# #SBATCH --time=01:00:00
# #SBATCH --mem-per-cpu=4G
# /usr/lib/R/bin/Rscript --vanilla /usr/local/lib/R/site-library/slurmR/example.R
}
