% Generated by roxygen2 (4.1.1): do not edit by hand
% Please edit documentation in R/sjPlotLinreg.R
\name{sjp.lm}
\alias{sjp.lm}
\title{Plot estimates or predicted values of linear models}
\usage{
sjp.lm(fit, type = "lm", sort.est = TRUE, title = NULL,
  axisLabels.x = NULL, axisLabels.y = NULL, showAxisLabels.y = TRUE,
  axisTitle.x = "Estimates", axisLimits = NULL, geom.colors = "Set1",
  geom.size = NULL, interceptLineType = 2, interceptLineColor = "grey70",
  remove.estimates = NULL, breakTitleAt = 50, breakLabelsAt = 25,
  gridBreaksAt = NULL, coord.flip = TRUE, showValueLabels = TRUE,
  labelDigits = 2, showPValueLabels = TRUE, showModelSummary = FALSE,
  showCI = TRUE, pointAlpha = 0.2, showScatterPlot = TRUE,
  showLoess = FALSE, showLoessCI = FALSE, poly.term = NULL,
  showOriginalModelOnly = TRUE, completeDiagnostic = FALSE,
  printPlot = TRUE)
}
\arguments{
\item{fit}{fitted linear regression model (\code{\link{lm}}- or \code{plm}-object).}

\item{type}{type of plot. Use one of following:
\describe{
  \item{\code{"lm"}}{(default) for forest-plot like plot of estimates. If the fitted model only contains one predictor, intercept and slope are plotted.}
  \item{\code{"std"}}{for forest-plot like plot of standardized beta values. If the fitted model only contains one predictor, intercept and slope are plotted.}
  \item{\code{"std2"}}{for forest-plot like plot of standardized beta values, however, standardization is done by dividing by two sd (see 'Details'). If the fitted model only contains one predictor, intercept and slope are plotted.}
  \item{\code{"pred"}}{to plot regression lines for each single predictor of the fitted model, against the response (linear relationship between each model term and response).}
  \item{\code{"resid"}}{to plot regression lines for each single predictor of the fitted model, against the residuals (linear relationship between each model term and residuals). May be used for model diagnostics (see \url{https://www.otexts.org/fpp/5/4}).}
  \item{\code{"resp"}}{to plot predicted values for the response. Use \code{showCI} argument to plot standard errors as well.}
  \item{\code{"eff"}}{to plot marginal effects of all terms in \code{fit}. Note that interaction terms are excluded from this plot; use \code{\link{sjp.int}} to plot effects of interaction terms.}
  \item{\code{"poly"}}{to plot predicted values (marginal effects) of polynomial terms in \code{fit}. Use \code{poly.term} to specify the polynomial term in the fitted model (see 'Examples').}
  \item{\code{"ma"}}{to check model assumptions. Note that only three arguments are relevant for this option \code{fit}, \code{completeDiagnostic} and \code{showOriginalModelOnly}. All other arguments are ignored.}
  \item{\code{"vif"}}{to plot Variance Inflation Factors.}
}}

\item{sort.est}{logical, determines whether estimates should be sorted according to their values.}

\item{title}{plot title as string. Example: \code{title = "my title"}.
Use \code{NULL} to automatically detect variable names that will be used as title
(see \code{\link[sjmisc]{set_label}}) for details). If \code{title = ""},
no title is printed.}

\item{axisLabels.x}{name of predictor (independent variable) as string. Two things to consider:
\itemize{
  \item Only used if fitted model has only one predictor and \code{type = "lm"}.
  \item If you use the \code{\link[sjmisc]{read_spss}} and \code{\link[sjmisc]{get_label}} functions, you receive a character vector with variable label strings. You can use it like this: \code{axisLabels.x = get_label(efc)['quol_5']}
}}

\item{axisLabels.y}{labels or names of the predictor variables (independent vars). Must
be a character vector of same length as independent variables. The labels
may also be passed as list object; they will be coerced to character vector automatically.}

\item{showAxisLabels.y}{logical, whether labels of independent variables should be shown or not.}

\item{axisTitle.x}{title for the x-axis. Default is \code{"Estimates"}.}

\item{axisLimits}{numeric vector of length 2, defining the range of the plot axis.
By default, the limits range from the lowest confidence interval to the
highest, so plot has maximum zoom.}

\item{geom.colors}{user defined color palette for geoms. Must either be vector with two color values
or a specific color palette code. See 'Note' in \code{\link{sjp.grpfrq}}.}

\item{geom.size}{size resp. width of the geoms (bar width or point size, depending on \code{type} argument).}

\item{interceptLineType}{linetype of the intercept line (zero point). Default is \code{2} (dashed line).}

\item{interceptLineColor}{color of the intercept line. Default value is \code{"grey70"}.}

\item{remove.estimates}{character vector with coefficient names that indicate
which estimates should be removed from the plot.
\code{remove.estimates = "est_name"} would remove the estimate \emph{est_name}. Default
is \code{NULL}, i.e. all estimates are printed.}

\item{breakTitleAt}{determines how many chars of the plot title are displayed in
one line and when a line break is inserted into the title.}

\item{breakLabelsAt}{determines how many chars of the category labels are displayed in
one line and when a line break is inserted.}

\item{gridBreaksAt}{set breaks for the axis, i.e. at every \code{gridBreaksAt}'th
position a major grid is being printed.}

\item{coord.flip}{logical, if \code{TRUE} (default), predictors are plotted along the y-axis and estimate
values are plotted on the x-axis.}

\item{showValueLabels}{logical, whether value labels should be plotted to each dot or not.}

\item{labelDigits}{amount of digits for rounding the estimates (see \code{showValueLabels}).
Default is 2, i.e. estimates have 2 digits after decimal point.}

\item{showPValueLabels}{logical, whether the significance level of each coefficient
should be appended to values or not.}

\item{showModelSummary}{logical, if \code{TRUE}, a summary of the regression model with
Intercept, R-squared, F-Test and AIC-value is printed to the lower right corner
of the plot.}

\item{showCI}{logical, if \code{TRUE} (default), a confidence region for the regression line
will be plotted. Only applies if \code{type = "lm"} and fitted model has
only one predictor, or if \code{type = "pred"} or \code{type = "resid"}.}

\item{pointAlpha}{alpha value of point-geoms in the scatter plots.
Default is 0.2.}

\item{showScatterPlot}{logical, if \code{TRUE} (default), a scatter plot of
response and predictor values for each predictor of \code{fit} is plotted.
Only applies if \code{type = "lm"} and fitted model has only one predictor,
or if \code{type = "pred"} or \code{type = "resid"}.}

\item{showLoess}{logical, if \code{TRUE}, an additional loess-smoothed line is plotted.
For \code{\link{sjp.lmer}}, only applies \code{type = "fe.pred"} or
\code{type = "fe.resid"}.}

\item{showLoessCI}{logical, if \code{TRUE}, a confidence region for the loess-smoothed line
will be plotted. Default is \code{FALSE}. Only applies, if \code{showLoess = TRUE}
(and for \code{\link{sjp.lmer}}, only applies if \code{type = "fe.pred"}
or \code{type = "fe.resid"}).}

\item{poly.term}{name of a polynomial term in \code{fit} as string. Needs to be
         specified, if \code{type = "poly"}, in order to plot marginal effects
         for polynomial terms. See 'Examples'.}

\item{showOriginalModelOnly}{logical, if \code{TRUE} (default), only model assumptions of
\code{fit} are plotted. if \code{FALSE}, model assumptions of an updated
model where outliers are automatically excluded are also plotted.
Only applies if \code{type = "ma"}.}

\item{completeDiagnostic}{logical, if \code{TRUE}, additional tests are performed. Default is \code{FALSE}
         Only applies if \code{type = "ma"}.}

\item{printPlot}{logical, if \code{TRUE} (default), plots the results as graph. Use \code{FALSE} if you don't
want to plot any graphs. In either case, the ggplot-object will be returned as value.}
}
\value{
Depending on the \code{type}, in most cases (insisibily)
          returns the ggplot-object with the complete plot (\code{plot})
          as well as the data frame that was used for setting up the
          ggplot-object (\code{df}). For \code{type = "ma"}, an updated model
          with removed outliers is returned.
}
\description{
Depending on the \code{type}, this function plots coefficients (estimates)
               of linear regressions (including panel models fitted with the \code{plm}-function
               from the \pkg{plm}-package) with confidence intervals as dot plot (forest plot),
               model assumptions for linear models or slopes and scatter plots for each single
               coefficient. See \code{type} for details.
}
\details{
\describe{
           \item{\code{type = "lm"}}{if fitted model only has one predictor, no forest plot is shown. Instead, a regression line with confidence interval (in blue) is plotted by default, and a loess-smoothed line without confidence interval (in red) can be added if argument \code{showLoess} is \code{TRUE}.}
           \item{\code{type = "std2"}}{plots standardized beta values, however, standardization follows \href{http://www.stat.columbia.edu/~gelman/research/published/standardizing7.pdf}{Gelman's (2008)} suggestion, rescaling the estimates by dividing them by two standard deviations instead of just one. Resulting coefficients are then directly comparable for untransformed binary predictors. This standardization uses the \code{\link[arm]{standardize}}-function.}
           \item{\code{type = "pred"}}{regression lines (slopes) with confidence intervals for each single predictor of the fitted model are plotted, i.e. all predictors of the fitted model are extracted and for each of them, the linear relationship is plotted against the response variable.}
           \item{\code{type = "resid"}}{is similar to the \code{type = "pred"} option, however, each predictor is plotted against the residuals (instead of response).}
           \item{\code{type = "resp"}}{the predicted values of the response for each observation is plotted, which mostly results in a single linear line.}
           \item{\code{type = "eff"}}{computes the marginal effects for all predictors, using the \code{\link[effects]{allEffects}} function. I.e. for each predictor, the predicted values towards the response are plotted, with all remaining co-variates set to the mean. Due to possible different scales of predictors, a facted plot is printed (instead of plotting all lines in one plot). This function accepts following argument: \code{fit}, \code{title}, \code{geom.size}, \code{showCI} and \code{printPlot}.}
           \item{\code{type = "poly"}}{plots the marginal effects of polynomial terms in \code{fit}, using the \code{\link[effects]{effect}} function, but only for a selected polynomial term, which is specified with \code{poly.term}. This function helps undertanding the effect of polynomial terms by plotting the curvilinear relationships of response and quadratic, cubic etc. terms. This function accepts following argument: \code{fit}, \code{poly.term}, \code{geom.colors}, \code{geom.size}, \code{axisTitle.x}, \code{showCI} and \code{printPlot}.}
           \item{\code{type = "ma"}}{checks model assumptions. Please note that only three arguments are relevant: \code{fit}, \code{completeDiagnostic} and \code{showOriginalModelOnly}. All other arguments are ignored.}
           \item{\code{type = "vif"}}{Variance Inflation Factors (check for multicollinearity) are plotted. As a rule of thumb, values below 5 are considered as good and indicate no multicollinearity, values between 5 and 10 may be tolerable. Values greater than 10 are not acceptable and indicate multicollinearity between model's predictors.}
           }
}
\examples{
# --------------------------------------------------
# plotting estimates of linear models as forest plot
# --------------------------------------------------
# fit linear model
fit <- lm(airquality$Ozone ~ airquality$Wind + airquality$Temp + airquality$Solar.R)

# plot estimates with CI
sjp.lm(fit, gridBreaksAt = 2)

# plot estimates with CI
# and with narrower tick marks
# (because "gridBreaksAt" was not specified)
sjp.lm(fit)

# ---------------------------------------------------
# plotting regression line of linear model (done
# automatically if fitted model has only 1 predictor)
# ---------------------------------------------------
library(sjmisc)
data(efc)
# fit model
fit <- lm(neg_c_7 ~ quol_5, data=efc)
# plot regression line with label strings
sjp.lm(fit,
       axisLabels.x = "Quality of life",
       axisLabels.y = "Burden of care",
       showLoess = TRUE)

# --------------------------------------------------
# plotting regression lines of each single predictor
# of a fitted model
# --------------------------------------------------
library(sjmisc)
data(efc)
# fit model
fit <- lm(tot_sc_e ~ c12hour + e17age + e42dep, data=efc)

# reression line and scatter plot
sjp.lm(fit, type = "pred")

# reression line w/o scatter plot
sjp.lm(fit,
       type = "pred",
       showScatterPlot = FALSE)

# --------------------------
# plotting model assumptions
# --------------------------
sjp.lm(fit, type = "ma")

\dontrun{
# --------------------------
# plotting polynomial terms
# --------------------------
library(sjmisc)
data(efc)
# fit sample model
fit <- lm(tot_sc_e ~ c12hour + e17age + e42dep, data = efc)
# "e17age" does not seem to be linear correlated to response
# try to find appropiate polynomial. Grey line (loess smoothed)
# indicates best fit. Looks like x^3 has a good fit.
# (not checked for significance yet).
sjp.poly(fit, "e17age", 2:4, showScatterPlot = FALSE)
# fit new model
fit <- lm(tot_sc_e ~ c12hour + e42dep +
          e17age + I(e17age^2) + I(e17age^3),
          data = efc)
# plot marginal effects of polynomial term
sjp.lm(fit, type = "poly", poly.term = "e17age")


library(splines)
# fit new model with "splines"-package, "bs"
fit <- lm(tot_sc_e ~ c12hour + e42dep + bs(e17age, 3), data = efc)
# plot marginal effects of polynomial term, same call as above
sjp.lm(fit, type = "poly", poly.term = "e17age")}
}
\references{
Gelman A (2008) "Scaling regression inputs by dividing by two standard deviations." \emph{Statistics in Medicine 27: 2865–2873.} \url{http://www.stat.columbia.edu/~gelman/research/published/standardizing7.pdf}
}
\seealso{
\href{http://www.strengejacke.de/sjPlot/sjp.lm}{sjPlot manual: sjp.lm} for
           more details and examples of this function; use \code{\link{sjp.poly}}
           to see which polynomial degree fits best for possible polynomial terms.
}

