% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/sim_gs_n.R
\name{sim_gs_n}
\alias{sim_gs_n}
\title{Simulate group sequential designs with fixed sample size}
\usage{
sim_gs_n(
  n_sim = 1000,
  sample_size = 500,
  stratum = data.frame(stratum = "All", p = 1),
  enroll_rate = data.frame(duration = c(2, 2, 10), rate = c(3, 6, 9)),
  fail_rate = data.frame(stratum = "All", duration = c(3, 100), fail_rate = log(2)/c(9,
    18), hr = c(0.9, 0.6), dropout_rate = rep(0.001, 2)),
  block = rep(c("experimental", "control"), 2),
  test = wlr,
  cut = NULL,
  seed = 2024,
  ...
)
}
\arguments{
\item{n_sim}{Number of simulations to perform.}

\item{sample_size}{Total sample size per simulation.}

\item{stratum}{A data frame with stratum specified in \code{stratum},
probability (incidence) of each stratum in \code{p}.}

\item{enroll_rate}{Piecewise constant enrollment rates by time period.
Note that these are overall population enrollment rates and the \code{stratum}
argument controls the random distribution between stratum.}

\item{fail_rate}{Piecewise constant control group failure rates, hazard ratio
for experimental vs. control, and dropout rates by stratum and time period.}

\item{block}{As in \code{\link[=sim_pw_surv]{sim_pw_surv()}}. Vector of treatments to be included
in each block.}

\item{test}{One or more test functions such as \code{\link[=wlr]{wlr()}}, \code{\link[=maxcombo]{maxcombo()}}, or
\code{\link[=rmst]{rmst()}}. If a single test function is provided, it will be applied at each
cut. Alternatively a list of functions created by \code{\link[=create_test]{create_test()}}. The list
form is experimental and currently limited. It only accepts one test per
cutting (in the future multiple tests may be accepted), and all the tests
must consistently return the same exact results (again this may be more
flexible in the future). Importantly, note that the simulated data set is
always passed as the first positional argument to each test function
provided.}

\item{cut}{A list of cutting functions created by \code{\link[=create_cut]{create_cut()}}, see
examples.}

\item{seed}{Random seed.}

\item{...}{Arguments passed to the test function(s) provided by the argument
\code{test}.}
}
\value{
A data frame summarizing the simulation ID, analysis date,
z statistics or p-values.
}
\description{
WARNING: This experimental function is a work-in-progress. The function
arguments will change as we add additional features.
}
\examples{
\dontshow{if (rlang::is_installed("gsDesign2")) (if (getRversion() >= "3.4") withAutoprint else force)(\{ # examplesIf}
library(gsDesign2)

# Parameters for enrollment
enroll_rampup_duration <- 4 # Duration for enrollment ramp up
enroll_duration <- 16 # Total enrollment duration
enroll_rate <- define_enroll_rate(
  duration = c(
    enroll_rampup_duration,
    enroll_duration - enroll_rampup_duration
  ),
  rate = c(10, 30)
)

# Parameters for treatment effect
delay_effect_duration <- 3 # Delay treatment effect in months
median_ctrl <- 9 # Survival median of the control arm
median_exp <- c(9, 14) # Survival median of the experimental arm
dropout_rate <- 0.001
fail_rate <- define_fail_rate(
  duration = c(delay_effect_duration, 100),
  fail_rate = log(2) / median_ctrl,
  hr = median_ctrl / median_exp,
  dropout_rate = dropout_rate
)

# Other related parameters
alpha <- 0.025 # Type I error
beta <- 0.1 # Type II error
ratio <- 1 # Randomization ratio (experimental:control)

# Define cuttings of 2 IAs and 1 FA
# IA1
# The 1st interim analysis will occur at the later of the following 3 conditions:
# - At least 20 months have passed since the start of the study.
# - At least 100 events have occurred.
# - At least 20 months have elapsed after enrolling 200/400 subjects, with a
#   minimum of 20 months follow-up.
# However, if events accumulation is slow, we will wait for a maximum of 24 months.
ia1 <- create_cut(
  planned_calendar_time = 20,
  target_event_overall = 100,
  max_extension_for_target_event = 24,
  min_n_overall = 200,
  min_followup = 20
)

# IA2
# The 2nd interim analysis will occur at the later of the following 3 conditions:
# - At least 32 months have passed since the start of the study.
# - At least 250 events have occurred.
# - At least 10 months after IA1.
# However, if events accumulation is slow, we will wait for a maximum of 34 months.
ia2 <- create_cut(
  planned_calendar_time = 32,
  target_event_overall = 200,
  max_extension_for_target_event = 34,
  min_time_after_previous_analysis = 10
)

# FA
# The final analysis will occur at the later of the following 2 conditions:
# - At least 45 months have passed since the start of the study.
# - At least 300 events have occurred.
fa <- create_cut(
  planned_calendar_time = 45,
  target_event_overall = 350
)

# Test 1: regular logrank test
sim_gs_n(
  n_sim = 3,
  sample_size = 400,
  enroll_rate = enroll_rate,
  fail_rate = fail_rate,
  test = wlr,
  cut = list(ia1 = ia1, ia2 = ia2, fa = fa),
  seed = 2024,
  weight = fh(rho = 0, gamma = 0)
)

# Test 2: weighted logrank test by FH(0, 0.5)
sim_gs_n(
  n_sim = 3,
  sample_size = 400,
  enroll_rate = enroll_rate,
  fail_rate = fail_rate,
  test = wlr,
  cut = list(ia1 = ia1, ia2 = ia2, fa = fa),
  seed = 2024,
  weight = fh(rho = 0, gamma = 0.5)
)

# Test 3: weighted logrank test by MB(3)
sim_gs_n(
  n_sim = 3,
  sample_size = 400,
  enroll_rate = enroll_rate,
  fail_rate = fail_rate,
  test = wlr,
  cut = list(ia1 = ia1, ia2 = ia2, fa = fa),
  seed = 2024,
  weight = mb(delay = 3)
)

# Test 4: weighted logrank test by early zero (6)
sim_gs_n(
  n_sim = 3,
  sample_size = 400,
  enroll_rate = enroll_rate,
  fail_rate = fail_rate,
  test = wlr,
  cut = list(ia1 = ia1, ia2 = ia2, fa = fa),
  seed = 2024,
  weight = early_zero(6)
)

# Test 5: RMST
sim_gs_n(
  n_sim = 3,
  sample_size = 400,
  enroll_rate = enroll_rate,
  fail_rate = fail_rate,
  test = rmst,
  cut = list(ia1 = ia1, ia2 = ia2, fa = fa),
  seed = 2024,
  tau = 20
)

# Test 6: Milestone
sim_gs_n(
  n_sim = 3,
  sample_size = 400,
  enroll_rate = enroll_rate,
  fail_rate = fail_rate,
  test = milestone,
  cut = list(ia1 = ia1, ia2 = ia2, fa = fa),
  seed = 2024,
  ms_time = 10
)

# Test 7: MaxCombo (WLR-FH(0,0) + WLR-FH(0, 0.5))
# for all analyses
sim_gs_n(
  n_sim = 3,
  sample_size = 400,
  enroll_rate = enroll_rate,
  fail_rate = fail_rate,
  test = maxcombo,
  cut = list(ia1 = ia1, ia2 = ia2, fa = fa),
  seed = 2024,
  rho = c(0, 0),
  gamma = c(0, 0.5)
)

# Test 8: MaxCombo (WLR-FH(0,0.5) + milestone(10))
# for all analyses
\dontrun{
sim_gs_n(
  n_sim = 3,
  sample_size = 400,
  enroll_rate = enroll_rate,
  fail_rate = fail_rate,
  test = maxcombo(test1 = wlr, test2 = milestone),
  cut = list(ia1 = ia1, ia2 = ia2, fa = fa),
  seed = 2024,
  test1_par = list(weight = fh(rho = 0, gamma = 0.5)),
  test2_par = list(ms_time = 10)
)
}

# Test 9: MaxCombo (WLR-FH(0,0) at IAs
# and WLR-FH(0,0) + milestone(10) + WLR-MB(4,2) at FA)
\dontrun{
sim_gs_n(
  n_sim = 3,
  sample_size = 400,
  enroll_rate = enroll_rate,
  fail_rate = fail_rate,
  test = list(ia1 = wlr, ia2 = wlr, fa = maxcombo),
  cut = list(ia1 = ia1, ia2 = ia2, fa = fa),
  seed = 2024,
  test_par = list(
    ia1 = list(weight = fh(rho = 0, gamma = 0)),
    ia2 = list(weight = fh(rho = 0, gamma = 0)),
    ia3 = list(
      test1_par = list(weight = fh(rho = 0, gamma = 0)),
      test2_par = list(ms_time = 10),
      test3_par = list(delay = 4, w_max = 2)
    )
  )
)
}
\dontshow{\}) # examplesIf}
}
