## createData: Create data from model implied means and covariance matrix.
## Optionally, the distribution of indicators can be specified in this method
## using data distribution objects and the function simDataDist or bindDist.
## Alternatively, data can be generated using the sequential method, which
## allows for the specification of the distribution of factors and distribution
## of errors.  The third approach is the model bootstrap approach, where data
## is generated by bootstrapping a real data set.  For every output, a data set
## is generated.

createData <- function(paramSet, n, indDist = NULL, sequential = FALSE, facDist = NULL, 
    errorDist = NULL, indLab = NULL, modelBoot = FALSE, realData = NULL) {
    require(MASS)
    if (modelBoot) {
        if (sequential) 
            stop("The model-based bootstrap and sequential cannot be used at the same time.")
        if (is.null(realData)) 
            stop("If the data are generated by model bootstrap, the real data are needed.")
        if (!is.null(indLab)) 
            realData <- realData[, indLab]
        if (sum(is.na(realData)) > 0) 
            stop("The model-based bootstrap is not available for data with missingness.")
    }
    if (is.null(n)) 
        n <- nrow(realData)
    if (!is.null(errorDist)) {
        if (!is.null(paramSet$BE) && is.null(paramSet$LY)) 
            # Model is path analysis
        stop("errorDist is not allowed for path analysis model. The distribution of each indicator should be specified in facDist if sequential=TRUE.")
    }
    if (!sequential & !is.null(facDist)) 
        stop("facDist is not allowed when using model-implied method in data generation")
    if (!sequential & !is.null(errorDist)) 
        stop("errorDist is not allowed when using model-implied method in data generation")
    if (sequential & !is.null(indDist)) 
        stop("indDist is not allowed when using sequential method in data generation")
    # classes <- sapply(list(facDist,indDist,errorDist),class) could be a check
    # soon
    
    Data <- NULL
    param <- paramSet$param
    usedParam <- NULL
    if (!is.null(paramSet$misspec)) {
        usedParam <- paramSet$misspec
    } else {
        usedParam <- param
    }
    
    if (modelBoot) {
        require(lavaan)
        originalData <- realData
        if (!is.null(indLab)) 
            originalData <- originalData[, indLab]
        implied <- createImpliedMACS(usedParam)
        S <- cov(originalData)
        Sigma <- implied$CM
        M <- colMeans(originalData)
        M <- matrix(rep(M, n), nrow = n, byrow = TRUE)
        Mu <- implied$M
        Mu <- matrix(rep(Mu, n), nrow = n, byrow = TRUE)
        z <- (scale(originalData, scale = FALSE) %*% (solve(sqrtSymmetricMatrix(S)) %*% 
            sqrtSymmetricMatrix(Sigma))) + Mu
        index <- sample(1:n, replace = TRUE)
        Data <- z[index, ]
    } else {
        if (sequential) {
            if (is.null(usedParam$BE) && !is.null(usedParam$LY)) {
                # CFA
                fac <- dataGen(facDist, n, usedParam@AL, usedParam@PS)
                trueScore <- fac %*% t(usedParam@LY)
                errorScore <- dataGen(errorDist, n, usedParam@TY, usedParam@TE)
                Data <- trueScore + errorScore
            } else {
                usedParam2 <- NULL
                if (!is.null(usedParam$BE)) {
                  # SEM or Path
                  usedParam2 <- usedParam
                } else {
                  stop("Incorrect model type")
                }
                set <- findRecursiveSet(usedParam2$BE)
                iv <- set[[1]]
                fac <- dataGen(extractSimDataDist(facDist, iv), n, usedParam2$AL[iv], 
                  usedParam2$PS[iv, iv])
                for (i in 2:length(set)) {
                  dv <- set[[i]]
                  pred <- fac %*% t(usedParam2$BE[dv, iv, drop = FALSE])
                  res <- dataGen(extractSimDataDist(facDist, dv), n, usedParam2$AL[dv], 
                    usedParam2$PS[dv, dv])
                  new <- pred + res
                  fac <- cbind(fac, new)
                  iv <- c(iv, set[[i]])
                }
                if (is.null(usedParam$LY)) {
                  # Path
                  Data <- fac
                } else {
                  # SEM
                  trueScore <- fac %*% t(usedParam2$LY)
                  errorScore <- dataGen(errorDist, n, usedParam2$TY, usedParam2$TE)
                  Data <- trueScore + errorScore
                }
            }
        } else {
            macs <- createImpliedMACS(usedParam)
            if (!is.null(indDist)) {
                Data <- dataGen(indDist, n, macs$M, macs$CM)
            } else {
                Data <- mvrnorm(n, macs$M, macs$CM)
            }
        }
    }
    varnames <- NULL
    if (!is.null(indLab)) {
        varnames <- indLab
    } else {
        varnames <- paste0("x", 1:ncol(Data))
    }
    colnames(Data) <- varnames
    Data <- as.data.frame(Data)
    return(Data)
}

dataGen <- function(dataDist, n, m, cm) {
    require(MASS)
    Data <- NULL
    # Check dim(M) dim(CM) dim(copula) are equal
    if (!is.null(dataDist)) {
        require(copula)
        if (dataDist@p > 1) {
            varNotZeros <- diag(cm) != 0
            dataDist2 <- dataDist
            cm2 <- cm
            if (sum(varNotZeros) < dataDist@p) {
                dataDist2 <- extractSimDataDist(dataDist, which(varNotZeros))
                cm2 <- cm[which(varNotZeros), which(varNotZeros), drop=FALSE]
            }
            r <- cov2cor(as.matrix(cm2))
            for (i in 1:dataDist2@p) {
                if (dataDist2@reverse[i] == TRUE) {
                  r[i, ] <- -1 * r[i, ]
                  r[, i] <- -1 * r[, i]
                }
            }
            listR <- r[lower.tri(diag(dataDist2@p))]
            CopNorm <- ellipCopula(family = "normal", dim = dataDist2@p, dispstr = "un", 
                param = listR)
            
            Mvdc <- mvdc(CopNorm, dataDist2@margins, dataDist2@paramMargins)
            Data <- rMvdc(n, Mvdc)
            if (sum(varNotZeros) < dataDist@p) {
                varZeros <- diag(cm) == 0
                constant <- matrix(0, n, sum(varZeros))
                Data <- data.frame(Data, constant)
                Data[, c(which(varNotZeros), which(varZeros))] <- Data
            }
        } else if (dataDist@p == 1) {
            if (as.matrix(cm)[1, 1] == 0) {
                Data <- rep(m[1], n)
            } else {
                # Data <- as.matrix(run(dataDist@dist[[1]], n = n))
                temp <- c(list(get(paste0("r", dataDist@margins[[1]]))), dataDist@paramMargins[[1]], 
                  list(n = n))
                Data <- as.matrix(eval(as.call(temp)))
            }
        } else {
            stop("when creating a data distribution object, p cannot equal 0.")
        }
        for (i in 1:dataDist@p) {
            if (dataDist@reverse[i] == TRUE) {
                meanOld <- mean(Data[, i])
                anchor <- max(Data[, i])
                datNew <- anchor - Data[, i]
                Data[, i] <- datNew - mean(datNew) + meanOld
            }
        }
        if (!is.matrix(Data)) 
            Data <- as.matrix(Data)
        if (any(dataDist@keepScale)) {
            Data <- scale(Data)
            Data[is.na(Data)] <- 0
            fakeDat <- mvrnorm(n, m, cm)
            fakeMean <- apply(fakeDat, 2, mean)
            fakeSD <- apply(fakeDat, 2, sd)
            Data <- t(apply(Data, 1, function(y, m, s) {
                y * s + m
            }, m = fakeMean, s = fakeSD))
        }
        if (nrow(Data) == 1) 
            Data <- t(Data)
    } else {
        Data <- mvrnorm(n, m, cm)
    }
    return(Data)
}

extractSimDataDist <- function(object, pos) {
    return(new("SimDataDist", margins = object@margins[pos], paramMargins = object@paramMargins[pos], 
        p = length(pos), keepScale = object@keepScale[pos], reverse = object@reverse[pos]))
} 
