#' calculate_energy
#'
#' @description Calculate mean energy
#'
#' @param pattern List with reconstructed patterns.
#' @param return_mean Return the mean energy.
#' @param method String to specifiy if spatial pattern or marks were reconstructed.
#' @param comp_fast If pattern contains more points than threshold, summary functions are estimated in a computational fast way.
#' @param verbose Print progress report.
#'
#' @details
#' The function calculates the mean energy (or deviation) between the observed
#' pattern and all reconstructed patterns (for more information see Tscheschel &
#' Stoyan (2006) or Wiegand & Moloney (2014)). The pair correlation function and the
#' nearest neighbour distance function are used to describe the patterns. For large
#' patterns `comp_fast = TRUE` decreases the computational demand because no edge
#' correction is used and the pair correlation function is estimated based on Ripley's
#' K-function. For more information see \code{\link{estimate_pcf_fast}}.
#'
#' @seealso
#' \code{\link{reconstruct_pattern}} \cr
#' \code{\link{plot_randomized_pattern}}
#'
#' @return numeric
#'
#' @examples
#' pattern_random <- fit_point_process(species_a, n_random = 19)
#' calculate_energy(pattern_random)
#' calculate_energy(pattern_random, return_mean = TRUE)
#'
#' \dontrun{
#' marks_sub <- spatstat::subset.ppp(species_a, select = dbh)
#' marks_recon <- reconstruct_marks(pattern_random[[1]], marks_sub, n_random = 19, max_runs = 1000)
#' calculate_energy(marks_recon, return_mean = FALSE, method = "marks")
#' }
#'
#' @aliases calculate_energy
#' @rdname calculate_energy
#'
#' @references
#' Tscheschel, A., & Stoyan, D. (2006). Statistical reconstruction of random point
#' patterns. Computational Statistics and Data Analysis, 51(2), 859-871.
#'
#' Wiegand, T., & Moloney, K. A. (2014). Handbook of spatial point-pattern analysis
#' in ecology. Boca Raton: Chapman and Hall/CRC Press.

#' @export
calculate_energy <- function(pattern,
                             return_mean = FALSE,
                             method = "spatial",
                             comp_fast = 1000,
                             verbose = TRUE){

  # check if randomized and observed is present
  if(!all(c(paste0("randomized_", seq_len(length(pattern) - 1)), "observed") == names(pattern)) || is.null(names(pattern))) {
    stop("Input must named 'randomized_1' to 'randomized_n' and includ 'observed' pattern.",
         call. = FALSE)
  }

  # extract observed pattern
  pattern_observed <- pattern$observed

  # extract randomized patterns
  pattern_reconstructed <- pattern[names(pattern) != "observed"]

  # calculate r sequence
  r <- seq(from = 0,
           to = spatstat::rmax.rule(W = pattern_observed$window,
                                    lambda = spatstat::intensity.ppp(pattern_observed)),
           length.out = 250)

  if (method == "spatial") {

    if(verbose) {
      # check if pattern is marked
      if(spatstat::is.marked(pattern_observed) || all(vapply(pattern_reconstructed,
                                                             spatstat::is.marked,
                                                             FUN.VALUE = logical(1)))) {

        warning("Only energy of spatial summary functions are considered.", call. = FALSE)
      }
    }

    # check if number of points exceed comp_fast limit
    if(pattern_observed$n > comp_fast) {
      comp_fast <- TRUE
    }

    else {
      comp_fast <- FALSE
    }

    # calculate summary functions for observed pattern
    if(comp_fast) {

      gest_observed <- spatstat::Gest(X = pattern_observed,
                                      correction = "none",
                                      r = r)

      pcf_observed <- shar::estimate_pcf_fast(pattern = pattern_observed,
                                              correction = "none",
                                              method = "c",
                                              spar = 0.5,
                                              r = r)
    }

    else{

      gest_observed <- spatstat::Gest(X = pattern_observed,
                                      correction = "han",
                                      r = r)

      pcf_observed <- spatstat::pcf(X = pattern_observed,
                                    correction = "best",
                                    divisor = "d",
                                    r = r)
    }

    # loop through all reconstructed patterns
    result <- vapply(seq_along(pattern_reconstructed), function(x) {

      # fast computation of summary stats
      if(comp_fast) {

        gest_reconstruction <- spatstat::Gest(X = pattern_reconstructed[[x]],
                                              correction = "none",
                                              r = r)

        pcf_reconstruction <- shar::estimate_pcf_fast(pattern = pattern_reconstructed[[x]],
                                                      correction = "none",
                                                      method = "c",
                                                      spar = 0.5,
                                                      r = r)
      }

      # normal computation of summary stats
      else{

        gest_reconstruction <- spatstat::Gest(X = pattern_reconstructed[[x]],
                                              correction = "han",
                                              r = r)

        pcf_reconstruction <- spatstat::pcf(X = pattern_reconstructed[[x]],
                                            correction = "best",
                                            divisor = "d",
                                            r = r)
      }

      # difference between observed and reconstructed pattern
      energy <- mean(abs(gest_observed[[3]] - gest_reconstruction[[3]]), na.rm = TRUE) +
        mean(abs(pcf_observed[[3]] - pcf_reconstruction[[3]]), na.rm = TRUE)

      # print progress
      if(verbose) {
        message("\r> Progress: ", x, "/", length(pattern_reconstructed), appendLF = FALSE)
      }

      return(energy)

    }, FUN.VALUE = numeric(1))
  }

  else if( method == "marks") {

    # check if pattern is marked
    if(!spatstat::is.marked(pattern_observed) || !all(vapply(pattern_reconstructed,
                                                             spatstat::is.marked,
                                                             FUN.VALUE = logical(1)))) {

      stop("Please provide pattern with reconstruced marks.", call. = FALSE)
    }

    # calculate summary functions
    kmmr_observed <- spatstat::markcorr(pattern_observed,
                                        correction = "Ripley",
                                        r = r)

    result <- vapply(seq_along(pattern_reconstructed), function(x) {

      # calculate summary functions
      kmmr_reconstruction <- spatstat::markcorr(pattern_reconstructed[[x]],
                                                correction = "Ripley",
                                                r = r)

      # difference between observed and reconstructed pattern
      energy <- mean(abs(kmmr_observed[[3]] - kmmr_reconstruction[[3]]), na.rm = TRUE) +
        mean(abs(kmmr_observed[[3]] - kmmr_reconstruction[[3]]), na.rm = TRUE)

      # print progress
      if(verbose) {
        message("\r> Progress: ", x, "/", length(pattern_reconstructed), appendLF = FALSE)
      }

      return(energy)

    }, FUN.VALUE = numeric(1))
  }

  else {
    stop("Please select either 'method = spatial' or 'method = marks'.",
         call. = FALSE)
  }

  # return mean for all reconstructed patterns
  if(return_mean) {
    result <- mean(result)
  }

  # write result in new line if progress was printed
  if(verbose) {
    message("\r")
  }

  return(result)
}
