% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/semPower-package.R, R/semPower.R
\docType{package}
\name{semPower}
\alias{semPower}
\alias{semPower-package}
\title{semPower: Power analyses for structural equation models (SEM).}
\usage{
semPower(type, ...)
}
\arguments{
\item{type}{type of power analysis, one of \code{'a-priori'}, \code{'post-hoc'}, \code{'compromise'}.}

\item{...}{other parameters related to the specific type of power analysis requested.}
}
\value{
list
}
\description{
semPower allows for performing a-priori, post-hoc, and compromise power-analyses for structural equation models (SEM).

Perform a power analysis. This is a wrapper function for a-priori, post-hoc, and compromise power analyses.
}
\details{
\itemize{
\item A-priori power analysis \code{\link{semPower.aPriori}} computes the required N, given an effect, alpha, power, and the model df
\item Post-hoc power analysis \code{\link{semPower.postHoc}} computes the achieved power, given an effect, alpha, N, and the model df
\item Compromise power analysis \code{\link{semPower.compromise}} computes the implied alpha and power, given an effect, the alpha/beta ratio, N, and the model df
}

In SEM, the discrepancy between H0 and H1 (the magnitude of effect) refers to the difference in fit between two models. If only one model is defined (which is the default), power refers to the global chi-square test. If both models are explicitly defined, power is computed for nested model tests.  semPower allows for expressing the magnitude of effect by one of the following measures: F0, RMSEA, Mc, GFI, or AGFI.

Alternatively, the implied effect can also be computed from the discrepancy between the population (or a certain model-implied) covariance matrix defining H0 and the hypothesized (model-implied) covariance matrix from a nested model defining H1. See the examples below how to use this feature in conjunction with lavaan.
}
\examples{
# a-priori power analyses using rmsea = .05 a target power (1-beta) of .80
ap1 <- semPower.aPriori(0.05, 'RMSEA', alpha = .05, beta = .20, df = 200)
summary(ap1)
# generic version
gap1 <- semPower(type = 'a-priori', 0.05, 'RMSEA', alpha = .05, beta = .20, df = 200)
summary(gap1)
# a-priori power analyses using f0 = .75 and a target power of .95
ap2 <- semPower.aPriori(0.75, 'F0', alpha = .05, power = .95, df = 200)
summary(ap2)
# create a plot showing how power varies by N (given a certain effect)
semPower.powerPlot.byN(.05, 'RMSEA', alpha=.05, df=200, power.min=.05, power.max=.99)
# post-hoc power analyses using rmsea = .08
ph <- semPower.postHoc(.08, 'RMSEA', alpha = .05, N = 250, df = 50)
summary(ph)
# generic version
gph1 <- semPower(type = 'post-hoc', .08, 'RMSEA', alpha = .05, N = 250, df = 50)
summary(gph1)
# create a plot showing how power varies by the magnitude of effect (given a certain N)
semPower.powerPlot.byEffect('RMSEA', alpha=.05, N = 100, df=200, effect.min=.001, effect.max=.10)
# compromise power analyses using rmsea = .08 and an abratio of 2
cp <- semPower.compromise(.08, 'RMSEA', abratio = 2, N = 1000, df = 200)
summary(cp)
# generic version
gcp <- semPower(type = 'compromise', .08, 'RMSEA', abratio = 2, N = 1000, df = 200)
summary(gcp)

# use lavaan to define effect through covariance matrices:
\dontrun{
library(lavaan)

# define population model (= H1)
model.pop <- '
f1 =~ .8*x1 + .7*x2 + .6*x3
f2 =~ .7*x4 + .6*x5 + .5*x6
f1 ~~ 1*f1
f2 ~~ 1*f2
f1 ~~ 0.5*f2
'
# define (wrong) H0 model
model.h0 <- '
f1 =~ x1 + x2 + x3
f2 =~ x4 + x5 + x6
f1 ~~ 0*f2
'

# get population covariance matrix; equivalent to a perfectly fitting H1 model
cov.h1 <- fitted(sem(model.pop))$cov
# get covariance matrix as implied by H0 model
res.h0 <- sem(model.h0, sample.cov = cov.h1, sample.nobs = 1000, 
              likelihood='wishart', sample.cov.rescale = F)
df <- res.h0@test[[1]]$df
cov.h0 <- fitted(res.h0)$cov

## do power analyses

# post-hoc
ph <- semPower.postHoc(SigmaHat = cov.h0, Sigma = cov.h1, alpha = .05, N = 1000, df = df)
summary(ph)
# => Power to reject the H1 model is > .9999 (1-beta = 1-1.347826e-08) with N = 1000 at alpha=.05

# compare:
ph$fmin * (ph$N-1)
fitmeasures(res.h1, 'chisq')
# => expected chi-square matches empirical chi-square

# a-priori
ap <- semPower.aPriori(SigmaHat = cov.h0, Sigma = cov.h1, alpha = .05, power = .80, df = df)
summary(ap)
# => N = 194 gives a power of ~80\% to reject the H1 model at alpha = .05

# compromise
cp <- semPower.compromise(SigmaHat = cov.h0, Sigma = cov.h1, abratio = 1, N = 1000, df = df)
summary(cp)
# => A critical Chi-Squared of 33.999 gives balanced alpha-beta
#    error probabilities of alpha=beta=0.000089 with N = 1000.

}

\dontrun{

ap <- semPower(type = 'a-priori', 
               effect = .08, effect.measure = "RMSEA", 
               alpha = .05, beta = .05, df = 200)
summary(ph)

ph <- semPower(type = 'post-hoc', 
               effect = .08, effect.measure = "RMSEA", 
               alpha = .05, N = 250, df = 200)
summary(ph)

cp <- semPower(type = 'compromise', 
               effect = .08, effect.measure = "RMSEA", 
               abratio = 1, N = 250, df = 200)
summary(ph)
}
}
\seealso{
Useful links:
\itemize{
  \item \url{https://github.com/moshagen/semPower}
  \item Report bugs at \url{https://github.com/moshagen/semPower/issues}
}


\code{\link[=semPower.aPriori]{semPower.aPriori()}}, \code{\link[=semPower.postHoc]{semPower.postHoc()}}, \code{\link[=semPower.compromise]{semPower.compromise()}}
}
\author{
Morten Moshagen \email{morten.moshagen@uni-ulm.de}
}
