% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/get-index-sims.R
\name{get_index_sims}
\alias{get_index_sims}
\title{Calculate a population index via simulation from the joint precision matrix}
\usage{
get_index_sims(
  obj,
  level = 0.95,
  return_sims = FALSE,
  area = rep(1, nrow(obj)),
  est_function = stats::median,
  area_function = function(x, area) x + log(area),
  agg_function = function(x) sum(exp(x))
)
}
\arguments{
\item{obj}{\code{\link[=predict.sdmTMB]{predict.sdmTMB()}} output with \code{nsim > 0}.}

\item{level}{The confidence level.}

\item{return_sims}{Logical. Return simulation draws? The default (\code{FALSE}) is
a quantile summary of those simulation draws.}

\item{area}{A vector of grid cell/polyon areas for each year-grid cell (row
of data) in \code{obj}. Adjust this if cells are not of unit area or not all
the same area (e.g., some cells are partially over land/water). Note that
the area vector is added as \code{log(area)} to the raw values in \code{obj}. In
other words, the function assumes a log link, which typically makes sense.}

\item{est_function}{Function to summarize the estimate (the expected value).
\code{mean()} would be an alternative to \code{median()}.}

\item{area_function}{Function to apply area weighting.
Assuming a log link, the \code{function(x, area) x + log(area)} default makes sense.
If in natural space, \code{function(x, area) x * area} makes sense.}

\item{agg_function}{Function to aggregate samples within each time slice.
Assuming a log link, the \code{function(x) sum(exp(x))} default makes sense.
If in natural space, \code{function(x) sum(x)} makes sense.}
}
\value{
A data frame. If \code{return_sims = FALSE}:
\itemize{
\item name of column (e.g. \code{year}) that was supplied to \code{\link[=sdmTMB]{sdmTMB()}} time argument
\item \code{est}: estimate
\item \code{lwr}: lower confidence interval value
\item \code{upr}: upper confidence interval value
\item \code{log_est}: log estimate
\item \code{se}: standard error on the log estimate
}

If \code{return_sims = TRUE}, samples from the index values in a long-format data frame:
\itemize{
\item name of column (e.g. \code{year}) that was supplied to \code{\link[=sdmTMB]{sdmTMB()}} time argument
\item \code{.value}: sample value
\item \code{.iteration}: sample number
}
}
\description{
\ifelse{html}{\href{https://lifecycle.r-lib.org/articles/stages.html#experimental}{\figure{lifecycle-experimental.svg}{options: alt='[Experimental]'}}}{\strong{[Experimental]}}

Calculate a population index via simulation from the joint precision matrix.
Compared to \code{\link[=get_index]{get_index()}}, this version can be faster if bias correction was
turned on in \code{\link[=get_index]{get_index()}} while being approximately equivalent. \strong{This is an
experimental function.} This function usually works reasonably well, but we
make no guarantees. It is recommended to use \code{\link[=get_index]{get_index()}} with \code{bias_correct = TRUE} for final inference.
}
\details{
Can also be used to produce an index from a model fit with
\pkg{tmbstan}.

This function does nothing more than summarize and reshape the
matrix of simulation draws into a data frame.
}
\examples{
if (inla_installed()) {

m <- sdmTMB(density ~ 0 + as.factor(year) + depth_scaled + depth_scaled2,
  data = pcod_2011, mesh = pcod_mesh_2011, family = tweedie(link = "log"),
  time = "year"
)
qcs_grid_2011 <- replicate_df(qcs_grid, "year", unique(pcod_2011$year))
p <- predict(m, newdata = qcs_grid_2011, nsim = 100)
x <- get_index_sims(p)
x_sims <- get_index_sims(p, return_sims = TRUE)

if (require("ggplot2", quietly = TRUE)) {
  ggplot(x, aes(year, est, ymin = lwr, ymax = upr)) +
    geom_line() +
    geom_ribbon(alpha = 0.4)
  ggplot(x_sims, aes(as.factor(year), .value)) +
    geom_violin()
}

# Demo custom functions if working in natural space:
ind <- get_index_sims(
  exp(p),
  agg_function = function(x) sum(x),
  area_function = function(x, area) x * area
)
}
}
\seealso{
\code{\link[=get_index]{get_index()}}
}
