#' @title Evaluate forecasts
#'
#' @description The function \code{eval_forecasts} is an easy to use wrapper
#' of the lower level functions in the \code{scoringutils} package.
#' It can be used to score probabilistic or quantile forecasts of
#' continuous, integer-valued or binary variables.
#'
#' @details the following metrics are used where appropriate:
#' \itemize{
#'   \item {Interval Score} for quantile forecasts. Smaller is better. See
#'   \code{\link{interval_score}} for more information.
#'   \item {Brier Score} for a probability forecast of a binary outcome.
#'   Smaller is better. See \code{\link{brier_score}} for more information.
#'   \item {Bias} 0 is good, 1 and -1 are bad.
#'   See \code{\link{bias}} for more information.
#'   \item {Sharpness} Smaller is better. See \code{\link{sharpness}} for more
#'   information.
#'   \item {Calibration} represented through the p-value of the
#'   Anderson-Darling test for the uniformity of the Probability Integral
#'   Transformation (PIT). For integer valued forecasts, this p-value also
#'   has a standard deviation. Larger is better.
#'   See \code{\link{pit}} for more information.
#'   \item {DSS} Dawid-Sebastiani-Score. Smaller is better.
#'   See \code{\link{dss}} for more information.
#'   \item {CRPS} Continuous Ranked Probability Score. Smaller is better.
#'   See \code{\link{crps}} for more information.
#'   \item {LogS} Log Score. Smaller is better. Only for continuous forecasts.
#'   See \code{\link{logs}} for more information.
#' }
#'
#' @param data A data.frame or data.table with the following columns:
#' \itemize{
#'   \item \code{true_values} the true observed values
#'   \item \code{id} A unique identifier of the true values. Could be a date
#'   or just a running index
#'   \item \code{predictions} predictions or predictive samples for one
#'   true value. Not necessary in case of quantile forecasts
#'   \item \code{model} name of the model that generated the predictions
#'   \item \code{sample} an index to identify the predictive samples in the
#'   predictiions column generated by one model for one true value. Only
#'   necessary for continuous and integer forecasts, not for quantile or
#'   binary predictions.
#'   \item {quantile forecasts} a number of pairs of columns with
#'   quantile predictions for a certain range. For a 50% interval
#'   (corresponding to the 25% and
#'   75% quantile), one column has to be named \code{lower_50} and one
#'   \code{upper_50}.
#'   For the median, there has to a column \code{lower_0} and one
#'   \code{upper_0}
#'   }
#' @param summarised if \code{TRUE}, only one average score is returned per model
#'
#' @return A data.table with appropriate scores. For binary predictions,
#' the Brier Score will be returned, for quantile predictions the interval
#' score. For integer forecasts, Sharpness, Bias, DSS, CRPS, LogS, and
#' pit_p_val (as an indicator of calibration) are returned. For integer
#' forecasts, pit_sd is returned (to account for the randomised PIT),
#' but no Log Score is returned (the internal estimation relies on a
#' kernel density estimate which is difficult for integer-valued forecasts).
#' If \code{summarised = TRUE} the average score per model is returned.
#'
#' @importFrom data.table ':=' setDT
#'
#' @examples
#' ## Probability Forecast for Binary Target
#' binary_example <- data.table::setDT(scoringutils::binary_example_data)
#' eval <- scoringutils::eval_forecasts(binary_example)
#' eval <- scoringutils::eval_forecasts(binary_example, summarised = FALSE)
#'
#' ## Quantile Forecasts
#' quantile_example <- data.table::setDT(scoringutils::quantile_example_data)
#' eval <- scoringutils::eval_forecasts(quantile_example)
#' eval <- scoringutils::eval_forecasts(quantile_example, summarised = FALSE)
#'
#' ## Integer Forecasts
#' integer_example <- data.table::setDT(scoringutils::integer_example_data)
#' eval <- scoringutils::eval_forecasts(integer_example)
#' eval <- scoringutils::eval_forecasts(integer_example, summarised = FALSE)
#'
#' ## Continuous Forecasts
#' continuous_example <- data.table::setDT(scoringutils::continuous_example_data)
#' eval <- scoringutils::eval_forecasts(continuous_example)
#' eval <- scoringutils::eval_forecasts(continuous_example, summarised = FALSE)
#'
#' @author Nikos Bosse \email{nikosbosse@gmail.com}
#' @references Funk S, Camacho A, Kucharski AJ, Lowe R, Eggo RM, Edmunds WJ
#' (2019) Assessing the performance of real-time epidemic forecasts: A
#' case study of Ebola in the Western Area region of Sierra Leone, 2014-15.
#' PLoS Comput Biol 15(2): e1006785.
#' \url{https://doi.org/10.1371/journal.pcbi.1006785}
#' @export
#'



eval_forecasts <- function(data,
                           summarised = TRUE) {

  data.table::setDT(data)

  ## check if predictions are integer, continuous, etc.
  if (any(grepl("lower", names(data)))) {
    prediction_type <- "quantile"
  } else if (all.equal(data$predictions, as.integer(data$predictions)) == TRUE) {
    prediction_type <- "integer"
  } else {
    prediction_type <- "continuous"
  }

  if (all.equal(data$true_values, as.integer(data$true_values)) == TRUE) {
    if (all(data$true_values %in% c(0,1))) {
      target_type = "binary"
    } else {
      target_type = "integer"
    }
  } else {
    target_type = "continuous"
  }

  # ============================================

  models <- unique(data$model)
  n <- length(unique(data$id))

  # Brier Score for binary prediction
  if (target_type == "binary") {

    res <- data[, "Brier_score" := scoringutils::brier_score(true_values, predictions),
         by = .(model, id)]

    if (summarised) {
      res <- data[, .(Brier_score = mean(Brier_score)), by  = model]
    }
    return(res)
  }

  # interval score for quantile prediction
  if (prediction_type == "quantile") {

    # get column names
    colnames <- colnames(data)
    ranges <- colnames[grepl("lower", colnames) | grepl("upper", colnames)]

    data <- data.table::melt(data,
                     id.vars = c("id", "true_values", "model"),
                     measure.vars = ranges,
                     variable.name = "range",
                     value.name = "predictions")
    data[, type := gsub("_.*", "", range)]
    data[, range := as.numeric(gsub("^.*?_","", range))]
    data <- data.table::dcast(data, id + true_values + range + model ~ type,
                              value.var = "predictions")
    res <- data[, "Interval_Score" := scoringutils::interval_score(true_values,
                                                                   lower,
                                                                   upper,
                                                                   range)]

    # question: what should be the correct input format for quantile forecasta?
    if (summarised) {
      res <- res[, .("Interval_Score" = mean(Interval_Score)), by = model]
    }
    return(res)
  }

  ## scoring for integer or continuous forecasts
  # sharpness
  data[, sharpness := scoringutils::sharpness(t(predictions)), by = .(id, model)]

  # bias
  data[, bias := scoringutils::bias(unique(true_values),
                                     t(predictions)), by = .(id, model)]

  # DSS
  data[, DSS := scoringutils::dss(unique(true_values),
                                    t(predictions)), by = .(id, model)]

  # CRPS
  data[, CRPS := scoringutils::crps(unique(true_values),
                                    t(predictions)), by = .(id, model)]

  # Log Score
  if (prediction_type == "continuous") {
    data[, LogS := scoringutils::logs(unique(true_values),
                                       t(predictions)), by = .(id, model)]
  }

  # calibration
  # reformat data.table to wide format
  dat <- data.table::dcast(data, model + id + true_values ~ paste("sampl_", sample, sep = ""),
                           value.var = "predictions")

  # compute pit p-values
  dat[, c("pit_p_val", "pit_sd") := pit(true_values,
                                as.matrix(.SD),
                                plot = FALSE), .SDcols = names(dat)[grepl("sampl_", names(dat))], by = model]
  dat[, names(dat)[grepl("sampl_", names(dat))] := NULL]


  # merge with previous data
  res <- merge(data, dat, by = c("id", "model", "true_values"))

  if (prediction_type == "continuous") {
    scores <- c("sharpness", "bias", "DSS", "CRPS", "LogS", "pit_p_val")
  } else {
    scores <- c("sharpness", "bias", "DSS", "CRPS", "pit_p_val", "pit_sd")
  }

  # aggregate scores so there is only one score per observed value
  res <- res[, lapply(.SD, mean, na.rm = TRUE),
             .SDcols = scores,
             by = .(model, id)]

  if (summarised) {
    res <- res[, lapply(.SD, mean, na.rm = TRUE),
               .SDcols = scores,
               by = .(model)]
  }
  return (res)
}










