% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/Rcpp_ADFun.R
\docType{class}
\name{Rcpp_ADFun-class}
\alias{Rcpp_ADFun-class}
\alias{Rcpp_ADFun}
\alias{ADFun}
\title{A Class for CppAD Tapes}
\description{
Objects of type \code{Rcpp_ADFun} contain a tape of a \verb{C++} function (which has class \code{ADFun} in \code{CppAD}). These tapes are a record of operations performed by a function. Tapes can be evaluated and differentiated, and have properties (such as domain and range dimensions). Tapes also have dynamic parameters that can be updated. This class, \code{Rcpp_ADFun} uses reference semantics, so that copies all point to the same object and changes modify in place (i.e. changes modify the same object).
Properties and methods of an \code{Rcpp_ADFun} object are accessed via \code{$}.
}
\details{
An object of class \code{Rcpp_ADFun} wraps an \code{ADFun} object from \code{CppAD}. Many of the properties and behaviour of an \code{Rcpp_ADFun} object come directly from \code{ADFun} objects so more details and context can be found by looking at the \code{ADFun} object help in the \code{CppAD} \href{https://cppad.readthedocs.io}{\code{help}}.
The methods \code{eval()}, \code{Jac()} and \code{Hes()} have been added by \code{scorematchingad} as there were many cases where this seemed like an easier way to evaluate a tape.

Default printing of an \code{Rcpp_ADFun} object gives a short summary of the object, see \code{\link{print,Rcpp_ADFun}}.

Tapes cannot be saved from session to session.
}
\section{Methods - Tape Properties:}{
\itemize{
\item \verb{$size_order} Number of Taylor coefficient orders, per variable and direction, currently calculated and stored in the object.
\item \verb{$domain} Dimension of the domain space (i.e., length of the independent variables vector \code{x}).
\item \verb{$range} Dimension of the range space (i.e., length of the vector returned by \verb{$eval()}).
\item \verb{$size_dyn_ind} Number of independent dynamic parameters (i.e., length of the vector of dynamic parameters \code{dyn}).
\item \verb{$name} A name for the tape (may be empty). This is yet to incorporate the \code{CppAD} \code{function_name} property.
\item \verb{$xtape} The values of the independent variables used for the initial taping.
\item \verb{$dyntape} The values of the dynamic parameters used for the initial taping.
\item \verb{$get_check_for_nan()} Debugging: Return whether the tape is configured to check for NaN values during computation. The check for NaN only occurs if the \verb{C++} compilation enables debugging.
\item \verb{$set_check_for_nan(bool)} Set whether the tape should check for NaN values during computation (only effective if C++ debugging is enabled).
\item \verb{$parameter(i)} Check if the \code{i}th component of the range corresponds to a constant parameter. Indexing is by the \verb{C++} default, that is the first component has index \code{0}, the last component has index \verb{$range - 1}.
\item \verb{$new_dynamic(dyn)} Specify new values for the dynamic parameters.
}
}

\section{Methods - Tape Evaluation:}{
\itemize{
\item \verb{$eval(x, dyn)} Evaluate the function at new values of the variables and dynamic parameters. Returns a vector of length \verb{$range}.
\item \verb{$Jac(x, dyn)} Compute the Jacobian at new values of the variables and dynamic parameters. Returns a vector of length \verb{$range * $domain} arranged so that the first \verb{$domain} elements correspond to the gradient of the first element of the range. The next \verb{$domain} elements correspond to the gradient of the second element of the range, and so on.
\item \verb{$Hes(x, dyn)} Compute the Hessian of the first element of the range at new values of the variables and dynamic parameters. Returns a vector of length \verb{$domain * $domain} where the \code{j*n + l} element corresponds to differentiating with respect to the \code{l}th element of the domain, then with respect to the \code{j}th element of the domain, with \code{n} the size of the domain.
\item \verb{$Jacobian(x)} Evaluate the Jacobian of the function at the current set of dynamic parameters.
\item \verb{$Hessiani(x, i)} Evaluate the Hessian for the \code{i}-th element of the range (where \code{i = 0, 1, ...}). Returns a vector arranged the same as \verb{$Hes()}.
\item \verb{$Hessian0(x)} Evaluate the Hessian for the first element of the range (like \verb{$Hes()} but uses the current values of the dynamic parameters). Returns a vector arranged the same as \verb{$Hes()}.
\item \verb{$Hessianw(x, w)} Evaluate the Hessian for a weighted sum of the range. Returns a vector arranged the same as \verb{$Hes()}.
\item \verb{$forward(q, x)} Perform forward mode evaluation for the specified Taylor coefficient order \code{q}. See the \href{https://cppad.readthedocs.io}{\code{CppAD} help} for more.
}
}

\section{Method Arguments}{
\itemize{
\item \code{x} A vector of independent variables.
\item \code{dyn} A vector of dynamic parameters.
\item \code{q} Taylor coefficient order for evaluating derivatives with \verb{$forward()}.
\item \code{i} Index of range result. \verb{i = 0, 1, ..., $range - 1}.
\item \code{bool} Either \code{TRUE} or \code{FALSE} to set \code{check_for_nan} behaviour using \verb{$set_check_for_nan()}.
\item \code{w} Weights assigned to each element of the range, for use with \verb{$Hessianw()}.
}
}

\section{Extends}{
Extends class \code{C++Object} from the \code{Rcpp} package (\code{\link[Rcpp:CppObject-class]{Rcpp::C++Object}}), which is a \verb{reference class}.
For those familiar with \verb{C++}, an object of class \code{Rcpp_ADFun} contains a pointer to a \code{CppAD} \code{ADFun} object.
}

\section{Introduction to CppAD Tapes}{
This package uses version 2024000.5 of the algorithmic differentiation library \code{CppAD} \insertCite{bell2023cp}{scorematchingad} to build score matching estimators.
Full help for \code{CppAD} can be found at \url{https://cppad.readthedocs.io/}.

When using \code{CppAD} one first creates a \emph{tape} of the basic (\emph{atomic}) operations of a function.
The atomic operations include multiplication, division, addition, sine, cosine, exponential and many more.
These tapes can then be used for evaluating the function and its derivatives, and generating further tapes through argument swapping, differentiation and composition (see for example \code{\link[=tape_swap]{tape_swap()}} and \code{\link[=tape_Jacobian]{tape_Jacobian()}}).
Tapes can have both \emph{independent} variables and \emph{dynamic} parameters, and the differentiation occurs with respect to the independent variables.
The atomic operations within a function are taped by following the function evaluation on example values for the variables and parameters, so care must be taken with any conditional (e.g. if-then) operations, and \href{https://cppad.readthedocs.io/}{\code{CppAD}} has a special tool for this called \code{CondExp} (short for \verb{conditional expressions}).

The result of taping, called a \emph{tape}, is exposed as an object of class \code{\link{Rcpp_ADFun}}, which contains a \code{CppAD} \code{ADFun} object.
Although the algorithmic differentiation is with respect to the independent variables, a new tape (see \code{\link[=tape_swap]{tape_swap()}}) can be created where the dynamic parameters become independent variables.
For the purposes of score matching, there are also \emph{fixed} parameters, which are the elements of the model's parameter vector that are given and not estimated.

The example values used for taping are saved in the \verb{$xtape} and \verb{$dyntape} properties of \code{\link{Rcpp_ADFun}} objects.
}

\section{Warning: multiple CPU}{
Each time a tape is evaluated the corresponding \verb{C++} object is altered. Parallel use of the same \code{ADFun} object thus requires care and is not tested. For now I recommend creating a new \code{ADFun} object for each CPU.
}

\section{Improvements}{
A few methods for \code{CppAD} \code{ADFun} objects are not yet available through \code{Rcpp_ADFun} objects. These ones would be nice to include:
\itemize{
\item \code{optimize()}
\item \code{function_name_set()} and \code{function_name_get()} working with \verb{$name}
\item \code{Reverse()}
}
}

\examples{
tape <- tape_uld_inbuilt("dirichlet", c(0.1, 0.4, 0.5), c(-0.5, -0.4, -0.2))
# Convenient evaluation
tape$eval(x = c(0.2, 0.3, 0.5), dyn = c(-0.1, -0.1, -0.5))
tape$Jac(x = c(0.2, 0.3, 0.5), dyn = c(-0.1, -0.1, -0.5))
matrix(tape$Hes(x = c(0.2, 0.3, 0.5), dyn = c(-0.1, -0.1, -0.5)), nrow = tape$domain)

# Properties
tape$domain
tape$range
tape$size_dyn_ind
tape$name
tape$xtape
tape$dyntape
tape$size_order
tape$new_dynamic(dyn = c(-0.1, -0.1, -0.5))
tape$parameter(0)
tape$set_check_for_nan(FALSE)
tape$get_check_for_nan()

# Further methods
tape$forward(order = 0, x = c(0.2, 0.3, 0.5))
tape$Jacobian(x = c(0.2, 0.3, 0.5))
tape$Hessiani(x = c(0.2, 0.3, 0.5), i = 0)
tape$Hessian0(x = c(0.2, 0.3, 0.5))
tape$Hessianw(x = c(0.2, 0.3, 0.5), w = c(2))
}
