% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/sar_habitat.R
\name{sar_habitat}
\alias{sar_habitat}
\title{Fit habitat SAR models}
\usage{
sar_habitat(data, modType = "power_log", con = NULL,
  logT = log, startPar = NULL)
}
\arguments{
\item{data}{A dataset in the form of a dataframe with at least three columns:
the first with island/site areas (A), the second with island / site habitat
diversity (H), and the third with the species richness of each island/site
(S).}

\item{modType}{What underlying SAR model form should be used. Should be one
of "power" (non-linear power), "logarithmic" (logarithmic SAR), or
"power_log" (log-log power; default).}

\item{con}{The constant to add to the species richness values in cases where
at least one of the islands has zero species.}

\item{logT}{The log-transformation to apply to the area and richness values.
Can be any of \code{log}(default), \code{log2} or \code{log10}.}

\item{startPar}{Optional starting parameter values (default =
NULL) for the jigsaw and Kallimanis models. Needs to be a
matrix of dimension [2,3], where the first row corresponds
to the jigsaw model, and the second to the Kallimanis model.
The columns correspond to the c, z, and d parameters,
respectively. Only used if
\code{modType = "power"} or \code{modType =
"logarithmic"}.}
}
\value{
A list of class "habitat" and "sars" with up to four elements, each
  holding one of the individual model fit objects (either \code{\link{nls}}
  or \code{\link{lm}} class objects). \code{\link{summary.sars}} provides a
  more user-friendly ouput (including a model summary table ranked by AICc
  and presenting the model coefficients, and R2 and information criteria
  values etc.) and \code{\link{plot.habitat}} provides a simple bar of
  information criteria weights. For the models fitted using non-linear
  regression, the R2 and adjusted R2 are 'pseudo R2' values and are
  calculated using the same approach as in the rest of the package (e.g.,
  \code{\link{sar_power}}).
  
  Note that if any of the models cannot be fitted - this is particularly the
  case when fitting the untransformed or logarithmic models which use
  non-linear regression (see above) - they are removed from the returned
  object.
}
\description{
Fit three SAR regression models that include habitat diversity:
the choros model, the Kallimanis model, and the jigsaw model.
}
\details{
These functions are described in more detail in the accompanying paper
  (Furness et al., 2023). The code to fit the models was also taken from this
  paper.
  
  Three habitat SAR models are available:
  \itemize{ 
  \item \strong{choros model}:  
  Proposes that species richness is better
  predicted by the product of habitat heterogeneity and area (S = c.(A.H)^z) 
  \item \strong{Kallimanis model}:  
  Proposes that increasing habitat heterogeneity increases species richness
  by increasing the slope (on a log-log plot) of the Arrhenius power model
  (S = c1.A^(z + d.H)) 
  \item \strong{jigsaw model}: 
  Models species richness in an area as the sum of the species richness
  values of several smaller component subareas, which can be visualised as
  pieces of a jigsaw puzzle, i.e., it partitions the species–area and
  species–heterogeneity scaling relationships (S = (c1.H^d).((A / H)^z)) }
  
  In addition to these three models, a simple 'non-habitat' SAR model is also
  fit, which varies depending on \code{modType}: the non-linear power, the
  logarithmic or the log-log power model.
  
  The untransformed (\code{modType = "power"}) and logarithmic (\code{modType
  = "logarithmic"}) models are fitted using non-linear regression and the
  \code{\link[minpack.lm]{nlsLM}} function. For the jigsaw and Kallimanis
  models in untransformed space, a grid search process is used
  to test multiple starting parameter values for the
  \code{\link[minpack.lm]{nlsLM}} function - see details in the
  documentation for \code{\link{sar_average}} - if multiple
  model fits are returned, the fit with the lowest \code{AIC}
  is returned. Providing starting parameter estimates for
  multiple datasets is tricky, and thus you may find the
  jigsaw and Kallimanis models cannot be fitted in
  untransformed space or with the logarithmic models. If this
  is the case, the \code{startPar} argument can be used to
  manually provide starting parameter values. The log-log
  models (\code{modType = "power_log"}) are all fitted using
  linear regression ( \code{\link{lm}} function).
  
  \code{sar_habitat()} uses the \code{\link[minpack.lm]{nlsLM}} from the
  \code{minpack.lm} package rather than \code{\link{nls}} as
  elsewhere in the package as we found that this resulted in
  better searches of the parameter space for the habitat
  models (and less convergence errors), particularly for the
  logarithmic models. \code{\link[minpack.lm]{nlsLM}} is a
  modified version of \code{\link{nls}} that uses the
  Levenberg-Marquardt fitting algorithm, but returns a
  standard \code{\link{nls}} object and thus all the normal
  subsequent \code{\link{nls}} functions can be used. Note
  also that occasionally a warning is returned of NaNs being
  present, normally relating to the jigsaw model (logarithmic
  version). We believe this mostly relates to models fitted
  during the optimisation process rather than the final
  returned model. Nonetheless, users are still recommended to
  check the convergence information of the returned model
  fits.
}
\note{
The jigsaw model is equivalent to the trivariate power-law model of 
Tjørve (2009), see Furness et al. (2023).

The jigsaw model (power-law form) cannot have a poorer fit than the choros or
power model based on RSS and thus R2. Comparing models using information
criteria is thus advised.
}
\examples{
data(habitat)
#Fit the models in log-log space
s <- sar_habitat(data = habitat, modType = "power_log", 
con = NULL, logT = log)
#Look at the model comparison summary
s2 <- summary(s)
s2
#Make a simple plot of AICc weights
plot(s, IC = "AICc", col = "darkred")

#Fit the logarithmic version of the models
s3 <- sar_habitat(data = habitat, modType = "logarithmic",
con = NULL, logT = log)
summary(s3)
plot(s, IC = "BIC", col = "darkblue")

#Provide starting parameter values for the jigsaw and
#Kallimanis models
SP2 <- t(matrix(rep(c(5, 1, 0.5),2), ncol = 2))
s <- sar_habitat(data = habitat, modType = "power",
                con = NULL, logT = log, startPar = SP2)
}
\references{
Furness, E.N., Saupe, E.E., Garwood, R.J., Mannion, P.D. &
  Sutton, M.D. (2023) The jigsaw model: a biogeographic model that partitions
  habitat heterogeneity from area. Frontiers of Biogeography, 15, e58477.
  
  Kallimanis, A.S., Mazaris, A.D., Tzanopoulos, J., Halley, J.M., Pantis,
  J.D., & Sgardelis, S.P. (2008) How does habitat diversity affect the
  species–area relationship? Global Ecology and Biogeography, 17, 532-538
  
  Tjørve, E. (2009) Shapes and functions of species– area curves (II): a
  review of new models and parameterizations. Journal of Biogeography, 36,
  1435-1445.
  
  Triantis, K.A., Mylonas, M., Lika, K. & Vardinoyannis, K. (2003) A model
  for the species-area-habitat relationship. Journal of Biogeography, 30,
  19–27.
}
\author{
Euan N. Furness and Thomas J. Matthews
}
