% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/griddata_class.R
\name{grid}
\alias{grid}
\title{R6 class holding sf grid data with data and analysis functions}
\description{
R6 class holding sf grid data with data and analysis functions

R6 class holding sf grid data with data and analysis functions
}
\details{
Grid data consists of the computational grid over the area of interest. Outcomes and
covariates are projected onto the grid, which can then be sent to the LGCP model.

If \code{zcol} is not specified then only the geometry is plotted, otherwise the covariates specified will be plotted.
The user can also use sf plotting functions on grid$grid_data directly.

Case counts are generated for each grid cell for each time period. The user
can specify the length of each time period; currently \code{day}, \code{week}, and \code{month}
are supported.

The user must also specify the number of time periods to include with the
\code{laglength} argument. The total number of time periods is the specified lag
length counting back from the most recent case. The columns in the output
will be named \code{t1}, \code{t2},... up to the lag length, where the highest number
is the most recent period.

\emph{Spatially-varying data only} \verb{cov_data`` is an sf object describing covariate values for a set of polygons over the area of interest. The values are mapped onto }grid_data\verb{. For each grid cell in }grid_data\verb{a weighted average of each covariate listed in}zcols\verb{is generated with weights either equal to the area of intersection of the grid cell and the polygons in}cov_data\verb{ (}weight_type="area"\verb{), or this area multiplied by the population density of the polygon for population weighted (}weight_type="pop"\verb{). Columns with the names in }zcols` are added to the output.

\emph{Temporally-varying only data} \code{cov_data} is a data frame with number of rows
equal to the number of time periods. One of the columns must be called \code{t} and
have values from 1 to the number of time periods. The other columns of the data
frame have the values of the covariates for each time period. See
\code{get_dow()} for day of week data. A total of
length(zcols)*(number of time periods) columns are added to the output: for each
covariate there will be columns appended with each time period number. For example,
\code{dayMon1}, \code{dayMon2}, etc.

\emph{Spatially and temporally varying data} There are two ways to add data that
vary both spatially and temporally. The final output for use in analysis must
have a column for each covariate and each time period with the same name appended
by the time period number, e.g. \code{covariateA1},\code{covariateA2},... If the covariate
values for different time periods are in separate sf objects, one can follow
the method for spatially-varying only data above and append the time period number
using the argument \code{t_label}. If the values for different time periods are in the same
sf object then they should be named as described above and then can be added
as for spatially-varying covariates, e.g. \code{zcols=c("covariateA1","covariateA2")}.

The grid data must contain columns \verb{t*}, giving the case
count in each time period (see \code{points_to_grid}), as well as any covariates to include in the model
(see \code{add_covariates}) and the population density.

Our statistical model is a Log Gaussian cox process,
whose realisation is observed on the Cartesian area of interest
A and time period T. The resulting data are relaisations of an inhomogeneous
Poisson process with stochastic intensity function \eqn{\{\lambda{s,t}:s\in A, t \in T\}}.
We specify a log-linear model for the intensity:

\deqn{\lambda(s,t) = r(s,t)exp(X(s,t)'\gamma + Z(s,t))}

where r(s,t) is a spatio-temporally varying Poisson offset.
X(s,t) is a length Q vector of covariates including an intercept and
Z(s,t) is a latent field. We use an auto-regressive specification for the
latent field, with spatial innovation in each field specified as a spatial
Gaussian process.

We use the fast and accurate approximation for fully Bayesian Gaussian
Processes proposed by Solin and Särkkä (1), using basis function
approximations based on approximation
via Laplace eigenfunctions for stationary covariance functions.
See references (1) and (2) for complete details. The approximation is a linear sum
of \code{m} eigenfunctions with the boundary conditions in each dimension \verb{[-L,L]}.
Coordinates in each dimension are scaled to \verb{[-1,1]}, so L represents the
proportionate extension of the analysis area.

\emph{Priors}
The priors should be provided as a list to the griddata object:\preformatted{griddata$priors <- list(
  prior_lscale=c(0,0.5),
  prior_var=c(0,0.5),
  prior_linpred_mean=c(-5,rep(0,7)),
  prior_linpred_sd=c(3,rep(1,7))
)
}

where these refer to the priors:
\code{prior_lscale}: the length scale parameter has a half-normal prior \eqn{N(a,b^2)I[0,\infty)}. The vector is \code{c(a,b)}.
\code{prior_var}: the standard deviation term has a half normal prior \eqn{\sigma ~ N(a,b^2)I[0,\infty)}. The vector is \code{c(a,b)}.
\code{prior_linpred_mean} and \code{prior_linpred_sd}: The parameters of the linear predictor.
If X is the nT x Q matrix of covariates, with the first column as ones for the intercept,
then the linear prediction contains the term \eqn{X'\gamma}. Each parameter in \eqn{\gamma} has prior
\eqn{\gamma_q ~ N(a_q,b_q^2)}.
\code{prior_linpred_mean} should be the vector \verb{(a_1,a_2,...,a_Q)} and
\code{prior_linpred_sd} should be \verb{(b_1,b_2,...,b_Q)}.

Three outputs can be extracted from the model fit, which will be added as columns to \code{grid_data}:

Predicted incidence: If type includes \code{pred} then \code{pred_mean_total} and
\code{pred_mean_total_sd} provide the
predicted mean total incidence and its standard deviation, respectively.
\code{pred_mean_pp} and \code{pred_mean_pp_sd} provide the predicted population
standardised incidence and its standard deviation.

Relative risk: if type includes \code{rr} then the relative risk is reported in
the columns \code{rr} and \code{rr_sd}. The relative risk here is the exponential
of the latent field, which describes the relative difference between
expexted mean and predicted mean incidence.

Incidence risk ratio: if type includes \code{irr} then the incidence rate ratio (IRR)
is reported in the columns \code{irr} and \code{irr_sd}. This is the ratio of the predicted
incidence in the last period (minus \code{t_lag}) to the predicted incidence in the
last period minus \code{irr_lag} (minus \code{t_lag}). For example, if the time period
is in days then setting \code{irr_lag} to 7 and leaving \code{t_lag=0} then the IRR
is the relative change in incidence in the present period compared to a week
prior.
}
\examples{

## ------------------------------------------------
## Method `grid$new`
## ------------------------------------------------

b1 = sf::st_sf(sf::st_sfc(sf::st_polygon(list(cbind(c(0,3,3,0,0),c(0,0,3,3,0))))))
g1 <- grid$new(b1,0.5)

## ------------------------------------------------
## Method `grid$plot`
## ------------------------------------------------

b1 = sf::st_sf(sf::st_sfc(sf::st_polygon(list(cbind(c(0,3,3,0,0),c(0,0,3,3,0))))))
g1 <- grid$new(b1,0.5)
g1$plot()

## ------------------------------------------------
## Method `grid$points_to_grid`
## ------------------------------------------------

b1 <- sf::st_sf(sf::st_sfc(sf::st_polygon(list(cbind(c(0,3,3,0,0),c(0,0,3,3,0))))))
g1 <- grid$new(b1,0.5)
dp <- data.frame(y=runif(10,0,3),x=runif(10,0,3),date=paste0("2021-01-",11:20))
dp <- create_points(dp,pos_vars = c('y','x'),t_var='date')
g1$points_to_grid(dp, laglength=5)

## ------------------------------------------------
## Method `grid$add_covariates`
## ------------------------------------------------

b1 <-  sf::st_sf(sf::st_sfc(sf::st_polygon(list(cbind(c(0,3,3,0,0),c(0,0,3,3,0))))))
g1 <- grid$new(b1,0.5)
cov1 <- grid$new(b1,0.8)
cov1$grid_data$cov <- runif(nrow(cov1$grid_data))
g1$add_covariates(cov1$grid_data,
                  zcols="cov",
                  verbose = FALSE)

## ------------------------------------------------
## Method `grid$get_dow`
## ------------------------------------------------

b1 <- sf::st_sf(sf::st_sfc(sf::st_polygon(list(cbind(c(0,3,3,0,0),c(0,0,3,3,0))))))
g1 <- grid$new(b1,0.5)
dp <- data.frame(y=runif(10,0,3),x=runif(10,0,3),date=paste0("2021-01-",11:20))
dp <- create_points(dp,pos_vars = c('y','x'),t_var='date')
g1$points_to_grid(dp, laglength=5)
g1$get_dow()

## ------------------------------------------------
## Method `grid$lgcp_fit`
## ------------------------------------------------

\dontrun{
b1 <- sf::st_sf(sf::st_sfc(sf::st_polygon(list(cbind(c(0,3,3,0,0),c(0,0,3,3,0))))))
g1 <- grid$new(b1,0.5)
dp <- data.frame(y=runif(10,0,3),x=runif(10,0,3),date=paste0("2021-01-",11:20))
dp <- create_points(dp,pos_vars = c('y','x'),t_var='date')
cov1 <- grid$new(b1,0.8)
cov1$grid_data$cov <- runif(nrow(cov1$grid_data))
g1$add_covariates(cov1,
                  zcols="cov",
                  verbose = FALSE)
g1$points_to_grid(dp, laglength=5)
g1$priors <- list(
  prior_lscale=c(0,0.5),
  prior_var=c(0,0.5),
  prior_linpred_mean=c(0),
  prior_linpred_sd=c(5)
  )
res <- g1$lgcp_fit(popdens="cov")
}

## ------------------------------------------------
## Method `grid$extract_preds`
## ------------------------------------------------

\dontrun{
b1 <- sf::st_sf(sf::st_sfc(sf::st_polygon(list(cbind(c(0,3,3,0,0),c(0,0,3,3,0))))))
g1 <- grid$new(b1,0.5)
dp <- data.frame(y=runif(10,0,3),x=runif(10,0,3),date=paste0("2021-01-",11:20))
dp <- create_points(dp,pos_vars = c('y','x'),t_var='date')
cov1 <- grid$new(b1,0.8)
cov1$grid_data$cov <- runif(nrow(cov1$grid_data))
g1$add_covariates(cov1,
                  zcols="cov",
                  verbose = FALSE)
g1$points_to_grid(dp, laglength=5)
g1$priors <- list(
  prior_lscale=c(0,0.5),
  prior_var=c(0,0.5),
  prior_linpred_mean=c(0),
  prior_linpred_sd=c(5)
  )
res <- g1$lgcp_fit(popdens="cov")
g1$extract_preds(res,
                 type=c("pred","rr"),
                 popdens="cov")
}

## ------------------------------------------------
## Method `grid$hotspots`
## ------------------------------------------------

\dontrun{
b1 <- sf::st_sf(sf::st_sfc(sf::st_polygon(list(cbind(c(0,3,3,0,0),c(0,0,3,3,0))))))
g1 <- grid$new(b1,0.5)
dp <- data.frame(y=runif(10,0,3),x=runif(10,0,3),date=paste0("2021-01-",11:20))
dp <- create_points(dp,pos_vars = c('y','x'),t_var='date')
cov1 <- grid$new(b1,0.8)
cov1$grid_data$cov <- runif(nrow(cov1$grid_data))
g1$add_covariates(cov1,
                  zcols="cov",
                  verbose = FALSE)
g1$points_to_grid(dp, laglength=5)
g1$priors <- list(
  prior_lscale=c(0,0.5),
  prior_var=c(0,0.5),
  prior_linpred_mean=c(0),
  prior_linpred_sd=c(5)
  )
res <- g1$lgcp_fit(popdens="cov")
g1$hotspots(res,
            incidence.threshold=1,
            popdens="cov")
}

## ------------------------------------------------
## Method `grid$aggregate_output`
## ------------------------------------------------

\dontrun{
b1 <- sf::st_sf(sf::st_sfc(sf::st_polygon(list(cbind(c(0,3,3,0,0),c(0,0,3,3,0))))))
g1 <- grid$new(b1,0.5)
dp <- data.frame(y=runif(10,0,3),x=runif(10,0,3),date=paste0("2021-01-",11:20))
dp <- create_points(dp,pos_vars = c('y','x'),t_var='date')
cov1 <- grid$new(b1,0.8)
cov1$grid_data$cov <- runif(nrow(cov1$grid_data))
g1$add_covariates(cov1,
                  zcols="cov",
                  verbose = FALSE)
g1$points_to_grid(dp, laglength=5)
g1$priors <- list(
  prior_lscale=c(0,0.5),
  prior_var=c(0,0.5),
  prior_linpred_mean=c(0),
  prior_linpred_sd=c(5)
  )
res <- g1$lgcp_fit(popdens="cov")
g1$extract_preds(res,
                 type=c("pred","rr"),
                 popdens="cov")
new1 <- g1$aggregate_output(cov1,
                            zcols="rr")
}

## ------------------------------------------------
## Method `grid$scale_conversion_factor`
## ------------------------------------------------

b1 = sf::st_sf(sf::st_sfc(sf::st_polygon(list(cbind(c(0,3,3,0,0),c(0,0,3,3,0))))))
g1 <- grid$new(b1,0.5)
g1$scale_conversion_factor()
}
\references{
(1) Solin A, Särkkä S. Hilbert space methods for reduced-rank Gaussian
process regression. Stat Comput. 2020;30:419–46.
doi:10.1007/s11222-019-09886-w.

(2) Riutort-Mayol G, Bürkner P-C, Andersen MR, Solin A, Vehtari A.
Practical Hilbert space approximate Bayesian Gaussian processes for
probabilistic programming. 2020. http://arxiv.org/abs/2004.11408.
}
\seealso{
\link[rts2]{create_points}

points_to_grid, add_covariates
}
\section{Public fields}{
\if{html}{\out{<div class="r6-fields">}}
\describe{
\item{\code{grid_data}}{sf object with the grid data}

\item{\code{priors}}{list of prior distributions for the analysis}

\item{\code{boundary}}{sf object showing the boundary of the area of interest}
}
\if{html}{\out{</div>}}
}
\section{Methods}{
\subsection{Public methods}{
\itemize{
\item \href{#method-new}{\code{grid$new()}}
\item \href{#method-print}{\code{grid$print()}}
\item \href{#method-plot}{\code{grid$plot()}}
\item \href{#method-points_to_grid}{\code{grid$points_to_grid()}}
\item \href{#method-add_covariates}{\code{grid$add_covariates()}}
\item \href{#method-get_dow}{\code{grid$get_dow()}}
\item \href{#method-lgcp_fit}{\code{grid$lgcp_fit()}}
\item \href{#method-extract_preds}{\code{grid$extract_preds()}}
\item \href{#method-hotspots}{\code{grid$hotspots()}}
\item \href{#method-aggregate_output}{\code{grid$aggregate_output()}}
\item \href{#method-scale_conversion_factor}{\code{grid$scale_conversion_factor()}}
\item \href{#method-clone}{\code{grid$clone()}}
}
}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-new"></a>}}
\if{latex}{\out{\hypertarget{method-new}{}}}
\subsection{Method \code{new()}}{
Create a new griddata object

Produces a regular grid over an area of interest as an sf object

Given a contiguous boundary describing an area of interest, which is stored as an sf
object of a regular grid within the limits of the boundary at \verb{$grid_data}. The boundary
is also stored in the object as \verb{$boundary}
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{grid$new(boundary, cellsize)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{boundary}}{An sf object containing one polygon describing the area of interest}

\item{\code{cellsize}}{The dimension of the grid cells}
}
\if{html}{\out{</div>}}
}
\subsection{Returns}{
NULL
}
\subsection{Examples}{
\if{html}{\out{<div class="r example copy">}}
\preformatted{b1 = sf::st_sf(sf::st_sfc(sf::st_polygon(list(cbind(c(0,3,3,0,0),c(0,0,3,3,0))))))
g1 <- grid$new(b1,0.5)
}
\if{html}{\out{</div>}}

}

}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-print"></a>}}
\if{latex}{\out{\hypertarget{method-print}{}}}
\subsection{Method \code{print()}}{
Prints the $grid_data sf object
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{grid$print()}\if{html}{\out{</div>}}
}

}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-plot"></a>}}
\if{latex}{\out{\hypertarget{method-plot}{}}}
\subsection{Method \code{plot()}}{
Plots the grid data
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{grid$plot(zcol)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{zcol}}{Vector of strings specifying names of columns of \code{grid_data} to plot}
}
\if{html}{\out{</div>}}
}
\subsection{Returns}{
A plot
}
\subsection{Examples}{
\if{html}{\out{<div class="r example copy">}}
\preformatted{b1 = sf::st_sf(sf::st_sfc(sf::st_polygon(list(cbind(c(0,3,3,0,0),c(0,0,3,3,0))))))
g1 <- grid$new(b1,0.5)
g1$plot()
}
\if{html}{\out{</div>}}

}

}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-points_to_grid"></a>}}
\if{latex}{\out{\hypertarget{method-points_to_grid}{}}}
\subsection{Method \code{points_to_grid()}}{
Generates case counts of points over the grid

Counts the number of cases in each time period in each grid cell

Given the sf object with the point locations and date output from
\code{create_points()}, the functions will add columns to \code{grid_data} indicating
the case count in each cell in each time period.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{grid$points_to_grid(
  point_data,
  t_win = c("day"),
  laglength = 14,
  verbose = TRUE
)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{point_data}}{sf object describing the point location of cases with a column
\code{t} of the date of the case in YYYY-MM-DD format. See \link[rts2]{create_points}}

\item{\code{t_win}}{character string. One of "day", "week", or "month" indicating the
length of the time windows in which to count cases}

\item{\code{laglength}}{integer The number of time periods to include counting back from the most
recent time period}

\item{\code{verbose}}{Logical indicating whether to report detailed output}
}
\if{html}{\out{</div>}}
}
\subsection{Returns}{
NULL
}
\subsection{Examples}{
\if{html}{\out{<div class="r example copy">}}
\preformatted{b1 <- sf::st_sf(sf::st_sfc(sf::st_polygon(list(cbind(c(0,3,3,0,0),c(0,0,3,3,0))))))
g1 <- grid$new(b1,0.5)
dp <- data.frame(y=runif(10,0,3),x=runif(10,0,3),date=paste0("2021-01-",11:20))
dp <- create_points(dp,pos_vars = c('y','x'),t_var='date')
g1$points_to_grid(dp, laglength=5)
}
\if{html}{\out{</div>}}

}

}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-add_covariates"></a>}}
\if{latex}{\out{\hypertarget{method-add_covariates}{}}}
\subsection{Method \code{add_covariates()}}{
Adds covariate data to the grid

Maps spatial, temporal, or spatio-temporal covariate data onto the grid
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{grid$add_covariates(
  cov_data,
  zcols,
  weight_type = "area",
  popdens = NULL,
  verbose = TRUE,
  t_label = NULL
)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{cov_data}}{sf object or data.frame. See details.}

\item{\code{zcols}}{vector of character strings with the names of the columns of \code{cov_data}
to include}

\item{\code{weight_type}}{character string. Either "area" for area-weighted average or "pop"
for population-weighted average}

\item{\code{popdens}}{character string. The name of the column in \code{cov_data} with the
population density. Required if weight_type="pop"}

\item{\code{verbose}}{logical. Whether to provide a progress bar}

\item{\code{t_label}}{integer. If adding spatio-temporally varying data by time period,
this time label should be appended to the column name. See details.}
}
\if{html}{\out{</div>}}
}
\subsection{Returns}{
NULL
}
\subsection{Examples}{
\if{html}{\out{<div class="r example copy">}}
\preformatted{b1 <-  sf::st_sf(sf::st_sfc(sf::st_polygon(list(cbind(c(0,3,3,0,0),c(0,0,3,3,0))))))
g1 <- grid$new(b1,0.5)
cov1 <- grid$new(b1,0.8)
cov1$grid_data$cov <- runif(nrow(cov1$grid_data))
g1$add_covariates(cov1$grid_data,
                  zcols="cov",
                  verbose = FALSE)
}
\if{html}{\out{</div>}}

}

}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-get_dow"></a>}}
\if{latex}{\out{\hypertarget{method-get_dow}{}}}
\subsection{Method \code{get_dow()}}{
Generate day of week data

Create data frame with day of week indicators

Generates a data frame with indicator
variables for each day of the week for use in the \code{add_covariates()} function.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{grid$get_dow()}\if{html}{\out{</div>}}
}

\subsection{Returns}{
data.frame with columns \code{t}, \code{day}, and \code{dayMon} to \code{daySun}
}
\subsection{Examples}{
\if{html}{\out{<div class="r example copy">}}
\preformatted{b1 <- sf::st_sf(sf::st_sfc(sf::st_polygon(list(cbind(c(0,3,3,0,0),c(0,0,3,3,0))))))
g1 <- grid$new(b1,0.5)
dp <- data.frame(y=runif(10,0,3),x=runif(10,0,3),date=paste0("2021-01-",11:20))
dp <- create_points(dp,pos_vars = c('y','x'),t_var='date')
g1$points_to_grid(dp, laglength=5)
g1$get_dow()
}
\if{html}{\out{</div>}}

}

}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-lgcp_fit"></a>}}
\if{latex}{\out{\hypertarget{method-lgcp_fit}{}}}
\subsection{Method \code{lgcp_fit()}}{
Fit an approximate log-Gaussian Cox Process model

Fit an approximate log-Gaussian Cox Process model
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{grid$lgcp_fit(
  popdens,
  covs = NULL,
  m = 10,
  L = 1.5,
  dir = NULL,
  iter_warmup = 500,
  iter_sampling = 500,
  chains = 3,
  parallel_chains = 3,
  verbose = TRUE,
  use_cmdstanr = FALSE,
  ...
)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{popdens}}{character string. Name of the population density column}

\item{\code{covs}}{vector of character string. Base names of the covariates to
include. For temporally-varying covariates only the stem is required and not
the individual column names for each time period (e.g. \code{dayMon} and not \code{dayMon1},
\code{dayMon2}, etc.)}

\item{\code{m}}{integer. Number of basis functions. See Details.}

\item{\code{L}}{integer. Boundary condition as proportionate extension of area, e.g.
\code{L=2} is a doubling of the analysis area. See Details.}

\item{\code{dir}}{character string. Directory to save ouptut.}

\item{\code{iter_warmup}}{integer. Number of warmup iterations}

\item{\code{iter_sampling}}{integer. Number of sampling iterations}

\item{\code{chains}}{integer. Number of chains}

\item{\code{parallel_chains}}{integer. Number of parallel chains}

\item{\code{verbose}}{logical. Provide feedback on progress}

\item{\code{use_cmdstanr}}{logical. Defaults to false. If true then cmdstanr will be used
instead of rstan.}

\item{\code{...}}{additional options to pass to `$sample()``, see \link[cmdstanr]{sample}}

\item{\code{priors}}{list. See Details}
}
\if{html}{\out{</div>}}
}
\subsection{Returns}{
A \link[rstan]{stanfit} or a \link[cmdstanr]{CmdStanMCMC} object
}
\subsection{Examples}{
\if{html}{\out{<div class="r example copy">}}
\preformatted{\dontrun{
b1 <- sf::st_sf(sf::st_sfc(sf::st_polygon(list(cbind(c(0,3,3,0,0),c(0,0,3,3,0))))))
g1 <- grid$new(b1,0.5)
dp <- data.frame(y=runif(10,0,3),x=runif(10,0,3),date=paste0("2021-01-",11:20))
dp <- create_points(dp,pos_vars = c('y','x'),t_var='date')
cov1 <- grid$new(b1,0.8)
cov1$grid_data$cov <- runif(nrow(cov1$grid_data))
g1$add_covariates(cov1,
                  zcols="cov",
                  verbose = FALSE)
g1$points_to_grid(dp, laglength=5)
g1$priors <- list(
  prior_lscale=c(0,0.5),
  prior_var=c(0,0.5),
  prior_linpred_mean=c(0),
  prior_linpred_sd=c(5)
  )
res <- g1$lgcp_fit(popdens="cov")
}
}
\if{html}{\out{</div>}}

}

}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-extract_preds"></a>}}
\if{latex}{\out{\hypertarget{method-extract_preds}{}}}
\subsection{Method \code{extract_preds()}}{
Extract predictions

Extract incidence and relative risk predictions
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{grid$extract_preds(
  stan_fit,
  type = c("pred", "rr", "irr"),
  irr.lag = NULL,
  t.lag = 0,
  popdens = NULL
)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{stan_fit}}{A \link[rstan]{stanfit} or \link[cmdstanr]{CmdStanMCMC} object.
Output of \code{lgcp_fit()}}

\item{\code{type}}{Vector of character strings. Any combination of "pred", "rr", and "irr", which are,
posterior mean incidence (overall and population standardised), relative risk,
and incidence rate ratio, respectively.}

\item{\code{irr.lag}}{integer. If "irr" is requested as \code{type} then the number of time
periods lag previous the ratio is in comparison to}

\item{\code{t.lag}}{integer. Extract predictions for previous time periods.}

\item{\code{popdens}}{character string. Name of the column in \code{grid_data} with the
population density data}
}
\if{html}{\out{</div>}}
}
\subsection{Returns}{
NULL
}
\subsection{Examples}{
\if{html}{\out{<div class="r example copy">}}
\preformatted{\dontrun{
b1 <- sf::st_sf(sf::st_sfc(sf::st_polygon(list(cbind(c(0,3,3,0,0),c(0,0,3,3,0))))))
g1 <- grid$new(b1,0.5)
dp <- data.frame(y=runif(10,0,3),x=runif(10,0,3),date=paste0("2021-01-",11:20))
dp <- create_points(dp,pos_vars = c('y','x'),t_var='date')
cov1 <- grid$new(b1,0.8)
cov1$grid_data$cov <- runif(nrow(cov1$grid_data))
g1$add_covariates(cov1,
                  zcols="cov",
                  verbose = FALSE)
g1$points_to_grid(dp, laglength=5)
g1$priors <- list(
  prior_lscale=c(0,0.5),
  prior_var=c(0,0.5),
  prior_linpred_mean=c(0),
  prior_linpred_sd=c(5)
  )
res <- g1$lgcp_fit(popdens="cov")
g1$extract_preds(res,
                 type=c("pred","rr"),
                 popdens="cov")
}
}
\if{html}{\out{</div>}}

}

}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-hotspots"></a>}}
\if{latex}{\out{\hypertarget{method-hotspots}{}}}
\subsection{Method \code{hotspots()}}{
Hotspots

Generate hotspot probabilities

Given a definition of a hotspot in terms of threshold(s) for incidence,
relative risk, and/or incidence rate ratio, returns the probabilities
each area is a "hotspot". See Details of \code{extract_preds}. Columns
will be added to \code{grid_data}
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{grid$hotspots(
  stan_fit,
  incidence.threshold = NULL,
  irr.threshold = NULL,
  irr.lag = NULL,
  rr.threshold = NULL,
  popdens,
  col_label = NULL
)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{stan_fit}}{A \link[rstan]{stanfit} or \link[cmdstanr]{CmdStanMCMC} object.
Output of \code{lgcp_fit()}}

\item{\code{incidence.threshold}}{Numeric. Threshold of population standardised incidence
above which an area is a hotspot}

\item{\code{irr.threshold}}{Numeric. Threshold of incidence rate ratio
above which an area is a hotspot.}

\item{\code{irr.lag}}{integer. Lag of time period to calculate the incidence rate ratio.
Only required if \code{irr.threshold} is not \code{NULL}.}

\item{\code{rr.threshold}}{numeric. Threshold of local relative risk
above which an area is a hotspot}

\item{\code{popdens}}{character string. Name of variable in \code{grid_data}
specifying the population density. Needed if \code{incidence.threshold} is not
\code{NULL}}

\item{\code{col_label}}{character string. If not NULL then the name of the column
for the hotspot probabilities.}
}
\if{html}{\out{</div>}}
}
\subsection{Returns}{
NULL
}
\subsection{Examples}{
\if{html}{\out{<div class="r example copy">}}
\preformatted{\dontrun{
b1 <- sf::st_sf(sf::st_sfc(sf::st_polygon(list(cbind(c(0,3,3,0,0),c(0,0,3,3,0))))))
g1 <- grid$new(b1,0.5)
dp <- data.frame(y=runif(10,0,3),x=runif(10,0,3),date=paste0("2021-01-",11:20))
dp <- create_points(dp,pos_vars = c('y','x'),t_var='date')
cov1 <- grid$new(b1,0.8)
cov1$grid_data$cov <- runif(nrow(cov1$grid_data))
g1$add_covariates(cov1,
                  zcols="cov",
                  verbose = FALSE)
g1$points_to_grid(dp, laglength=5)
g1$priors <- list(
  prior_lscale=c(0,0.5),
  prior_var=c(0,0.5),
  prior_linpred_mean=c(0),
  prior_linpred_sd=c(5)
  )
res <- g1$lgcp_fit(popdens="cov")
g1$hotspots(res,
            incidence.threshold=1,
            popdens="cov")
}
}
\if{html}{\out{</div>}}

}

}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-aggregate_output"></a>}}
\if{latex}{\out{\hypertarget{method-aggregate_output}{}}}
\subsection{Method \code{aggregate_output()}}{
Aggregate output

Aggregate \code{lgcp_fit} output to another geography
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{grid$aggregate_output(
  new_geom,
  zcols,
  weight_type = "area",
  popdens = NULL,
  verbose = TRUE
)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{new_geom}}{sf object. A set of polygons covering the same area as \code{boundary}}

\item{\code{zcols}}{vector of character strings. Names of the variables in \code{grid_data} to
map to the new geography}

\item{\code{weight_type}}{character string, either "area" or "pop" for area-weighted
or population weighted averaging, respectively}

\item{\code{popdens}}{character string. If \code{weight_type} is equal to "pop" then the
name of the column in \code{grid_data} with population density data}

\item{\code{verbose}}{logical. Whether to provide progress bar.}
}
\if{html}{\out{</div>}}
}
\subsection{Returns}{
An \code{sf} object identical to \code{new_geom} with additional columns with the
variables specified in \code{zcols}
}
\subsection{Examples}{
\if{html}{\out{<div class="r example copy">}}
\preformatted{\dontrun{
b1 <- sf::st_sf(sf::st_sfc(sf::st_polygon(list(cbind(c(0,3,3,0,0),c(0,0,3,3,0))))))
g1 <- grid$new(b1,0.5)
dp <- data.frame(y=runif(10,0,3),x=runif(10,0,3),date=paste0("2021-01-",11:20))
dp <- create_points(dp,pos_vars = c('y','x'),t_var='date')
cov1 <- grid$new(b1,0.8)
cov1$grid_data$cov <- runif(nrow(cov1$grid_data))
g1$add_covariates(cov1,
                  zcols="cov",
                  verbose = FALSE)
g1$points_to_grid(dp, laglength=5)
g1$priors <- list(
  prior_lscale=c(0,0.5),
  prior_var=c(0,0.5),
  prior_linpred_mean=c(0),
  prior_linpred_sd=c(5)
  )
res <- g1$lgcp_fit(popdens="cov")
g1$extract_preds(res,
                 type=c("pred","rr"),
                 popdens="cov")
new1 <- g1$aggregate_output(cov1,
                            zcols="rr")
}
}
\if{html}{\out{</div>}}

}

}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-scale_conversion_factor"></a>}}
\if{latex}{\out{\hypertarget{method-scale_conversion_factor}{}}}
\subsection{Method \code{scale_conversion_factor()}}{
Returns scale conversion factor

Coordinates are scaled to \verb{[-1,1]} for \code{lgcp_fit()}. This function
returns the scaling factor for this conversion.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{grid$scale_conversion_factor()}\if{html}{\out{</div>}}
}

\subsection{Returns}{
numeric
}
\subsection{Examples}{
\if{html}{\out{<div class="r example copy">}}
\preformatted{b1 = sf::st_sf(sf::st_sfc(sf::st_polygon(list(cbind(c(0,3,3,0,0),c(0,0,3,3,0))))))
g1 <- grid$new(b1,0.5)
g1$scale_conversion_factor()
}
\if{html}{\out{</div>}}

}

}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-clone"></a>}}
\if{latex}{\out{\hypertarget{method-clone}{}}}
\subsection{Method \code{clone()}}{
The objects of this class are cloneable with this method.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{grid$clone(deep = FALSE)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{deep}}{Whether to make a deep clone.}
}
\if{html}{\out{</div>}}
}
}
}
