% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/generate_items_parameters.R
\name{generate_slopes}
\alias{generate_slopes}
\title{Generating parameters of items - slopes (discrimination)}
\usage{
generate_slopes(
  nItems,
  scoringMatrix,
  ...,
  FUN = identity,
  nReversed = 0L,
  reverseTraits = "i",
  reverseIndep = FALSE
)
}
\arguments{
\item{nItems}{number of items for which slopes will be generated}

\item{scoringMatrix}{\emph{scoring matrix} that will be used for the
generated items, specifically generated with
\code{\link{make_scoring_matrix_aem}} or
\code{\link{make_scoring_matrix_trivial}}}

\item{...}{arguments that will be passed to \code{FUN}; should rather be
named and in typical applications should be numeric vectors of the length of
one or of the number of columns of the \code{scoringMatrix}}

\item{FUN}{function that will be used to generate slopes, typically}

\item{nReversed}{number of \emph{reversed} (\emph{revers-keyed}) items, i.e.
items that}

\item{reverseTraits}{character vector containing names of traits for which
items should be \emph{reversed} (if \code{nReversed > 0}); default value is
"i", that is the name assigned to the trait that is assumed to describe
\emph{the trait that scale is supposed to measure} (i.e. trait that is not
related with response styles) by \code{\link{make_scoring_matrix_aem}} when
called with argument \code{sequence = "simultaneous"}}

\item{reverseIndep}{logical value indicating whether sampling items that are
\emph{reversed} should be performed independently for each trait (given by
\code{reverseTraits}) or for all the traits simultaneously (this argument
is only considered if \code{nReversed > 0} and there is more than one trait
given by \code{reverseTraits})
\code{\link[stats]{Normal}}, \code{\link[stats]{Lognormal}},
\code{\link[stats]{Uniform}} or \code{\link[truncnorm]{rtruncnorm}}
- see details}
}
\value{
Matrix of \code{nItems} rows and number of columns equal to the
length of vectors provided by \code{...}.
}
\description{
Function generates a matrix of items' slope (discrimination)
parameters.
}
\details{
There are two typical ways of using this function (compare examples
below):
\itemize{
  \item{If one wants to generate constant slopes for all the items, one
        should simply provide a vector which elements describe slope for
        each of the consecutive traits. You may name elements of this vector
        to get this names assigned to columns of the returned matrix.}
  \item{If one wants to sample slopes using specific distribution, one
        must provide a sampling function using \code{FUN} argument.
        Arguments that should be passed to this function one should provide
        as additional arguments to a \code{generate_slopes} call. If one
        wants to generate slopes for more than one trait, one needs to
        provide at least one of these arguments as a vector with as many
        elements, as there are traits (even if all of its elements would be
        equal). Moreover one may name elements of this vector, to get these
        names assigned to columns of the returned matrix. Other arguments may
        be provided as single values if one wants to hold them constant
        across traits.}
}
In a case slopes are sampled, this is done independently for each of the
traits. The same sampling function is used for each trait and there is no
way to change this (but one may still use \code{\link{cbind}} to collapse
results of several separate \code{generate_slopes} calls to achieve this
goal).
}
\examples{
# 5 items with slopes equals 1 on all the latent traits
sM <- make_scoring_matrix_aem(5, sequence = "mae")
generate_slopes(5, sM, 1)
# 10 items with slopes on all the latent traits generated from a uniform
# distribution with limits 0.5 and 3
sM <- make_scoring_matrix_aem(5, sequence = "mae")
generate_slopes(10, sM, FUN = runif, min = 0.5, max = 3)
# 10 items with slopes generated from a normal distributions with parameters set to:
# trait 'm' (i.e. the first column in the scoring matrix): mean 1, sd 0.2
# trait 'a' (i.e. the second column in the scoring matrix): mean 1.2, sd 0.3
# trait 'e' (i.e. the third column in the scoring matrix): mean 1.5, sd 0.5
sM <- make_scoring_matrix_aem(5, sequence = "mae")
generate_slopes(10, sM, FUN = rnorm,
                mean = c(1, 1.2, 1.5),
                sd = c(0.2, 0.3, 0.5))
# 10 items with slopes generated from a truncated-normal distributions with
# parameters set to:
# trait 'm' (i.e. the first column in the scoring matrix): mean 1, sd 0.5
# trait 'a' (i.e. the second column in the scoring matrix): mean 1.2, sd 0.7
# trait 'e' (i.e. the third column in the scoring matrix): mean 1.5, sd 1
# and bounds equals to 0.5 (lower) and 2.5 (upper) for all the traits
sM <- make_scoring_matrix_aem(5, sequence = "mae")
require(truncnorm)
generate_slopes(10, sM, FUN = rtruncnorm,
                mean = c(1, 1.2, 1.5),
                sd = c(0.5, 0.7, 1),
                a = 0.5,
                b = 2.5)
# 10 items with slopes generated from a normal distributions with mean of 1
# and standard deviation of 0.2 with half of the items "reverse-keyed" on
# the trait "i"
sM <- make_scoring_matrix_aem(5, sequence = "simultaneous")
generate_slopes(10, sM, FUN = rnorm, mean = 1, sd = 0.2,
                nReversed = 5, reverseTraits = "i")
}
\seealso{
\code{\link{generate_intercepts}}, \code{\link{make_test}}
}
