% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/ext_filter.R
\name{ext_filter}
\alias{ext_filter}
\alias{cql2_json}
\alias{cql2_text}
\title{Filter extension}
\usage{
ext_filter(q, expr, lang = NULL, crs = NULL)

cql2_json(expr)

cql2_text(expr)
}
\arguments{
\item{q}{a \code{RSTACQuery} object expressing a STAC query
criteria.}

\item{expr}{a valid R expression to be translated to CQL2 (see details).}

\item{lang}{a character value indicating which CQL2 representation
to be used. It can be either \code{"cql2-text"} (for plain text) or
\code{"cql2-json"} (for JSON format). If \code{NULL} (default), \code{"cql2-text"} is
used for HTTP \code{GET} requests and \code{"cql2-json"} for \code{POST} requests.}

\item{crs}{an optional character value informing the coordinate reference
system used by geometry objects. If \code{NULL} (default), STAC services assume
\code{"WGS 84"}.}
}
\value{
A \code{RSTACQuery} object  with the subclass \code{ext_filter} containing
all request parameters to be passed to \code{get_request()} or
\code{post_request()} function.
}
\description{
\ifelse{html}{\href{https://lifecycle.r-lib.org/articles/stages.html#experimental}{\figure{lifecycle-experimental.svg}{options: alt='[Experimental]'}}}{\strong{[Experimental]}}
\code{ext_filter()} implements Common Query Language (CQL2) filter extension
on \code{rstac}. This extension expands the filter capabilities providing a
query language to construct more complex expressions. CQL2 is an OGC
standard and defines how filters can be constructed. It supports predicates
for standard data types like strings, numbers, and boolean as well as
for spatial geometries (point, lines, polygons) and temporal
data (instants and intervals).

\ifelse{html}{\href{https://lifecycle.r-lib.org/articles/stages.html#experimental}{\figure{lifecycle-experimental.svg}{options: alt='[Experimental]'}}}{\strong{[Experimental]}}
\code{cql2_json()} and \code{cql2_text()} are helper functions that can be used
to show how expressions are converted into CQL2 standard, either
JSON or TEXT formats.

\code{rstac} translates R expressions to CQL2, allowing users to express their
filter criteria using R language. For more details on how to create
CQL2 expressions in \code{rstac}. See the details section.
}
\details{
To allow users to express filter criteria in R language, \code{rstac} takes
advantage of the abstract syntax tree (AST) to translate R expressions
to CQL2 expressions. The following topics describe the correspondences
between \code{rstac} expressions and CQL2 operators.
\subsection{Standard comparison operators}{
\itemize{
\item \code{==}, \code{>=}, \code{<=}, \code{>}, \code{<}, and \code{!=} operators correspond to
\code{=}, \code{>=}, \code{<=}, \code{>}, \code{<}, and \verb{<>} in CQL2, respectively.
\item function \code{is_null(a)} and \code{!is_null(a)} corresponds to \verb{a IS NULL} and
\verb{a IS NOT NULL} CQL2 operators, respectively.
}
}

\subsection{Advanced comparison operators}{
\itemize{
\item \code{a \%like\% b} corresponds to CQL2 \verb{a LIKE b}, \code{a} and \code{b} \code{strings} values.
\item \code{between(a, b, c)} corresponds to CQL2 \verb{a BETWEEN b AND c}, where
\code{b} and \code{c} \code{integer} values.
\item \code{a \%in\% b} corresponds to CQL2 \verb{a IN (b)}, where \code{b} should be
a list of values of the same type as \code{a}.
}
}

\subsection{Spatial operators}{
\itemize{
\item functions \code{s_intersects(a, b)}, \code{s_touches(a, b)}, \code{s_within(a, b)},
\code{s_overlaps(a, b)}, \code{s_crosses(a, b)}, and \code{s_contains(a, b)} corresponds
to CQL2 \code{S_INTERSECTS(a, b)}, \code{S_TOUCHES(a, b)}, \code{S_WITHIN(a, b)},
\code{S_OVERLAPS(a, b)}, \code{S_CROSSES(a, b)}, and \code{S_CONTAINS(a, b)} operators,
respectively. Here, \code{a} and \code{b} should be \code{geometry} objects. \code{rstac}
accepts \code{sf}, \code{sfc}, \code{sfg}, \code{list} (representing GeoJSON objects), or
\code{character} (representing either GeoJSON or WKT).
\item \strong{NOTE}: All of the above spatial object types, except for the
\code{character}, representing a WKT, may lose precision due to numeric
truncation when R converts numbers to JSON text. WKT strings are
sent "as is" to the service. Therefore, the only way for users to
retain precision on spatial objects is to represent them as a WKT
string. However, user can control numeric precision using the
\code{options(stac_digits = ...)}. The default value is 15 digits.
}
}

\subsection{Temporal operators}{
\itemize{
\item functions \code{date(a)}, \code{timestamp(a)}, and \code{interval(a, b)} corresponds to
CQL2 \code{DATE(a)}, \code{TIMESTAMP(a)}, and \code{INTERVAL(a, b)} operators,
respectively. These functions create literal \code{temporal} values.
The first two define an \code{instant} type, and the third an \code{interval} type.
\item functions \code{t_after(a, b)}, \code{t_before(a, b)}, \code{t_contains(a, b)},
\code{t_disjoint(a, b)}, \code{t_during(a, b)}, \code{t_equals(a, b)},
\code{t_finishedby(a, b)}, \code{t_finishes(a, b)}, \code{t_intersects(a, b)},
\code{t_meets(a, b)}, \code{t_meet(a, b)}, \code{t_metby(a, b)}, \code{t_overlappedby(a, b)},
\code{t_overlaps(a, b)}, \code{t_startedby(a, b)}, and \code{t_starts(a, b)} corresponds
to CQL2 \code{T_AFTER(a, b)}, \code{T_BEFORE(a, b)}, \code{T_CONTAINS(a, b)},
\code{T_DISJOINT(a, b)}, \code{T_DURING(a, b)}, \code{T_EQUALS(a, b)},
\code{T_FINISHEDBY(a, b)}, \code{T_FINISHES(a, b)}, \code{T_INTERSECTS(a, b)},
\code{T_MEETS(a, b)}, \code{T_MEET(a, b)}, \code{T_METBY(a, b)}, \code{T_OVERLAPPEDBY(a, b)},
\code{T_OVERLAPS(a, b)}, \code{T_STARTEDBY(a, b)}, and \code{T_STARTS(a, b)} operators,
respectively. Here, \code{a} and \code{b} are \code{temporal} values (\code{instant} or
\code{interval}, depending on function).
}
}

\subsection{Array Operators}{
\itemize{
\item R unnamed lists (or vectors of size > 1) are translated to arrays by
\code{rstac}. \code{list()} and \code{c()} functions always create \code{array} values
in CQL2 context, no matter the number of its arguments.
\item functions \code{a_equals(a, b)}, \code{a_contains(a, b)}, \code{a_containedby(a, b)},
and \code{a_overlaps(a, b)} corresponds to CQL2 \code{A_EQUALS(a, b)},
\code{A_CONTAINS(a, b)}, \code{A_CONTAINEDBY(a, b)}, and \code{A_OVERLAPS(a, b)}
operators, respectively. Here, \code{a} and \code{b} should be \code{arrays}.
}
}
}
\note{
The specification states that double-quoted identifiers should be
interpreted as properties. However, the R language does not distinguish
double quote from single quote strings. The right way to represent
double quoted properties in R is to use the escape character (\verb{), for example }"date"`.
}
\examples{
\dontrun{
# Standard comparison operators in rstac:
# Creating a stac search query
req <- rstac::stac("https://planetarycomputer.microsoft.com/api/stac/v1") \%>\%
  rstac::stac_search(limit = 5)

# Equal operator '=' with collection property
req \%>\% ext_filter(collection == "sentinel-2-l2a") \%>\% post_request()

# Not equal operator '!=' with collection property
req \%>\% ext_filter(collection != "sentinel-2-l2a") \%>\% post_request()

# Less than or equal operator '<=' with datetime property
req \%>\% ext_filter(datetime <= "1986-01-01") \%>\% post_request()

# Greater than or equal '>=' with AND operator
req \%>\% ext_filter(collection == "sentinel-2-l2a"   &&
                   `s2:vegetation_percentage` >= 50 &&
                   `eo:cloud_cover` <= 10) \%>\% post_request()
# Advanced comparison operators
# 'LIKE' operator
req \%>\% ext_filter(collection \%like\% "modis\%") \%>\% post_request()

# 'IN' operator
req \%>\% ext_filter(
  collection \%in\% c("landsat-c2-l2", "sentinel-2-l2a") &&
    datetime > "2019-01-01" &&
    datetime < "2019-06-01") \%>\%
  post_request()

# Spatial operator
# Lets create a polygon with list
polygon <- list(
  type = "Polygon",
  coordinates = list(
    matrix(c(-62.34499836, -8.57414572,
             -62.18858174, -8.57414572,
             -62.18858174, -8.15351185,
             -62.34499836, -8.15351185,
             -62.34499836, -8.57414572),
           ncol = 2, byrow = TRUE)
  )
)
# 'S_INTERSECTS' spatial operator with polygon and geometry property
req \%>\% ext_filter(collection == "sentinel-2-l2a" &&
                   s_intersects(geometry, {{polygon}})) \%>\% post_request()

# 'S_CONTAINS' spatial operator with point and geometry property
point <- list(type = "Point", coordinates = c(-62.45792211, -8.61158488))
req \%>\% ext_filter(collection == "landsat-c2-l2" &&
                   s_contains(geometry, {{point}})) \%>\% post_request()

# 'S_CROSSES' spatial operator with linestring and geometry property
linestring <- list(
  type = "LineString",
  coordinates = matrix(
         c(-62.55735320, -8.43329465, -62.21791603, -8.36815014),
         ncol = 2, byrow = TRUE
  )
)
req \%>\% ext_filter(collection == "landsat-c2-l2" &&
                   s_crosses(geometry, {{linestring}})) \%>\% post_request()

# Temporal operator
# 'T_INTERSECTS' temporal operator with datetime property
req \%>\% ext_filter(
  collection == "landsat-c2-l2" &&
    t_intersects(datetime, interval("1985-07-16T05:32:00Z",
                                    "1985-07-24T16:50:35Z"))) \%>\%
 post_request()

# 'T_DURING' temporal operator with datetime property
req \%>\%
 ext_filter(collection == "landsat-c2-l2" &&
            t_during(datetime,
            interval("2022-07-16T05:32:00Z", ".."))) \%>\%
 post_request()

# 'T_BEFORE' temporal operator with datetime property
req \%>\%
 ext_filter(collection == "landsat-c2-l2" &&
            t_before(datetime, timestamp("2022-07-16T05:32:00Z"))) \%>\%
 post_request()

# 'T_AFTER' temporal operator with datetime property
req \%>\%
 ext_filter(collection == "landsat-c2-l2" &&
            t_after(datetime, timestamp("2022-07-16T05:32:00Z"))) \%>\%
  post_request()

# Shows how CQL2 expression (TEXT format)
cql2_text(collection == "landsat-c2-l2" &&
  s_crosses(geometry, {{linestring}}))

# Shows how CQL2 expression (JSON format)
cql2_json(collection == "landsat-c2-l2" &&
            t_after(datetime, timestamp("2022-07-16T05:32:00Z")))
}

}
\seealso{
\code{\link[=ext_query]{ext_query()}}, \code{\link[=stac_search]{stac_search()}}, \code{\link[=post_request]{post_request()}},
\code{\link[=endpoint]{endpoint()}}, \code{\link[=before_request]{before_request()}},
\code{\link[=after_response]{after_response()}}, \code{\link[=content_response]{content_response()}}
}
