% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/iptw_weights.R
\name{calc_iptw_weights}
\alias{calc_iptw_weights}
\title{Calculate Propensity Scores and IPTW Weights}
\usage{
calc_iptw_weights(
  data,
  treatment,
  covariates,
  method = "logistic",
  weight_type = "ATE",
  stabilize = TRUE,
  trim_weights = TRUE,
  trim_quantiles = c(0.01, 0.99),
  verbose = FALSE
)
}
\arguments{
\item{data}{A data frame containing treatment and covariate data}

\item{treatment}{Character string naming the binary treatment variable}

\item{covariates}{Character vector of covariate names for propensity score model}

\item{method}{Method for propensity score estimation: "logistic" (default), "probit", or "cloglog"}

\item{weight_type}{Type of weights to calculate: "ATE" (average treatment effect, default),
"ATT" (average treatment effect on treated), "ATC" (average treatment effect on controls)}

\item{stabilize}{Logical indicating whether to use stabilized weights (default: TRUE)}

\item{trim_weights}{Logical indicating whether to trim extreme weights (default: TRUE)}

\item{trim_quantiles}{Vector of length 2 specifying quantiles for weight trimming (default: c(0.01, 0.99))}

\item{verbose}{Logical indicating whether to print diagnostic information (default: FALSE)}
}
\value{
A list containing:
\describe{
\item{data}{Original data with added propensity scores and weights}
\item{ps_model}{Fitted propensity score model}
\item{weights}{Vector of calculated weights}
\item{ps}{Vector of propensity scores}
\item{diagnostics}{List of diagnostic information including balance statistics}
\item{method}{Method used for propensity score estimation}
\item{weight_type}{Type of weights calculated}
}
}
\description{
Calculates propensity scores and inverse probability of treatment weights
for use in standardized risk difference estimation. Implements multiple
approaches for weight calculation and includes diagnostic tools.
}
\details{
\subsection{Propensity Score Estimation}{

The function fits a model predicting treatment assignment from covariates:
\itemize{
\item \strong{Logistic regression}: Standard approach, assumes logit link
\item \strong{Probit regression}: Uses probit link, may be more robust with extreme probabilities
\item \strong{Complementary log-log}: Useful when treatment is rare
}
}

\subsection{Weight Types}{
\itemize{
\item \strong{ATE weights}: 1/pi(X) for treated, 1/(1-pi(X)) for controls
\item \strong{ATT weights}: 1 for treated, pi(X)/(1-pi(X)) for controls
\item \strong{ATC weights}: (1-pi(X))/pi(X) for treated, 1 for controls
}

Where pi(X) is the propensity score (probability of treatment given X).
}

\subsection{Stabilized Weights}{

When stabilize=TRUE, weights are multiplied by marginal treatment probabilities
to reduce variance while maintaining unbiasedness (Robins et al., 2000).
}

\subsection{Weight Trimming}{

Extreme weights can cause instability. Trimming replaces weights outside
specified quantiles with the quantile values (Crump et al., 2009).
}
}
\examples{
data(cachar_sample)

# Calculate ATE weights for areca nut use
iptw_result <- calc_iptw_weights(
  data = cachar_sample,
  treatment = "areca_nut",
  covariates = c("age", "sex", "residence", "smoking"),
  weight_type = "ATE"
)

# Check balance
print(iptw_result$diagnostics$balance_table)

# Calculate ATT weights (effect on the treated)
iptw_att <- calc_iptw_weights(
  data = cachar_sample,
  treatment = "tobacco_chewing",
  covariates = c("age", "sex", "residence", "areca_nut"),
  weight_type = "ATT"
)

}
\references{
Austin PC (2011). "An Introduction to Propensity Score Methods for Reducing
the Effects of Confounding in Observational Studies." Multivariate Behavioral
Research, 46(3), 399-424. doi:10.1080/00273171.2011.568786

Crump RK, Hotz VJ, Imbens GW, Mitnik OA (2009). "Dealing with Limited Overlap
in Estimation of Average Treatment Effects." Biometrika, 96(1), 187-199.

Hernan MA, Robins JM (2020). Causal Inference: What If. Boca Raton: Chapman & Hall/CRC.

Robins JM, Hernan MA, Brumback B (2000). "Marginal Structural Models and Causal
Inference in Epidemiology." Epidemiology, 11(5), 550-560.
}
