/*****************************************************************************
* (c) Copyright 2012-2013 F.Hoffmann-La Roche AG                             *
* Contact: bioinfoc@bioinfoc.ch, Detlef.Wolf@Roche.com.                      *
*                                                                            *
* This file is part of BIOINFO-C. BIOINFO-C is free software: you can        *
* redistribute it and/or modify it under the terms of the GNU Lesser         *
* General Public License as published by the Free Software Foundation,       *
* either version 3 of the License, or (at your option) any later version.    *
*                                                                            *
* BIOINFO-C is distributed in the hope that it will be useful, but           *
* WITHOUT ANY WARRANTY; without even the implied warranty of                 *
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU          *
* Lesser General Public License for more details. You should have            *
* received a copy of the GNU Lesser General Public License along with        *
* BIOINFO-C. If not, see <http://www.gnu.org/licenses/>.                     *
*****************************************************************************/
/** @file wwwsession.c
    @brief Handle WWW sessions for CGI programs.
    Module prefix wwwsess_
*/
/*
   The session name is used as cookie name; the username, the
   start time of the session and optionally the user's password
   are encrypted and transmitted into the cookies value.

   The security concept is that the en/decryption functions
   themselves are not located in the module, but in the application
   code; so each application programmer / application administrator
   can construct functions assuring their needs.

   For convenience, this module contains default implementations
   for the en/decryption functions, which are automatically used,
   if nothing else is registered. You can use these functions
   as templates. Your own functions most conform to the interface set.

   Note: the default encryption functions are easy to crack, if
   someone has this source code and can get hold of encrypted/decrypted
   value pairs, even if the encryptor string below was changed.
*/
#include <time.h>
#include "plabla.h"
#include "log.h"
#include "format.h"
#include "hlrmisc.h"
#include "alphatrans.h"
#include "html.h"
#include "wwwsession.h"

// #define DEBUG

#ifdef DEBUG
#include <ctype.h>

static void hexdump (char *s,int len) {
  int i;
  char c;
  for (i=0;i<len;i++)
    printf (" %X",s[i]);
  printf (" (%d bytes)<br>\n",len);
  for (i=0;i<len;i++) {
    c = s[i];
    printf (" %c",isalnum (c) ? c : '-');
  }
  printf ("<br>\n");
}
#endif

/* ------------------- mini module: plain 'encryptor' -----------
   here the most trival encrypt/decrypt: just leave the text as is
   good for testing
*/

char *wwwsession_identEncrypt (char *s,int len,int *outlen) {
  /**
     Encrypt string 's'.<br>
     Precondition: cgiInit() must have been called before.
     @param[in] s - start of memory block to encrypt
     @param[in] len - length of memory block to encrypt
     @param[out] outlen - length of the encrypted string
     @return encrypted version of 's'; this is not a C-string
             anymore; it can contain internal '\0';<br>
             The memory for the returned string is managed by this routine
  */
  static char *r = NULL;

  if (len < 1)
    die ("cannot encrypt %d bytes",len);
  hlr_free (r);
  r = hlr_malloc (len);
  memcpy (r,s,len);
  *outlen = len;
  return r;
}

char *wwwsession_identDecrypt (char *s,int len,int *outlen) {
  /**
     Decrypt s of length len, s can contain '\0' chars
     @param[in] s - the byte array to be decrypted
     @param[in] len - length of the byte array
     @param[out] outlen - number of bytes returned
     @return decrypted string; the memory of the returned string
             is managed by this routine and may not be free'd
             or realloc'd by its user
  */
  static char *r = NULL;

  hlr_free (r);
  if (len < 1)
    die ("identDecrypt: input was not generated by identEncrypt");
  r = hlr_malloc (len);
  memcpy (r,s,len);
  *outlen = len;
  return r;
}

/* --------------------- end trival encryptor module ----- */

static void normalizeByteOrder (char *s) {
  /**
     Assuming the first 4 bytes of s represent an integer:
     Make sure the first 4 bytes of 's' are in low-endian order,
     irrespecitive of machine type
  */
#if BIOS_PLATFORM == BIOS_PLATFORM_SOLARIS || BIOS_PLATFORM == BIOS_PLATFORM_IRIX
  // big-endian machine: swap byte order
  unsigned char c;
  c = s[0];
  s[0] = s[3];
  s[3] = c;
  c = s[1];
  s[1] = s[2];
  s[2] = c;
#else
  ; // do nothing
#endif
}

static char*(*gEncrypt_hook)(char *s,int len,int *outlen) = NULL;

void wwwsess_register_encrypt (char*(*f)(char *s,int len,int *outlen)) {
  /**
     Register the encryption function used in wwwsess_start();
     f must be the inverse function of the function used
     in wwwsess_register_decrypt().<br>
     Postcondition: wwwsess_start() will use f
  */
  gEncrypt_hook = f;
}

static char*(*sDecrypt_hook)(char *s,int len,int *outlen) = NULL;

void wwwsess_register_decrypt (char*(*f)(char *s,int len,int *outlen)) {
  /**
     Register the decryption function used in wwwsess_check();
     f must be the inverse function of the function used
     in wwwsess_register_encrypt().<br>
     Postcondition: wwwsess_check() will use f
  */
  sDecrypt_hook = f;
}

static int now (void) {
  return (int)time (0);
}

/// marker of a logged out session
#define LOGGED_OUT_MARKER "*!logged out!*"

void wwwsess_logout (char *sessionname) {
  /**
     Log out from WWW session
  */
  wwwsess_start (sessionname,LOGGED_OUT_MARKER,"",-1);
}

static char *bakeCookieValue (char *user,char *password) {
  /**
     Construct cookie value;
     used by wwwsess_start()
     @param[in] user
     @param[in] password
     @return cookieValue - memory belongs to this routine; contents stable
             until next call.
  */
  static char *cookieValue = NULL;
  static Array s = NULL;
  int eslen;
  char *es; // encrypted string
  int t = now ();
  int len;

  stringCreateOnce (s,40);

  if (sizeof (int) != 4)
    die ("wwwsess_bakeCookieValue: please port to this platform!");
  stringCpy (s,"    "); // reserve space for binary int
  if (strchr (user,'\t'))
    die ("wwwsess_bakeCookieValue: tab not allowed in username");
  stringCat (s,user);
  stringCat (s,"\t");
  stringCat (s,password);
  len = strlen (string (s))+1; // including the trailing '\0'
  memcpy (string (s),(char*)&t,4); // fill space reserved before
  normalizeByteOrder (string (s));
  if (gEncrypt_hook == NULL)
    die ("wwwsess_bakeCookieValue: no encryptor defined");
  es = (*gEncrypt_hook)(string (s),len,&eslen);
  altr_init ();
  strReplace (&cookieValue,altr_bytes (es,eslen));
  altr_end ();
  return cookieValue;
}

void wwwsess_fake (char *sessionname,char *user,char *password) {
  /**
     Faking a web session in a command line context.<br>
     Same as wwwsess_start(), except that it sets the HTTP-COOKIE using setenv,
     instead of pushing the cookie into the HTTP header.<br>
     Postcondition: wwwsession_read() will show the 'user' and 'password' info
     passed to this function; the time of login returned will be the time
     of calling this function.<br>
     Previous contents of getenv("HTTP_COOKIE") overwritten.
  */
  /*
    possible improvement:
    prevent accidental overwritting of cookies set by other routines
    by checking HTTP_COOKIE before overwriting
  */
  char *cookieValue = bakeCookieValue (user,password);
  static Stringa s = NULL;
  static Stringa e = NULL;
  stringCreateOnce (s,40);
  stringCreateOnce (e,40);
  cgiEncodeWord (cookieValue,e);
  stringCpy (s,"HTTP_COOKIE=");
  stringCat (s,sessionname);
  stringCat (s,"=");
  stringCat (s,string (e));
  putenv (string(s));
}

void wwwsess_start (char *sessionname,char *user,char *password,
                    int lifelength) {
  /**
     Assumes 'user' is the name of an identified user
     optionally her/his password can be passed along.<br>
     Precondition: - runs in a CGI program that has not
                     yet printed the mimetype line<br>
                   - encryption function was set using
                     wwwsess_register_encrypt()<br>
     Postcondition: the mimetype line is printed as text/html
                    a cookie is set that will be used in wwwsess_check()
     @param[in] sessionname - identifier for this session;
                the session name should be different among different
                identification and encryption/decryption domains
     @param[in] user - can be an emtpy string, too
     @param[in] password - can be an emtpy string, too
     @param[in] lifelength - 0: coincide with browser process
                             >0 : valid for 'lifelength' seconds
                             <0 : (delete cookie - not useful to start session)
  */
  char *cookieValue = bakeCookieValue (user,password);
  if (cgiHeaderIsPrinted ())
    die ("programmer's error in calling wwwsess_start():\n"
         "cannot be called after CGI header has been printed already.");
  cgiHeaderCookie (/* text/html */ NULL,
                   cgiConstructCookie(sessionname,cookieValue,lifelength));
#ifdef DEBUG
  printf ("to be encrypted:<br>");
  hexdump (string (s),len);
  printf ("encrypted:<br>");
  hexdump (es,eslen);
  printf ("<br>setenv HTTP_COOKIE '%s'<br>",
          cgiConstructCookie (sessionname,cookieValue,lifelength));
#endif
}

void wwwsess_check (char *sessionname,char **user,char **password,int *age) {
  /**
     Gets the session info from the environment of this process as defined by
     wwwsess_start(); rest of description see wwwsess_read()
     @param[in] sessionname - see comment under wwwsess_start()
     @param[out] user - the user name
     @param[out] password - the password (if in the session name)
     @param[out] age - the age of the cookie
  */
  wwwsess_read (cgiReadCookie (sessionname),user,password,age);
}

void wwwsess_read (char *value,char **user,char **password,int *age) {
  /**
     Decrypt and read the 'value' idenitifying this session;
     this process as defined by wwwsess_start()
     @param[in] value - encrypted userid, password sessionage
     @param[in] user - pointer to a place where to put  a pointer to the
                       user name; *user must be NULL
     @param[in] password - same for password,
                           NULL if no password should be returned
     @param[in] age - same for age, NULL if age not interesting
     @param[out] user - NULL if username could not be determined or was emtpy
                        or user logged out;<br>
                        The memory returned, if any, belongs to the caller of
                        this routine; the caller is responsible for free()ing
                        after use
     @param[out] password - NULL if no username, else the string set in
                            wwwsess_start();<br>
                            The memory returned, if any, belongs to the
                            caller of this routine; the caller is responsible
                            for free()ing after use
     @param[out] age - time since wwwsess_start() in seconds; only valid
                       if *user is not NULL
  */
  int len;
  char *s;
  char *r;
  int outlen;
  char *tabpos;

  if (user == NULL)
    die ("wwwsess_check() needs user output location");
  if (*user != NULL)
    die ("wwwsess_check(): *user must be NULL on input");
  if (password != NULL && **password != '\0')
    die ("wwwsess_check(): *password must be NULL on input");
  if (age != 0)
    *age = 0;
  if (value == NULL)
    return;

  altr_init ();
  s = altr_read (value,&len);
  if (len < 8) { // no decent length for decrypted value
    altr_end ();
    return;
  }
#ifdef DEBUG
  printf ("read from altr\n");
  hexdump (s,len);
#endif

  if (sDecrypt_hook == NULL)
    die ("wwwsess_check: no decryptor defined");
  r = (*sDecrypt_hook) (s,len,&outlen);
  altr_end ();
#ifdef DEBUG
  printf ("after decrypt\n");
  hexdump (r,outlen);
#endif

  if (outlen < 7)
    die ("wwwsess_check: outlen=%d",outlen);
  if (r[outlen-1] != '\0')
    die ("wwwsess_check: r[outlen]='%c'",r[outlen-1]);
  if (sizeof (int) != 4)
    die ("wwwsess_check: please port to this platform!");
  if (! (tabpos = strchr (r+4,'\t')))
    die ("wwwsess_check: no \\t found");
  *tabpos = '\0';

  if (strEqual (r+4,LOGGED_OUT_MARKER))
    return;
  strReplace (user,r+4);
  if (*age > 0) {
    normalizeByteOrder (r);
    memcpy ((char*)age,r,4);
    *age = now () - *age;
  }
  if (password != NULL)
     strReplace (password,tabpos+1);
}
