% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/neg_exp_profile.R
\name{neg_exp_profile}
\alias{neg_exp_profile}
\title{Negative Exponential Model for Geographic Profiling}
\usage{
neg_exp_profile(lat, lon, method = c("CrimeStat", "Dragnet", "Custom"),
  buffer = FALSE, a = NULL, b = NULL, n = NULL)
}
\arguments{
\item{lat}{a vector of latitudes for the crime incident series}

\item{lon}{a vector of latitudes for the crime incident series}

\item{method}{'CrimeStat', 'Dragnet', or a custom parameter based negative exponential
decay function. If using the 'CrimeStat' or 'Dragnet' method, values do not
need to be provided from 'a' and 'b' as the default parameters will be
used. Default parameters for the 'CrimeStat' are: \eqn{a = 1.89} \eqn{a = -0.06}.
Default parameters for the 'Dragnet' are: \eqn{a = b = 1}. If using a custom
model, values must be provided for '*a*' and '*b*'.}

\item{buffer}{TRUE/FALSE. Whether a buffer zone where a likelihood of zero
is fit around the incidents and a plateau of peak likelihood is fit prior
to the negative exponential decay. The function calculates the buffer zone
and the plateau area to each be half of the average nearest neighbor
distance.}

\item{a}{the slope coefficient which defines the function decrease in distance}

\item{b}{exponential multiplier for the distance decay function}

\item{n}{total number of cells within the spatial grid for the jeopardy surface.
If \code{NULL}, the default value for '*n*' is 40,000.}
}
\value{
A data frame of points depicting a spatial grid of the hunting area
    for the given incident locations. Also given are the resultant summed
    values (score) for each map point. A higher resultant score indicates
    a greater the probability that point contains the offender's anchor point.
}
\description{
An implementation of variations of the negative exponential
    decay model for serial crime analysis. In this model, the decline is at
    a constant rate, therefore the likelihood of the perpetrator's home base
    drops quickly from the incident locations until it approaches zero
    likelihood. The user can select different variants including the 'CrimeStat'
    base model, the 'Dragnet' model, or whether a buffer and plateau is present
    at the start of the decay function. This model assumes that the likelihood
    of the serial perpetrator's home base decreases in a exponential fashion
    as the distance increases from the crime incidents.
}
\examples{
\dontshow{
data(desalvo)
test <- neg_exp_profile(desalvo$lat, desalvo$lon, method = "CrimeStat", n = 4)
}
\donttest{
#Using provided dataset for the Boston Strangler Incidents:
data(desalvo)
test <- neg_exp_profile(desalvo$lat, desalvo$lon, method = "CrimeStat")
g_map = sp::SpatialPixelsDataFrame(points = test[c("lons", "lats")], data = test)
g_map <- raster::raster(g_map)
# Assign a Coordinate Reference System for the Raster
raster::crs(g_map) <- sp::CRS("+proj=longlat +ellps=WGS84 +datum=WGS84 +no_defs")
# Define a Parula Color Pallete for Resultant Jeopardy Surface
library(leaflet) #for mapping the geographic profile
pal <- colorNumeric(pals::parula(200), raster::values(g_map),
    na.color = "transparent")
leaflet() \%>\%
    addTiles() \%>\%
    addProviderTiles('Esri.WorldTopoMap', group = 'Topo') \%>\%
    addAwesomeMarkers(lng = -71.07357, lat = 42.41322, icon =
        awesomeIcons(icon = 'home', markerColor = 'green'), popup = 'Residence') \%>\%
    addRasterImage(g_map, colors = pal, opacity = 0.6) \%>\%
    addLegend(pal = pal, values = raster::values(g_map), title = 'Score') \%>\%
    addCircleMarkers(lng = data$lon, lat = data$lat, radius = 4, opacity = 1,
        fill = 'black', stroke = TRUE, fillOpacity = 0.75, weight = 2,
        fillColor = "red")
}
}
\references{
Ned Levine, \emph{CrimeStat IV: A Spatial Statistics Program for the
    Analysis of Crime Incident Locations (version 4.0)}. Ned Levine & Associates,
    Houston, TX, and the National Institute of Justice, Washington, DC, June 2013.

D Canter, T Coffey, M Huntley & C Missen. (2000). \emph{Predicting
    serial killers' home base using a decision support system.} Journal of
    quantitative criminology, 16(4), 457-478.
}
\author{
Jamie Spaulding, Keith Morris
}
\keyword{methods}
\keyword{spatial}
