% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/add_compactness_constraint.R
\name{add_compactness_constraint}
\alias{add_compactness_constraint}
\title{Add constraint to limit compactness}
\usage{
add_compactness_constraint(problem, max_diameter, unit = "m")
}
\arguments{
\item{problem}{\code{\link[=restopt_problem]{restopt_problem()}} Restoration problem object.}

\item{max_diameter}{\code{numeric} Maximum diameter value.}

\item{unit}{\code{unit} object or a \code{character} that can be coerced to an area
unit (see \code{unit} package), or "cells" for cell width of aggregated
habitat raster. Corresponds to the unit of the maximum diameter. If the
input habitat raster does not use a projected coordinate system, only "cells"
is available.}
}
\value{
An updated restoration problem (\code{\link[=restopt_problem]{restopt_problem()}}) object.
}
\description{
Add constraint to a restoration problem (\code{\link[=restopt_problem]{restopt_problem()}}) object
to specify the compactness of a solution.
}
\details{
The compactness constraint is defined according to the diameter of
the smallest enclosing circle which contains the center of selected planning
units for restoration (see https://w.wiki/4vfg). The unit of the diameter
corresponds either to a unit available in the \code{unit} package, or to planning
unit width ("cells"). Note that, as the computation occurs on aggregated cells,
if \code{max_diameter} is used with a different unit than "cells", it will be rounded
to the closest corresponding number of cells. For example, a diameter of 4 cells
means that no more than 4 cells can be found in line in the solution. In practice,
this constraint is useful to ensure the feasibility of a restoration project,
and to integrate economies of scale. Compact restoration areas are usually
associated with lower costs and easier management, because it ensures that
restoration sites are not too far away from each other (e.g. lower travel costs
between sites, less areas to monitor, etc.).
}
\examples{
\donttest{
# load data
habitat_data <- rast(
  system.file("extdata", "habitat_hi_res.tif", package = "restoptr")
)

# create problem
p <- restopt_problem(
    existing_habitat = habitat_data,
    aggregation_factor = 16,
    habitat_threshold = 0.7
  ) \%>\%
  add_restorable_constraint(
    min_restore = 200,
    max_restore = 300,
  ) \%>\%
  add_compactness_constraint(1800, unit = "m")

# plot preprocessed data
plot(rast(list(p$data$existing_habitat, p$data$restorable_habitat)), nc = 2)

# print problem
print(p)

# Solve problem
s <- solve(p)
# plot solution
plot(s)
}

}
\seealso{
Other constraints: 
\code{\link{add_available_areas_constraint}()},
\code{\link{add_components_constraint}()},
\code{\link{add_connected_constraint}()},
\code{\link{add_locked_out_constraint}()},
\code{\link{add_min_iic_constraint}()},
\code{\link{add_min_mesh_constraint}()},
\code{\link{add_restorable_constraint}()}
}
\concept{constraints}
