% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/calc_ehh.R
\docType{class}
\name{calc_ehh}
\alias{calc_ehh}
\alias{ehh-class}
\alias{ehh}
\title{EHH and iHH computation for a given focal marker}
\usage{
calc_ehh(haplohh, mrk, limhaplo = 2, limehh = 0.05,
  include_zero_values = FALSE, include_nhaplo = FALSE, phased = TRUE,
  polarized = TRUE, scalegap = NA, maxgap = NA,
  discard_integration_at_border = TRUE, lower_y_bound = limehh)
}
\arguments{
\item{haplohh}{an object of class \code{haplohh} (see \code{\link{data2haplohh}}).}

\item{mrk}{integer representing the number of the focal marker within the haplohh object
or string representing its ID/name.}

\item{limhaplo}{if there are less than \code{limhaplo} chromosomes that can be used for
the calculation of EHH, the calculation is stopped. The option is intended for the case of missing data,
which leads to the successive exclusion of haplotypes: the further away from the focal marker
the less haplotypes contribute to EHH.}

\item{limehh}{limit at which EHH stops to be evaluated}

\item{include_zero_values}{logical. If \code{FALSE}, return values only for those positions where the calculation is
actually performed, i.e. until stopped by reaching either \code{limehh} or \code{limhaplo}. If \code{TRUE}, report EHH values for
all markers, the additional ones being zero.}

\item{include_nhaplo}{logical. If \code{TRUE}, report the number of evaluated haplotypes at each marker
(only informative, if missing data leads to a decrease of evaluated haplotypes).}

\item{phased}{logical. If \code{TRUE} (default) chromosomes are expected to be phased. If \code{FALSE}, the haplotype data is assumed to
consist of pairwise ordered chromosomes belonging to diploid individuals.
EHH is then estimated over individuals which are homozygous at the focal marker.}

\item{polarized}{logical. \code{TRUE} by default. If \code{FALSE}, use major and minor allele instead of ancestral and derived. If there
are more than two alleles then the minor allele refers to the second-most frequent allele.}

\item{scalegap}{scale or cap gaps larger than the specified size to the specified size (default=\code{NA}, i.e. no scaling).}

\item{maxgap}{maximum allowed gap in bp between two markers. If exceeded, further calculation of EHH is stopped at the gap
(default=\code{NA}, i.e no limitation).}

\item{discard_integration_at_border}{logical. If \code{TRUE} (default) and computation reaches first or last marker or a gap larger than \code{maxgap},
iHH is set to \code{NA}.}

\item{lower_y_bound}{lower y boundary of the area to be integrated over (default: \code{limehh}). Can be set
to zero for compatibility with the program hapbin.}
}
\value{
The returned value is a list containing the following elements:
\describe{
\item{mrk.name}{The name/identifier of the focal marker.}
\item{freq}{A vector with the frequencies of the alleles of the focal marker.}
\item{ehh}{A data frame with EHH values for each allele of the focal marker.}
\item{ihh}{A vector with iHH (integrated EHH) values for each allele of the focal marker.}
}
}
\description{
Compute Extended Haplotype Homozygosity (EHH) and integrated EHH (iHH) for a given focal marker.
}
\details{
Values for allele-specific Extended Haplotype Homozygosity (EHH) are computed
upstream and downstream of the focal marker for each of its alleles.
These values are integrated with respect to their genomic positions to yield
an 'integrated EHH' (iHH) value for each allele.
}
\examples{
#example haplohh object (280 haplotypes, 1424 SNPs)
#see ?haplohh_cgu_bta12 for details
data(haplohh_cgu_bta12)
#computing EHH statistics for the marker "F1205400"
#which displays a strong signal of selection
ehh <- calc_ehh(haplohh_cgu_bta12, mrk = "F1205400")
}
\references{
Gautier, M. and Naves, M. (2011). Footprints of selection in the ancestral admixture of a New World Creole cattle breed. \emph{Molecular Ecology}, \strong{20}, 3128-3143.

Sabeti, P.C. et al. (2002). Detecting recent positive selection in the human genome from haplotype structure. \emph{Nature}, \strong{419}, 832-837.

Sabeti, P.C. et al. (2007). Genome-wide detection and characterization of positive selection in human populations. \emph{Nature}, \strong{449}, 913-918.

Tang, K. and Thornton, K.R. and Stoneking, M. (2007). A New Approach for Using Genome Scans to Detect Recent Positive Selection in the Human Genome. \emph{Plos Biology}, \strong{7}, e171.

Voight, B.F. and Kudaravalli, S. and Wen, X. and Pritchard, J.K. (2006). A map of recent positive selection in the human genome. \emph{Plos Biology}, \strong{4}, e72.
}
\seealso{
\code{\link{data2haplohh}}, \code{\link{plot.ehh}}, \code{\link{calc_ehhs}}, \code{\link{scan_hh}}.
}
