% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/MapSave-methods.R
\name{rangeMap.save}
\alias{rangeMap.save}
\title{Save, retrieve and export maps.}
\usage{
rangeMap.save(CON, tableName, FUN, biotab, biotrait, subset = list(), path,
  overwrite = FALSE, cl, ...)
}
\arguments{
\item{CON}{an sqlite connection pointing to a valid \code{rangeMapper}
project.}

\item{tableName}{name of the table (quoted) to be added to the sqlite database.
the prefix \sQuote{MAP} will be appended to \code{tableName}
prior to saving.}

\item{FUN}{the function to be applied to each pixel. If \code{FUN} is
missing then species richness (species count) is computed.}

\item{biotab}{character string identifying the \sQuote{BIO} table to use.}

\item{biotrait}{character string identifying the ID of the \sQuote{BIO}
table. see \code{\link{bio.save}}}

\item{subset}{a named \code{\link{list}}. See details}

\item{path}{path to the raster file(quoted) to be imported to the existing
project. \code{raster package} is required at this step.}

\item{overwrite}{if \code{TRUE} then the table is removed}

\item{cl}{the number of cores to use or a cluster object defined with
\code{\link[parallel]{makeCluster}} in package \code{parallel}
or \code{\link[snow]{makeCluster}} from \code{snow} package.}

\item{\dots}{when \code{FUN} is a function, \dots{} denotes any extra
arguments to be passed to it.}
}
\value{
\code{TRUE} when the MAP was created successfully.
}
\description{
Apply a chosen \code{SQL} or function at each grid cell, allowing for
complex subsetting at both ID (e.g. species) and pixel (e.g. assemblage)
levels.
}
\details{
The subset argument accepts a named list. Names refers to \sQuote{BIO},
\sQuote{MAP} and \sQuote{metadata_rages} table names while the strings in
the list are character strings containing the SQL \code{WHERE} clause. The
subset can point to either one table type (e.g.
\code{list(MAP_species_richness = "species_richness > 500")} ) or can point
to several table types (e.g. \code{list(BIO_lifeHistory = "clutch_size > 4",
MAP_meanAltitude = "meanAltitude < 1000", metadata_ranges = "Area < 1000")})

Any valid SQL expression can be used to build up a subset. See
\url{http://www.sqlite.org/lang_expr.html}

When using \code{cl} parameter you must load the apropiated packages used in
\code{FUN} by loading the packages inside the function or initializing the
cluster before calling rangeMap.save (e.g. \code{clusterEvalQ(cl=cl, library(caper))})).
}
\note{
\code{SQL} aggregate functions are more efficient then their R
                 counterparts. For simple aggregate functions like mean, median, sd, count
                 it is advisable to use \code{SQL} functions rather then R functions.
}
\examples{
require(rangeMapper)
require(rgdal)
breding_ranges = readOGR(system.file(package = "rangeMapper",
     "extdata", "wrens", "vector_combined"), "wrens", verbose = FALSE)
breding_ranges = spTransform(breding_ranges,
    CRS("+proj=cea +lon_0=0 +lat_ts=30 +x_0=0 +y_0=0
        +ellps=WGS84 +units=m +no_defs") )
data(wrens)
d = subset(wrens, select = c('sci_name', 'body_size', 'body_mass', 'clutch_size') )

con = ramp("wrens.sqlite", gridSize = 200000, spdf = breding_ranges, biotab = d,
            ID = "sci_name", metadata = rangeTraits(),
            FUN = "median", overwrite = TRUE)


lmSlope = function(formula, data) {
    fm = try(lm(formula, data = data), silent = TRUE)
    if (inherits(fm, "try-error"))
        res = NA else res = coef(fm)[2]
    as.numeric(res)
}

# Subsetting by Species and Assembladge
rangeMap.save(con, FUN = lmSlope, biotab = "biotab", biotrait = "body_mass",
    tableName = "slope_bodyMass_clutchSize", formula = log(body_mass) ~ clutch_size,
    list(MAP_species_richness = "species_richness >= 5",
        BIO_biotab = "body_size > 15"
        ), overwrite = TRUE)

\dontrun{
# Import raster maps to the current project
r = system.file(package = "rangeMapper", "extdata", "etopo1", "etopo1_Americas.tif")
rangeMap.save(con, path = r, tableName = "meanAltitude", FUN = mean, overwrite = TRUE)
}


m = rangeMap.fetch(con, spatial = FALSE)
plot(m)

}
\seealso{
\code{\link{metadata.update}}.
}

