#' @name plotCrossValidation
#' @title plotCrossValidation
#' @aliases plotCrossValidation
#' @export plotCrossValidation
#' @author Linus Olsson \email{linusmeol@@gmail.com}, Alexander Pothmann
#' @param cv_results The List of results returned by the calculateCrossValidation function.
#' @description Plots data generated by the calculateCrossValidation function as a boxplot. 
#' Includes max and min as whiskers as well as the average (marked by a crossed circle), 
#' median (marked by a horizontal bold line) and the 1st and 3rd quartile of the values. 
#' Visualizes outliers in the data as red triangles.
#' @return None.
#' @examples 
#' df <- data.frame(
#' Sol_1=c(7, 6, 5, 4, 3, 2, 1),
#' Sol_2=c(1, 2, 3, 4, 5, 7, 6),
#' Sol_3=c(1, 2, 3, 4, 7, 5, 6),
#' Ref=c(1, 2, 3, 4, 5, 6, 7))
#' 
#' cv_results <- rSRD::calculateCrossValidation(df, output_to_file = FALSE) 
#' rSRD::plotCrossValidation(cv_results)
plotCrossValidation <- function(cv_results){
  # order the solutions in the following order; everytime there are ties,
  # the ordering process goes one step deeper in the ordering hierarchy
  # ordering hierarchy: median -> quartile1 -> quartile3 -> minimum -> maximum
  cv_srd <- cv_results$SRD_values_of_different_folds
  boxplot_values <- cv_results$boxplot_values
  graph_data <- rbind(cv_srd, boxplot_values)
  rownames(graph_data) <- c(c(1 : nrow(cv_srd)),c("min", "xx1", "Q1", "median", "Q3", "xx19", "max"))
  medianVal <- as.numeric(graph_data["median",])
  quartile1Val <- as.numeric(graph_data["Q1",])
  quartile3Val <- as.numeric(graph_data["Q3",])
  minVal <- as.numeric(graph_data["min",])
  maxVal <- as.numeric(graph_data["max",])
  graph_data <- graph_data[,order(medianVal, quartile1Val, quartile3Val, minVal, maxVal)]
  test <- utils::stack(graph_data)
  index <- test$ind
  values <- test$values
  # plotting the values of each column as a single box-whisker-plot; 
  ggplot(test, aes(x = index, y = values)) +
    # stat_boxplot for showing the whiskers
    stat_boxplot(geom="errorbar", width=0.5) +
    # visualize errors
    geom_boxplot(outlier.color="red",outlier.size=3,outlier.shape=2) +
    # stat_summary for showing the median
    stat_summary(fun=mean, geom="point", shape=10, size=3) +
    # layout of the diagram
    scale_x_discrete() +
    labs(y="Values", x="Solutions", title="Box-whisker-plot") + 
    theme(plot.title = element_text(hjust = 0.5))
    # ggpubr::stat_compare_means(comparisons = cmp, tip.length = 0.01, label = "p.signif",
    #                            symnum.args = list(cutpoints = c(0, 0.05, 2),
    #                                               symbols = c(">", "~")))
}
