% Generated by roxygen2 (4.0.1): do not edit by hand
\name{psel}
\alias{psel}
\alias{psel.indices}
\title{Preference selection}
\usage{
psel(df, pref, ...)

psel.indices(df, pref, ...)
}
\arguments{
\item{df}{A dataframe or, for a grouped preference selection, a grouped dataframe. See below for details.}

\item{pref}{The preference order constructed via \code{\link{complex_pref}} and \code{\link{base_pref}}.
All variables occuring in the definition of \code{pref} must be either columns of the dataframe \code{df}
or variables/functions of the environment where \code{pref} was defined.}

\item{...}{Additional parameters for Top(-Level)-k selections:
 \describe{
   \item{\code{top}}{A \code{top} value of k means that the k-best tuples of the dataset are returned.
    This may be non-deterministic, see below for details.}
    \item{\code{at_least}}{A \code{at_least} value of k returns the Top-k tuples and additionally all tuples which are
    not dominated by the worst tuple (i.e., the minima) of the Top-k set. The number of tuples returned is greater or equal than
     \code{at_least}. In contrast to top-k, this is deterministic.}
    \item{\code{top_level}}{A \code{top_level} value of k returns all tuples from the k-best levels. See below for the definition of a level.}
    \item{\code{and_connected}}{Logical value, which is only relevant if more than one of the above \{top, at_least, top_level\}
    values are given. Then \code{and_connected = TRUE} (which is the default) means that all top-conditions must hold for the returned tuples, i.e.,
    \code{psel([...], cond1, cond2)} is equivalent to the intersection of \code{psel([...], cond1)} and \code{psel([...], cond2)}. If we have
    \code{and_connected = FALSE} the conditions are or-connected. This corresponds to the union.}
    \item{\code{show_level}}{This adds a column '.level' to the returned data frame, containing all level values (see below).
    This is only relevant if at least one of the \{top, at_least, top_level\} values are given.
    For \code{psel} this is \code{TRUE} by default, for \code{psel.indices} this is \code{FALSE} by default.}
}}
}
\description{
Evaluates a preference on a given dataset, i.e., return the maximal elements of a dataset for a given preference order.
}
\details{
The difference between the two variants of the preference selection is:

\itemize{
\item The \code{psel} function returns a subset of the dataset which are the maxima according to the given preference.
\item The function \code{psel.indices} returns just the row indices of the maxima
(except Top-k queries with \code{show_level = 1}, see Top-k preference selection).
Hence \code{psel(df,pref)} is equivalent to \code{df[psel.indices(df,pref),]} for non-grouped dataframes.
}
}
\section{Top-k preference selection}{


For a given \code{top} value "k" the k best elements and their level values are returned. The level values are determined as follows:

\itemize{
\item{All the maxima of a dataset w.r.t. a preference have level 1.}
\item{The maxima of the remainder, i.e. the dataset without the level-1 maxima, have level 2.}
\item{The n-th iteration of "Take the maxima from the remainder" leads to tuples of level n.}
}

By default \code{psel.indices} does not return the level values. By setting \code{show_level = TRUE} this function
returns a dataframe with the columns '.indices' and '.level'.

By definition, a top-k preference selection is non-deterministic. A top-1 query of two equivalent tuples (equivalence according to \code{pref})
can return on both of these tuples. E.g., for tuples {(a=1,b=1),(a=1,b=2)} a \code{top=1} selection with a \code{pref=low(a)} preference can return
either the 'b=1' or the 'b=2' tuple.

On the contrary a preference selection with \code{at_least} is deterministic by adding all tuples having the same level as the worst level
of the corresponding top-k query, i.e., which are incomparable to the minimum of the top-k result.
A preference selection with \code{top_level} k returns all tuples having level k or better.

If the \code{top} or \code{at_least} value is greater than the number of elements in \code{df}
(i.e. \code{nrow(df)}) or \code{top_level} is greater then the highest level in \code{df}
then all elements of \code{df} will be returned without further warning.
}

\section{Grouped preference selection}{


With \code{psel} it is also possible to perform a preference selection where the maxima are calculated for every group seperately.
The groups have to be created with \code{\link{group_by}} from the dplyr package. The preference selection preserves the grouping, i.e.,
the groups are restored after the preference selection.

For example the \code{summarize} function from dplyr refers to the set of maxima of each group.
This can be used to e.g. calculate the number of maxima in each group, see examples below.

A \{top, at_least, top_level\} preference selection is applied to each group seperately.
A \code{top=k} selection returns the k best tuples for each group.
Hence if there are 3 groups in \code{df}, each containing at least 2 elements,
and we have \code{top = 2} then 6 tuples will be returned.
}

\section{Parallel computation}{


On multicore machines the preference selection runs in parellel using a divide-and-conquer approach.
If multi-threaded compuation causes problems on your system or you prefer a single-threaded computation for other reasons,
use the following code to deactivate parallel compuation within rPref:

\code{options(rPref.parallel = FALSE)}

If this option is not set, rPref will use parallel computation by default.
}
\examples{
# Skyline and Top-K/At-least skyline
psel(mtcars, low(mpg) * low(hp))
psel(mtcars, low(mpg) * low(hp), top = 5)
psel(mtcars, low(mpg) * low(hp), at_least = 5)

# Visualize the skyline in a plot
sky1 <- psel(mtcars, high(mpg) * high(hp))
plot(mtcars$mpg, mtcars$hp)
points(sky1$mpg, sky1$hp, lwd=3)

# Grouped preference with dplyr
library(dplyr)
psel(group_by(mtcars, cyl), low(mpg))

# Return size of each maxima group
summarise(psel(group_by(mtcars, cyl), low(mpg)), n())
}
\seealso{
See \code{\link{complex_pref}} on how to construct a Skyline preference. See \code{\link{plot_front}} on how to plot the pareto front of a Skyline
}

