% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/ode_generics.R, R/ODEAdaptiveSolver.R,
%   R/DormandPrince45.R
\docType{methods}
\name{setTolerance}
\alias{setTolerance}
\alias{setTolerance,ODEAdaptiveSolver-method}
\alias{setTolerance,DormandPrince45-method}
\title{Set the tolerance for the solver}
\usage{
setTolerance(object, tol, ...)

\S4method{setTolerance}{ODEAdaptiveSolver}(object, tol)

\S4method{setTolerance}{DormandPrince45}(object, tol)
}
\arguments{
\item{object}{a class object}

\item{tol}{tolerance}

\item{...}{additional parameters}
}
\description{
Set the tolerance for the solver
}
\examples{
# ++++++++++++++++++++++++++++++++++++++++++++++++  example: ComparisonRK45App.R
# Compares the solution by the RK45 ODE solver versus the analytical solution
# Example file: ComparisonRK45App.R
# ODE Solver:   Runge-Kutta 45
# Class:        RK45

importFromExamples("ODETest.R")

 ComparisonRK45App <- function(verbose = FALSE) {
     ode <- new("ODETest")                     # create an `ODETest` object
     ode_solver <- RK45(ode)                   # select the ODE solver
     ode_solver <- setStepSize(ode_solver, 1)      # set the step
     ode_solver <- setTolerance(ode_solver, 1e-8)  # set the tolerance
     time <-  0
     rowVector <- vector("list")
     i <- 1
     while (time < 50) {
         rowVector[[i]] <- list(t  = ode_solver@ode@state[2],
                                s1 = getState(ode_solver@ode)[1],
                                s2 = getState(ode_solver@ode)[2],
                                xs = getExactSolution(ode_solver@ode, time),
                                rc = getRateCounts(ode),
                                time = time)
         ode_solver <- step(ode_solver)       # advance one step
         stepSize <-  ode_solver@stepSize     # update the step size
         time <- time + stepSize
         state <- getState(ode_solver@ode)    # get the `state` vector
         i <- i + 1
     }
     DT <- data.table::rbindlist(rowVector)    # a data table with the results
     return(DT)
 }
# show solution
solution <- ComparisonRK45App()                          # run the example
plot(solution)
# ++++++++++++++++++++++++++++++++++++++++++ example: KeplerDormandPrince45App.R
# Demostration of the use of ODE solver RK45 for a particle subjected to
# a inverse-law force. The difference with the example KeplerApp is we are
# seeing the effect in thex and y axis on the particle.
# The original routine used the Verlet ODE solver

importFromExamples("KeplerDormandPrince45.R")

KeplerDormandPrince45App <- function(verbose = FALSE) {
    # values for the examples
    x  <- 1
    vx <- 0
    y  <- 0
    vy <- 2 * pi
    dt <- 0.01          # step size
    tol <- 1e-3         # tolerance
    particle <- Kepler()                            # use class Kepler
    particle <- init(particle, c(x, vx, y, vy, 0))  # enter state vector
    odeSolver <- DormandPrince45(particle)      # select the ODE solver
    odeSolver <- init(odeSolver, dt)            # start the solver
    odeSolver <- setTolerance(odeSolver, tol)   # this works for adaptive solvers
    particle@odeSolver <- odeSolver             # copy the solver to ODE object
    initialEnergy <- getEnergy(particle)        # calculate the energy
    rowVector <- vector("list")
    i <- 1
    while (getTime(particle) < 1.5) {
        rowVector[[i]] <- list(t  = particle@state[5],
                               x  = particle@state[1],
                               vx = particle@state[2],
                               y  = particle@state[3],
                               vx = particle@state[4],
                               energy = getEnergy(particle) )
        particle <- doStep(particle)            # advance one step
        energy   <- getEnergy(particle)         # calculate energy
        i <- i + 1
    }
    DT <- data.table::rbindlist(rowVector)
    return(DT)
}


solution <- KeplerDormandPrince45App()
plot(solution)

# +++++++++++++++++++++++++++++++++++++++++ Example:      ComparisonRK45ODEApp.R
# Updates the ODE state instead of using the internal state in the ODE solver
# Also plots the solver solution versus the analytical solution at a
# tolerance of 1e-6
# ODE Solver:   Runge-Kutta 45
# Class:        RK45

importFromExamples("ODETest.R")

ComparisonRK45ODEApp <- function(verbose = FALSE) {
    ode <- new("ODETest")                         # new ODE instance
    ode_solver <- RK45(ode)                       # select ODE solver
    ode_solver <- setStepSize(ode_solver, 1)      # set the step
    ode_solver <- setTolerance(ode_solver, 1e-6)  # set the tolerance
    time <-  0
    rowVector <- vector("list")                   # row vector
    i <- 1    # counter
    while (time < 50) {
        # add solution objects to a row vector
        rowVector[[i]] <- list(t = ode_solver@ode@state[2],
                               ODE = getState(ode_solver@ode)[1],
                               s2  = getState(ode_solver@ode)[2],
                               exact = getExactSolution(ode_solver@ode, time),
                               rate.counts = getRateCounts(ode),
                               time = time )
        ode_solver <- step(ode_solver)            # advance solver one step
        stepSize <-  getStepSize(ode_solver)      # get the current step size
        time <- time + stepSize
        ode <- ode_solver@ode                     # get updated ODE object
        state <- getState(ode)                    # get the `state` vector
        i <- i + 1                                # add a row vector
    }
    DT <- data.table::rbindlist(rowVector)        # create data table
    return(DT)
}

solution <- ComparisonRK45ODEApp()
plot(solution)
library(ggplot2)
library(dplyr)
library(tidyr)
solution.multi <- solution \%>\%
    select(t, ODE, exact)
plot(solution.multi)
solution.2x1 <- solution.multi \%>\%
    gather(key, value, -t)
g <- ggplot(solution.2x1, mapping = aes(x = t, y = value, color = key))
g <-  g + geom_line(size = 1) + labs(title = "ODE vs Exact solution",
                                     subtitle = "tolerance = 1E-6")
print(g)




}
\seealso{
Other adaptive solver methods: \code{\link{getErrorCode}},
  \code{\link{getTolerance}}
}
