% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/travel_time_matrix.R
\name{travel_time_matrix}
\alias{travel_time_matrix}
\title{Calculate travel time matrix between origin destination pairs}
\usage{
travel_time_matrix(
  r5r_core,
  origins,
  destinations,
  mode = "WALK",
  mode_egress = "WALK",
  departure_datetime = Sys.time(),
  time_window = 1L,
  percentiles = 50L,
  breakdown = FALSE,
  breakdown_stat = "MEAN",
  max_walk_dist = Inf,
  max_bike_dist = Inf,
  max_trip_duration = 120L,
  walk_speed = 3.6,
  bike_speed = 12,
  max_rides = 3,
  max_lts = 2,
  n_threads = Inf,
  verbose = TRUE,
  progress = TRUE
)
}
\arguments{
\item{r5r_core}{a rJava object to connect with R5 routing engine}

\item{origins, destinations}{a spatial sf POINT object with WGS84 CRS, or a
data.frame containing the columns 'id', 'lon',
'lat'.}

\item{mode}{string. Transport modes allowed for the trips. Defaults to
"WALK". See details for other options.}

\item{mode_egress}{string. Transport mode used after egress from public
transport. It can be either 'WALK', 'BICYCLE', or 'CAR'.
Defaults to "WALK".}

\item{departure_datetime}{POSIXct object. If working with public transport
networks, please check \code{calendar.txt} within
the GTFS file for valid dates. See details for
further information on how datetimes are parsed.}

\item{time_window}{numeric. Time window in minutes for which r5r will
calculate multiple travel time matrices departing each
minute. By default, the number of simulations is 5 times
the size of 'time_window' set by the user. Defaults window
size to '1', the function only considers 5 departure
times. This parameter is only used with frequency-based
GTFS files. See details for further information.}

\item{percentiles}{numeric vector. Defaults to '50', returning the median
travel time for a given time_window. If a numeric vector is passed,
for example c(25, 50, 75), the function will return
additional columns with the travel times within percentiles
of trips. For example, if the 25 percentile of trips between
A and B is 15 minutes, this means that 25\% of all trips
taken between A and B within the set time window are shorter
than 15 minutes. Only the first 5 cut points of the percentiles
are considered. For more details, see R5 documentation at
'https://docs.conveyal.com/analysis/methodology#accounting-for-variability'}

\item{breakdown}{logic. If \code{FALSE} (default), the function returns a simple
output with columns origin, destination and travel time
percentiles. If \code{TRUE}, r5r breaks down the trip information
and returns more columns with estimates of \code{access_time},
\code{waiting_time}, \code{ride_time}, \code{transfer_time}, \code{total_time} , \code{n_rides}
and \code{route}. Warning: Setting \code{TRUE} makes the function
significantly slower.}

\item{breakdown_stat}{string. If \code{min}, all the brokendown trip informantion
is based on the trip itinerary with the smallest waiting time in the
time window. If \code{breakdown_stat = mean}, the information is based on
the trip itinerary whose waiting time is the closest to the average
waiting time in the time window.}

\item{max_walk_dist}{numeric. Maximum walking distance (in meters) to access
and egress the transit network, or to make transfers
within the network. Defaults to no restrictions as long
as \code{max_trip_duration} is respected. The max distance is
considered separately for each leg (e.g. if you set
\code{max_walk_dist} to 1000, you could potentially walk up
to 1 km to reach transit, and up to \emph{another}  1 km to
reach the destination after leaving transit). Obs: if you
want to set the maximum walking distance considering
walking-only trips you have to set the \code{max_trip_duration}
accordingly (e.g. to set a distance of 1 km assuming a
walking speed of 3.6 km/h you have to set \code{max_trip_duration = 1 / 3.6 * 60}).}

\item{max_bike_dist}{numeric. Maximum cycling distance (in meters) to access
and egress the transit network. Defaults to no
restrictions as long as \code{max_trip_duration} is respected.
The max distance is considered separately for each leg
(e.g. if you set \code{max_bike_dist} to 1000, you could
potentially cycle up to 1 km to reach transit, and up
to \emph{another} 1 km to reach the destination after leaving
transit). Obs: if you want to set the maximum cycling
distance considering cycling-only trips you have to set
the \code{max_trip_duration} accordingly (e.g. to set a
distance of 5 km assuming a cycling speed of 12 km/h you
have to set \code{max_trip_duration = 5 / 12 * 60}).}

\item{max_trip_duration}{numeric. Maximum trip duration in minutes. Defaults
to 120 minutes (2 hours).}

\item{walk_speed}{numeric. Average walk speed in km/h. Defaults to 3.6 km/h.}

\item{bike_speed}{numeric. Average cycling speed in km/h. Defaults to 12 km/h.}

\item{max_rides}{numeric. The max number of public transport rides allowed in
the same trip. Defaults to 3.}

\item{max_lts}{numeric (between 1 and 4). The maximum level of traffic stress
that cyclists will tolerate. A value of 1 means cyclists will
only travel through the quietest streets, while a value of 4
indicates cyclists can travel through any road. Defaults to 2.
See details for more information.}

\item{n_threads}{numeric. The number of threads to use in parallel computing.
Defaults to use all available threads (Inf).}

\item{verbose}{logical. \code{TRUE} to show detailed output messages (the default).}

\item{progress}{logical. \code{TRUE} to show a progress counter. Only works when
\code{verbose} is set to \code{FALSE}, so the progress counter does not
interfere with R5's output messages. Setting \code{progress} to \code{TRUE}
may impose a small penalty for computation efficiency, because
the progress counter must be synchronized among all active
threads.}
}
\value{
A data.table with travel time estimates (in minutes) between origin
destination pairs by a given transport mode. Note that origins/destinations
that were beyond the maximum travel time, and/or origins that were far from
the street network are not returned in the data.table.
}
\description{
Fast computation of travel time estimates between one or
multiple origin destination pairs.
}
\section{Transport modes:}{
R5 allows for multiple combinations of transport modes. The options include:
\subsection{Transit modes}{

TRAM, SUBWAY, RAIL, BUS, FERRY, CABLE_CAR, GONDOLA, FUNICULAR. The option
'TRANSIT' automatically considers all public transport modes available.
}

\subsection{Non transit modes}{

WALK, BICYCLE, CAR, BICYCLE_RENT, CAR_PARK
}
}

\section{max_lts, Maximum Level of Traffic Stress:}{
When cycling is enabled in R5, setting \code{max_lts} will allow cycling only on
streets with a given level of danger/stress. Setting \code{max_lts} to 1, for example,
will allow cycling only on separated bicycle infrastructure or low-traffic
streets; routing will revert to walking when traversing any links with LTS
exceeding 1. Setting \code{max_lts} to 3 will allow cycling on links with LTS 1, 2,
or 3.

The default methodology for assigning LTS values to network edges is based on
commonly tagged attributes of OSM ways. See more info about LTS in the original
documentation of R5 from Conveyal at \url{https://docs.conveyal.com/learn-more/traffic-stress}.
In summary:
\itemize{
\item \strong{LTS 1}: Tolerable for children. This includes low-speed, low-volume streets,
as well as those with separated bicycle facilities (such as parking-protected
lanes or cycle tracks).
\item \strong{LTS 2}: Tolerable for the mainstream adult population. This includes streets
where cyclists have dedicated lanes and only have to interact with traffic at
formal crossing.
\item \strong{LTS 3}: Tolerable for “enthused and confident” cyclists. This includes streets
which may involve close proximity to moderate- or high-speed vehicular traffic.
\item \strong{LTS 4}: Tolerable for only “strong and fearless” cyclists. This includes streets
where cyclists are required to mix with moderate- to high-speed vehicular traffic.
}

For advanced users, you can provide custom LTS values by adding a tag
<key = "lts> to the \code{osm.pbf} file
}

\section{Routing algorithm:}{
The travel_time_matrix function uses an R5-specific extension to the RAPTOR
routing algorithm (see Conway et al., 2017). This RAPTOR extension uses a
systematic sample of one departure per minute over the time window set by the
user in the 'time_window' parameter. A detailed description of base RAPTOR
can be found in Delling et al (2015).
\itemize{
\item Conway, M. W., Byrd, A., & van der Linden, M. (2017). Evidence-based transit
and land use sketch planning using interactive accessibility methods on
combined schedule and headway-based networks. Transportation Research Record,
2653(1), 45-53.
\item Delling, D., Pajor, T., & Werneck, R. F. (2015). Round-based public transit
routing. Transportation Science, 49(3), 591-604.
}
}

\section{Datetime parsing}{
\code{r5r} ignores the timezone attribute of datetime objects when parsing dates
and times, using the study area's timezone instead. For example, let's say
you are running some calculations using Rio de Janeiro, Brazil, as your study
area. The datetime \code{as.POSIXct("13-05-2019 14:00:00", format = "\%d-\%m-\%Y \%H:\%M:\%S")} will be parsed as May 13th, 2019, 14:00h in
Rio's local time, as expected. But \code{as.POSIXct("13-05-2019 14:00:00", format = "\%d-\%m-\%Y \%H:\%M:\%S", tz = "Europe/Paris")} will also be parsed as
the exact same date and time in Rio's local time, perhaps surprisingly,
ignoring the timezone attribute.
}

\examples{
if (interactive()) {
library(r5r)

# build transport network
data_path <- system.file("extdata/spo", package = "r5r")
r5r_core <- setup_r5(data_path = data_path, temp_dir = TRUE)

# load origin/destination points
points <- read.csv(file.path(data_path, "spo_hexgrid.csv"))[1:5,]

departure_datetime <- as.POSIXct("13-05-2019 14:00:00", format = "\%d-\%m-\%Y \%H:\%M:\%S")

# estimate travel time matrix
ttm <- travel_time_matrix(r5r_core,
                          origins = points,
                          destinations = points,
                          mode = c("WALK", "TRANSIT"),
                          departure_datetime = departure_datetime,
                          max_walk_dist = Inf,
                          max_trip_duration = 120L)

stop_r5(r5r_core)

}
}
\seealso{
Other routing: 
\code{\link{accessibility}()},
\code{\link{detailed_itineraries}()}
}
\concept{routing}
