# --------------------------------------
# Author: Andreas Alfons
#         Erasmus Universiteit Rotterdam
# --------------------------------------


## function to write style file, either to R console or to an actual file
#' Create the LaTeX style file for 'r2spss'
#'
#' Create the LaTeX style file required to compile LaTeX documents that include
#' tables created by package \pkg{r2spss}.  You can put the resulting file
#' \emph{r2spss.sty} in the folder containing your LaTeX document, and you
#' should include \preformatted{\usepackage{r2spss}} in the preamble of your
#' LaTeX document.
#'
#' @param path  a character string specifying the path to the folder in which
#' to put the style file, or \code{NULL} (the default) to print the contents
#' of the style file to the standard output connection (usually the \R
#' console).
#'
#' @return  Nothing is returned, the function is called for its side effects.
#'
#' @author Andreas Alfons
#'
#' @examples
#' # print contents of style file
#' r2spss.sty()
#'
#' \dontrun{
#'
#'   # put file 'r2spss.sty' in the current working directory
#'   r2spss.sty(".")
#' }
#'
#' @keywords IO
#'
#' @importFrom utils packageDate packageVersion
#' @export

r2spss.sty <- function(path = NULL) {

  # determine file name
  if (is.null(path)) file <- ""  # write to standard output (usually R console)
  else {
    sep <- if (endsWith(path, "/")) "" else "/"
    fileName <- paste(path, "r2spss.sty", sep = sep)
    file <- file(fileName, open = "w")
    on.exit(close(file))
  }

  # write author information
  cat("% ----------------------------------------------------------\n",
      file = file)
  cat("% Author: Andreas Alfons\n", file = file)
  cat("%         Erasmus University Rotterdam\n", file = file)
  cat("% \n", file = file)
  cat("% Generated by function r2spss.sty() from R package 'r2spss'\n",
      file = file)
  cat("% ----------------------------------------------------------\n\n",
      file = file)

  # write package identification
  cat("\\NeedsTeXFormat{LaTeX2e}\n", file = file)
  cat("\\ProvidesPackage{r2spss}%\n", file = file)
  cat("[", as.character(packageDate("r2spss"), format = "%Y/%m/%d"),
      " Format R Output to Look Like SPSS v",
      as.character(packageVersion("r2spss")), "]\n\n",
      file = file, sep = "")

  # write LaTeX statements for required packages
  cat("% packages generally required\n", file = file)
  cat("\\RequirePackage{amsmath}\n", file = file)
  cat("\\RequirePackage[T1]{fontenc}\n\n", file = file)
  cat("% packages required for modern theme\n", file = file)
  cat("\\RequirePackage{nicematrix}\n\n", file = file)
  cat("% packages required for legacy theme\n", file = file)
  cat("\\RequirePackage{makecell}\n\n", file = file)

  # write LaTeX statements to define colors
  cat("% define colors for modern theme\n", file = file)
  cat("\\definecolor{graySPSS}{RGB}{228,228,228}\n", file = file)
  cat("\\definecolor{lightgraySPSS}{RGB}{249,249,251}\n", file = file,
      append = TRUE)
  cat("\\definecolor{darkgraySPSS}{RGB}{174,174,174}\n", file = file)
  cat("\\definecolor{blueSPSS}{RGB}{38,74,96}\n", file = file)

}
