#' @title Receiver Operator Curves
#'
#' @description Construction of ROC curves.
#'
#' @details
#' Given a \code{glm} fit with \code{family = "binomial"} (either a log-link or
#' logit-link should be fine, a data set will be constructed and ROC plots
#' generated.
#'
#' The area under the curve (AUC) is determined by a trapezoid approximation.
#'
#' @param x a \code{glm} fit or \code{data.frame} generated by
#' \code{qroc_build_data_frame}.
#' @param n_threshold number of thresholds to test against.
#' @param ... passed to \code{stats::predict}
#'
#' @seealso \code{vignette("qwraps2-graphics", package = "qwraps2")} for more
#' examples.
#'
#' @return a ggplot.  Minimal aesthetics have been used so that the user may
#' modify the graphic as desired with ease.
#'
#' @examples
#' # load ggplot2 and the diamonds data set
#' data(diamonds, package = "ggplot2")
#'
#' # Create a logistic regression models
#' fit <- glm(I(price > 2800) ~ cut * color, data = diamonds, family = binomial())
#'
#' qroc(fit)
#'
#' @export
#' @rdname qroc
qroc <- function(x, ...) {
  UseMethod("qroc")
}

#' @export
qroc.default <- function(x, ...) {
  qroc_ggplot(x)
}

#' @export
qroc.glm <- function(x, n_threshold = 200, ...) {
  auc_data <- qroc_build_data_frame(x, n_threshold, ...)
  qroc_ggplot(auc_data)
}

#' @export
qroc.qwraps2_generated <- function(x, ...) {
  qroc_ggplot(x)
}

qroc_ggplot <- function(.data) {
  ggplot2::ggplot(.data) +
  ggplot2::aes_string(x = "false_positives", y = "true_positives") +
  ggplot2::geom_line() +
  ggplot2::geom_segment(ggplot2::aes(x = 0, y = 0, xend = 1, yend = 1), color = "black", linetype = 2) +
  ggplot2::xlab("1 - Specificity") +
  ggplot2::ylab("Sensitivity")
}

#' @export
#' @rdname qroc
#' @param fit a \code{glm} fit with \code{family = binomial()}.
qroc_build_data_frame <- function(fit, n_threshold = 200, ...) {

  # find the predicted values
  pred_vals <- stats::predict(fit, ..., type = "response")

  true_pos <- function(threshold){ sum((pred_vals >= threshold) &  (fit$y)) }
  true_neg <- function(threshold){ sum((pred_vals <  threshold) & !(fit$y)) }

  false_pos <- function(threshold){ sum((pred_vals >= threshold) & !(fit$y)) }
  false_neg <- function(threshold){ sum((pred_vals <  threshold) &  (fit$y)) }

  x <- matrix(seq(1, 0, length = n_threshold))

  true_positives  <- apply(x, 1, true_pos)
  true_negatives  <- apply(x, 1, true_neg)
  false_positives <- apply(x, 1, false_pos)
  false_negatives <- apply(x, 1, false_neg)

  sensitivity <- true_positives / (true_positives + false_negatives)
  specificity <- true_negatives / (true_negatives + false_positives)

  roc_data <- data.frame(false_positives = 1 - specificity,
                         true_positives  = sensitivity)

  # trapezoid rule approximation for the area under the curve
  auc <- sum((roc_data[2:n_threshold, 1] - roc_data[1:(n_threshold-1), 1]) * 1/2 *
             (roc_data[2:n_threshold, 2] + roc_data[1:(n_threshold-1), 2]))

  attr(roc_data, "auc") <- auc
  class(roc_data)       <- c("qwraps2_generated", class(roc_data))

  return(roc_data)
}

#' @export
#' @rdname qroc
#' @param .data a \code{data.frame} generated by \code{qroc_build_data_frame}.
#' @return AUC for the data set generated by
auc <- function(.data) {
  UseMethod("auc")
}

#' @export
auc.qwraps2_generated <- function(.data) {
  attr(.data, "auc")
}
