% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/textstat_simil.R
\name{textstat_simil}
\alias{textstat_simil}
\alias{textstat_dist}
\title{Similarity and distance computation between documents or features}
\usage{
textstat_simil(x, selection = NULL, margin = c("documents",
  "features"), method = c("correlation", "cosine", "jaccard", "ejaccard",
  "dice", "edice", "hamman", "simple matching", "faith"), upper = FALSE,
  diag = FALSE)

textstat_dist(x, selection = NULL, margin = c("documents", "features"),
  method = c("euclidean", "kullback", "manhattan", "maximum", "canberra",
  "minkowski"), upper = FALSE, diag = FALSE, p = 2)
}
\arguments{
\item{x}{a \link{dfm} object}

\item{selection}{a valid index for document or feature names (depending on
\code{margin}) from \code{x}, to be selected for comparison}

\item{margin}{identifies the margin of the dfm on which similarity or
difference will be computed:  \code{"documents"} for documents or
\code{"features"} for word/term features.}

\item{method}{method the similarity or distance measure to be used; see
Details.}

\item{upper}{whether the upper triangle of the symmetric \eqn{V \times V}
matrix is recorded. Only used when \code{value = "dist"}.}

\item{diag}{whether the diagonal of the distance matrix should be recorded. .
Only used when \code{value = "dist"}.}

\item{p}{The power of the Minkowski distance.}
}
\value{
By default, \code{textstat_simil} and \code{textstat_dist} return
  \code{\link{dist}} class objects if selection is \code{NULL}, otherwise, a
  matrix is returned matching distances to the documents or features
  identified in the selection.
  
  These can be transformed into a list format using
  \code{\link{as.list.dist}}, if that format is preferred.
}
\description{
These functions compute matrixes of distances and similarities between
documents or features from a \code{\link{dfm}} and return a
\code{\link[stats]{dist}} object (or a matrix if specific targets are
selected).  They are fast and robust because they operate directly on the
sparse \link{dfm} objects.
}
\details{
\code{textstat_simil} options are: \code{"correlation"} (default),
  \code{"cosine"}, \code{"jaccard"}, \code{"ejaccard"}, \code{"dice"},
  \code{"edice"}, \code{"simple matching"}, \code{"hamman"}, and
  \code{"faith"}.

\code{textstat_dist} options are: \code{"euclidean"} (default), 
  \code{"kullback"}. \code{"manhattan"}, \code{"maximum"}, \code{"canberra"},
  and \code{"minkowski"}.
}
\note{
If you want to compute similarity on a "normalized" dfm object
  (controlling for variable document lengths, for methods such as correlation
  for which different document lengths matter), then wrap the input dfm in
  \code{\link{dfm_weight}(x, "prop")}.
}
\examples{
# similarities for documents
mt <- dfm(data_corpus_inaugural, remove_punct = TRUE, remove = stopwords("english"))
(s1 <- textstat_simil(mt, method = "cosine", margin = "documents"))
as.matrix(s1)
as.list(s1)

# similarities for for specific documents
textstat_simil(mt, "2017-Trump", margin = "documents")
textstat_simil(mt, "2017-Trump", method = "cosine", margin = "documents")
textstat_simil(mt, c("2009-Obama" , "2013-Obama"), margin = "documents")

# compute some term similarities
s2 <- textstat_simil(mt, c("fair", "health", "terror"), method = "cosine",
                      margin = "features")
head(as.matrix(s2), 10)
as.list(s2, n = 8)

# create a dfm from inaugural addresses from Reagan onwards
mt <- dfm(corpus_subset(data_corpus_inaugural, Year > 1990), 
               remove = stopwords("english"), stem = TRUE, remove_punct = TRUE)
               
# distances for documents 
(d1 <- textstat_dist(mt, margin = "documents"))
as.matrix(d1)

# distances for specific documents
textstat_dist(mt, "2017-Trump", margin = "documents")
(d2 <- textstat_dist(mt, c("2009-Obama" , "2013-Obama"), margin = "documents"))
as.list(d1)

}
\references{
\code{"kullback"} is the Kullback-Leibler distance, which assumes that
  \eqn{P(x_i) = 0} implies \eqn{P(y_i)=0}, and in case either \eqn{P(x_i)} or
  \eqn{P(y_i)} equals to zero, then \eqn{P(x_i) * log(p(x_i)/p(y_i))} is
  assumed to be zero as the limit value.  The formula is:
   \deqn{\sum{P(x)*log(P(x)/p(y))}}
   
  All other measures are described in the \pkg{proxy} package.
}
\seealso{
\code{\link{textstat_dist}},
  \code{\link[quanteda]{as.matrix.simil}},
  \code{\link[quanteda]{as.list.dist}}, \code{\link[stats]{dist}},
  \code{\link[stats]{as.dist}}
}
