
#' stem words
#' 
#' Apply a stemmer to words.  This is a wrapper to \link[SnowballC]{wordStem} 
#' designed to allow this function to be called without loading the entire 
#' \pkg{SnowballC} package.  \link[SnowballC]{wordStem}  uses Martin Porter's 
#' stemming algorithm and the C libstemmer library generated by Snowball.
#' @param x a character vector or corpus, whose word stems are to be removed
#' @param language the name of a recognized language, as returned by 
#'   \link[SnowballC]{getStemLanguages}, or a two- or three-letter ISO-639 code 
#'   corresponding to one of these languages (see references for the list of 
#'   codes)
#' @return A character vector with as many elements as there are in the input 
#'   vector with the corresponding elements being the stem of the word. Elements
#'   of the vector are converted to UTF-8 encoding before the stemming is 
#'   performed, and the returned elements are marked as such when they contain 
#'   non-ASCII characters.
#' @seealso \link[SnowballC]{wordStem}
#'   
#' @references \url{http://snowball.tartarus.org/}
#'   
#'   \url{http://www.iso.org/iso/home/standards/language_codes.htm} for the 
#'   ISO-639 language codes
#' @export
#' @examples
#' #' Simple example
#' wordstem(c("win", "winning", "wins", "won", "winner"))
wordstem <- function(x, language = "porter") {
    UseMethod("wordstem")
}

#' @rdname wordstem
#' @import stringi 
#' @export
wordstem.character <- function(x, language = "porter") {
    if (any(stringi::stri_detect_fixed(x, " ")))
        stop("whitespace detected: you can only stem tokenized texts")
    SnowballC::wordStem(x, language)
}

# 
# toks <- unlist(tokenize(toLower(inaugTexts[1:5]), removePunct = TRUE, removeNumbers = TRUE), use.names = FALSE)
# microbenchmark(wordstem(toks), 
#                wordstemP(toks),
#                simplify2array(parallel::mclapply(toks, wordstem, language=language)))

