#' Transform Codes to Start-End Durations
#' 
#' Transforms the range coding structure(s) from cm_range.temp (in list format) 
#' into a data frame of start and end durations in long format.
#' 
#' @param \ldots list object(s) in the form generated by 
#' \code{\link[qdap]{cm_time.temp}}.
#' @param v.name sn optional name for the column created for the list.var 
#' argument.
#' @param list.var logical.  If TRUE creates a column for the data frame created 
#' by each time.list passed to \code{cm_t2l}.
#' @param debug logical. If TRUE debugging mode is on.  
#' \code{\link[qdap]{cm_time2long}} will return possible errors in time span 
#' inputs.
#' @return Generates a data frame of start and end times for each code.
#' @seealso 
#' \code{\link[qdap]{cm_df2long}},
#' \code{\link[qdap]{cm_time.temp}},
#' \code{\link[qdap]{cm_df.transcript}}
#' @references Miles, M. B. & Huberman, A. M. (1994). An expanded sourcebook: 
#' Qualitative   data analysis. 2nd ed. Thousand Oaks, CA: SAGE Publications.
#' @keywords coding, time span
#' @export
#' @examples
#' \dontrun{
#' foo <- list(
#'     AA = qcv(terms="1"),
#'     BB = qcv(terms="1:2, 3:10, 19"),
#'     CC = qcv(terms="1:9, 100:150")
#' )
#' 
#' foo2  <- list(
#'     AA = qcv(terms="40"),
#'     BB = qcv(terms="50:90"),
#'     CC = qcv(terms="60:90, 100:120, 150"),
#'     DD = qcv(terms="")
#' )
#' 
#' (dat <- cm_range2long(foo, foo2, v.name = "time"))
#' gantt_wrap(dat, "code", "time")
#' }
cm_range2long <-
function(..., v.name = "variable", list.var = TRUE, debug = TRUE){
    mf <- match.call(expand.dots = FALSE)
    objs <- as.character(mf[[2]])
    L1 <- lapply(objs, get)
    names(L1) <- objs
    if(debug){
        x <- suppressMessages(lapply(L1, function(x) {
            cm_debug2(x)
        }))
        m <- x[!sapply(x, is.null)]
        if (!identical(as.character(m), character(0))) {
            message("Warning: possible errors found:\n")
            print(m); stop("Check warnings")
        }  
    }  
    L2 <- lapply(L1, cm_r2l, list.var = FALSE)
    if (list.var) {
        L2 <- lapply(seq_along(L2), function(i) data.frame(L2[[i]], 
            VAR = objs[i]))
    }
    DF <- data.frame(do.call(rbind, L2), row.names = NULL)
    if (list.var) {
        colnames(DF)[ncol(DF)] <- v.name
    }
    comment(DF) <- "cmrange"
    DF
}
